<?php
// announcement.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
// Check permission - only moderators and root can access
if (!isset($_SESSION['admin_role']) || ($_SESSION['admin_role'] !== 'moderator' && $_SESSION['admin_role'] !== 'root')) {
    header('Location: overview.php');
    exit;
}


// Include database connection
require_once 'php/db.php';

// Fetch existing announcements
try {
    $stmt = $pdo->prepare("SELECT * FROM announcement ORDER BY created_at DESC");
    $stmt->execute();
    $announcements = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Announcements - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Add Announcement Button */
        .add-announcement-btn {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .add-announcement-btn:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Announcements Grid */
        .announcements-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(400px, 1fr));
            gap: 25px;
        }
        
        .announcement-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .announcement-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        }
        
        .announcement-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .announcement-type {
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .type-alert {
            background: linear-gradient(90deg, var(--warning), #d97706);
            color: white;
        }
        
        .type-scrolling {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .announcement-actions {
            display: flex;
            gap: 8px;
        }
        
        .edit-announcement-btn, .delete-announcement-btn {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
        }
        
        .edit-announcement-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-announcement-btn:hover {
            background: var(--primary-dark);
        }
        
        .delete-announcement-btn {
            background: var(--danger);
            color: white;
        }
        
        .delete-announcement-btn:hover {
            background: #dc2626;
        }
        
        .announcement-content {
            margin-bottom: 20px;
        }
        
        .announcement-text {
            font-size: 14px;
            line-height: 1.6;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .announcement-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .announcement-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .announcement-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .announcement-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            word-break: break-all;
        }
        
        .announcement-link {
            margin-top: 15px;
            padding-top: 15px;
            border-top: 1px solid var(--gray-light);
        }
        
        .announcement-link a {
            color: var(--primary);
            text-decoration: none;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
        }
        
        .announcement-link a:hover {
            color: var(--primary-dark);
            text-decoration: underline;
        }
        
        .announcement-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-top: 20px;
            padding-top: 15px;
            border-top: 1px solid var(--gray-light);
            font-size: 12px;
            color: var(--gray);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            background: white;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 25px;
        }
        
        /* Modals */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .announcement-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .announcement-modal {
            transform: translateY(0);
        }
        
        .announcement-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-group label.required::after {
            content: ' *';
            color: var(--danger);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid var(--gray-light);
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        textarea.form-control {
            min-height: 120px;
            resize: vertical;
        }
        
        .checkbox-group {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-top: 10px;
        }
        
        .checkbox-group input[type="checkbox"] {
            width: 18px;
            height: 18px;
            cursor: pointer;
        }
        
        .checkbox-group label {
            margin: 0;
            cursor: pointer;
            font-size: 14px;
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .form-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .form-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .form-btn.primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .form-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .form-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .form-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Loading Spinner */
        .spinner {
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Message Modal */
        .message-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .message-modal {
            transform: translateY(0);
        }
        
        .message-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .message-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .message-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .message-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .message-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .message-actions {
            display: flex;
            justify-content: center;
        }
        
        .message-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .message-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Confirmation Modal */
        .confirm-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .confirm-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .confirm-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .confirm-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .confirm-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .confirm-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .confirm-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .confirm-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .confirm-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .confirm-btn.secondary:hover {
            background: #d1d5db;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .announcements-grid {
                grid-template-columns: 1fr;
            }
            
            .announcement-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .message-modal, .confirm-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
        
        @media (max-width: 480px) {
            .form-actions, .confirm-actions {
                flex-direction: column;
            }
            
            .form-btn, .confirm-btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Announcements</h2>
                    <p>Manage system announcements</p>
                </div>
                <div class="header-actions">
                    <button class="add-announcement-btn" id="addAnnouncementBtn">
                        <i class="fas fa-plus"></i> Add Announcement
                    </button>
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Announcements List -->
            <div class="announcements-grid" id="announcementsList">
                <?php if (empty($announcements)): ?>
                    <div class="empty-state">
                        <i class="fas fa-bullhorn"></i>
                        <h3>No Announcements</h3>
                        <p>No announcements have been created yet</p>
                        <button class="add-announcement-btn" id="addAnnouncementBtn2">
                            <i class="fas fa-plus"></i> Create First Announcement
                        </button>
                    </div>
                <?php else: ?>
                    <?php foreach ($announcements as $announcement): ?>
                        <div class="announcement-card" id="announcement-<?php echo $announcement['id']; ?>">
                            <div class="announcement-header">
                                <span class="announcement-type type-<?php echo $announcement['type']; ?>">
                                    <?php echo ucfirst($announcement['type']); ?>
                                </span>
                                <div class="announcement-actions">
                                    <button class="edit-announcement-btn" onclick="editAnnouncement(<?php echo $announcement['id']; ?>)">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button class="delete-announcement-btn" onclick="deleteAnnouncement(<?php echo $announcement['id']; ?>)">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="announcement-content">
                                <div class="announcement-text">
                                    <?php echo nl2br(htmlspecialchars($announcement['text'])); ?>
                                </div>
                                
                                <div class="announcement-details">
                                    <div class="announcement-info-item">
                                        <label>Action Text</label>
                                        <span><?php echo $announcement['action_text'] ? htmlspecialchars($announcement['action_text']) : 'None'; ?></span>
                                    </div>
                                    <div class="announcement-info-item">
                                        <label>Expires At</label>
                                        <span>
                                            <?php if ($announcement['expired_at']): ?>
                                                <?php echo date('F j, Y H:i', strtotime($announcement['expired_at'])); ?>
                                            <?php else: ?>
                                                Never
                                            <?php endif; ?>
                                        </span>
                                    </div>
                                </div>
                                
                                <?php if ($announcement['link']): ?>
                                    <div class="announcement-link">
                                        <a href="<?php echo htmlspecialchars($announcement['link']); ?>" target="_blank">
                                            <i class="fas fa-external-link-alt"></i>
                                            <?php echo htmlspecialchars($announcement['link']); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                            
                            <div class="announcement-footer">
                                <span>Created: <?php echo date('F j, Y', strtotime($announcement['created_at'])); ?></span>
                                <span>ID: #<?php echo $announcement['id']; ?></span>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Announcement Modal -->
    <div class="modal-overlay" id="announcementModal">
        <div class="announcement-modal">
            <h3 id="modalTitle">Add New Announcement</h3>
            <form id="announcementForm">
                <input type="hidden" id="announcementId" name="id" value="">
                
                <div class="form-group">
                    <label for="announcementType" class="required">Announcement Type</label>
                    <select class="form-control" id="announcementType" name="type" required>
                        <option value="">Select Type</option>
                        <option value="alert">Alert</option>
                        <option value="scrolling">Scrolling</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="announcementText" class="required">Announcement Text</label>
                    <textarea class="form-control" id="announcementText" name="text" required placeholder="Enter announcement text here..."></textarea>
                </div>
                
                <div class="form-group">
                    <label for="announcementLink">Link (Optional)</label>
                    <input type="url" class="form-control" id="announcementLink" name="link" placeholder="https://example.com">
                </div>
                
                <div class="form-group">
                    <label for="actionText">Action Text (Optional)</label>
                    <input type="text" class="form-control" id="actionText" name="action_text" placeholder="Click here, Learn more, etc.">
                </div>
                
                <div class="form-group">
                    <label>Expiration Date</label>
                    <div class="checkbox-group">
                        <input type="checkbox" id="neverExpire" name="never_expire">
                        <label for="neverExpire">Never expire</label>
                    </div>
                    <input type="datetime-local" class="form-control" id="expiredAt" name="expired_at" style="margin-top: 10px;">
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelAnnouncementBtn">Cancel</button>
                    <button type="submit" class="form-btn primary" id="saveAnnouncementBtn">
                        <span id="saveBtnText">Save Announcement</span>
                        <div class="spinner" style="display: none;"></div>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Success/Error Message Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="message-modal">
            <div class="message-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="message-actions">
                <button class="message-btn" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <!-- Confirmation Modal -->
    <div class="modal-overlay" id="confirmModal">
        <div class="confirm-modal">
            <div class="confirm-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle">Confirm Delete</h3>
            <p id="confirmMessage">Are you sure you want to delete this announcement?</p>
            <div class="confirm-actions">
                <button class="confirm-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button class="confirm-btn primary" id="confirmActionBtn">Delete</button>
            </div>
        </div>
    </div>
    
    <script>
        // Global variables
        let currentAnnouncementId = null;
        let confirmCallback = null;
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon based on type
            icon.className = 'message-icon ' + type;
            if (type === 'success') {
                icon.innerHTML = '<i class="fas fa-check-circle"></i>';
            } else if (type === 'error') {
                icon.innerHTML = '<i class="fas fa-times-circle"></i>';
            }
            
            modal.classList.add('active');
        }
        
        // Show confirmation modal
        function showConfirm(title, message, callback) {
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmMessage').textContent = message;
            confirmCallback = callback;
            document.getElementById('confirmModal').classList.add('active');
        }
        
        // Add announcement button click
        document.getElementById('addAnnouncementBtn').addEventListener('click', () => {
            openAnnouncementModal();
        });
        
        document.getElementById('addAnnouncementBtn2')?.addEventListener('click', () => {
            openAnnouncementModal();
        });
        
        // Open announcement modal (add mode)
        function openAnnouncementModal() {
            document.getElementById('modalTitle').textContent = 'Add New Announcement';
            document.getElementById('announcementForm').reset();
            document.getElementById('announcementId').value = '';
            document.getElementById('saveBtnText').textContent = 'Save Announcement';
            document.getElementById('announcementModal').classList.add('active');
        }
        
        // Edit announcement
        async function editAnnouncement(id) {
            try {
                // Fetch announcement details
                const response = await fetch('b_announcement.php?action=get_announcement&id=' + id);
                const data = await response.json();
                
                if (data.success) {
                    const announcement = data.announcement;
                    
                    // Fill form
                    document.getElementById('modalTitle').textContent = 'Edit Announcement';
                    document.getElementById('announcementId').value = announcement.id;
                    document.getElementById('announcementType').value = announcement.type;
                    document.getElementById('announcementText').value = announcement.text;
                    document.getElementById('announcementLink').value = announcement.link || '';
                    document.getElementById('actionText').value = announcement.action_text || '';
                    
                    // Handle expiration date
                    const neverExpireCheckbox = document.getElementById('neverExpire');
                    const expiredAtInput = document.getElementById('expiredAt');
                    
                    if (announcement.expired_at) {
                        neverExpireCheckbox.checked = false;
                        const date = new Date(announcement.expired_at);
                        expiredAtInput.value = date.toISOString().slice(0, 16);
                        expiredAtInput.style.display = 'block';
                    } else {
                        neverExpireCheckbox.checked = true;
                        expiredAtInput.style.display = 'none';
                    }
                    
                    document.getElementById('saveBtnText').textContent = 'Update Announcement';
                    document.getElementById('announcementModal').classList.add('active');
                } else {
                    showMessage('Error', data.message || 'Failed to load announcement', 'error');
                }
            } catch (error) {
                console.error('Error loading announcement:', error);
                showMessage('Error', 'Error loading announcement', 'error');
            }
        }
        
        // Delete announcement
        async function deleteAnnouncement(id) {
            showConfirm(
                'Delete Announcement',
                'Are you sure you want to delete this announcement? This action cannot be undone.',
                async () => {
                    try {
                        const response = await fetch('b_announcement.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'delete_announcement',
                                id: id
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            // Remove announcement from DOM
                            const announcementElement = document.getElementById('announcement-' + id);
                            if (announcementElement) {
                                announcementElement.remove();
                            }
                            
                            // Check if no announcements left
                            const announcementsList = document.getElementById('announcementsList');
                            const announcementCards = announcementsList.querySelectorAll('.announcement-card');
                            if (announcementCards.length === 0) {
                                announcementsList.innerHTML = `
                                    <div class="empty-state">
                                        <i class="fas fa-bullhorn"></i>
                                        <h3>No Announcements</h3>
                                        <p>No announcements have been created yet</p>
                                        <button class="add-announcement-btn" id="addAnnouncementBtn2">
                                            <i class="fas fa-plus"></i> Create First Announcement
                                        </button>
                                    </div>
                                `;
                                
                                // Re-attach event listener to new button
                                document.getElementById('addAnnouncementBtn2').addEventListener('click', () => {
                                    openAnnouncementModal();
                                });
                            }
                            
                            showMessage('Success', 'Announcement deleted successfully', 'success');
                        } else {
                            showMessage('Error', data.message || 'Failed to delete announcement', 'error');
                        }
                    } catch (error) {
                        console.error('Error deleting announcement:', error);
                        showMessage('Error', 'Error deleting announcement', 'error');
                    }
                }
            );
        }
        
        // Never expire checkbox handler
        document.getElementById('neverExpire').addEventListener('change', function() {
            const expiredAtInput = document.getElementById('expiredAt');
            if (this.checked) {
                expiredAtInput.style.display = 'none';
                expiredAtInput.value = '';
            } else {
                expiredAtInput.style.display = 'block';
            }
        });
        
        // Announcement form submission
        document.getElementById('announcementForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            
            // Determine action based on whether we have an ID
            const announcementId = document.getElementById('announcementId').value;
            data.action = announcementId ? 'update_announcement' : 'add_announcement';
            
            // Handle never expire checkbox
            if (data.never_expire === 'on') {
                data.expired_at = null;
            }
            delete data.never_expire;
            
            // Get form elements
            const saveBtn = document.getElementById('saveAnnouncementBtn');
            const saveBtnText = document.getElementById('saveBtnText');
            const spinner = saveBtn.querySelector('.spinner');
            
            try {
                // Show loading state
                saveBtn.disabled = true;
                saveBtnText.textContent = 'Saving...';
                spinner.style.display = 'block';
                
                const response = await fetch('b_announcement.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Close modal
                    document.getElementById('announcementModal').classList.remove('active');
                    
                    // Show success message
                    showMessage('Success', result.message, 'success');
                    
                    // Reload page after 2 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to save announcement', 'error');
                }
            } catch (error) {
                console.error('Error saving announcement:', error);
                showMessage('Error', 'Error saving announcement', 'error');
            } finally {
                // Reset button state
                saveBtn.disabled = false;
                saveBtnText.textContent = announcementId ? 'Update Announcement' : 'Save Announcement';
                spinner.style.display = 'none';
            }
        });
        
        // Modal close buttons
        document.getElementById('cancelAnnouncementBtn').addEventListener('click', () => {
            document.getElementById('announcementModal').classList.remove('active');
        });
        
        document.getElementById('messageOkBtn').addEventListener('click', () => {
            document.getElementById('messageModal').classList.remove('active');
        });
        
        document.getElementById('confirmCancelBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            confirmCallback = null;
        });
        
        document.getElementById('confirmActionBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            if (confirmCallback) {
                confirmCallback();
                confirmCallback = null;
            }
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                    if (modal.id === 'confirmModal') {
                        confirmCallback = null;
                    }
                }
            });
        });
    </script>
</body>
</html>