<?php
// auction.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

// Check permission - only moderators and root can access
if (!isset($_SESSION['admin_role']) || ($_SESSION['admin_role'] !== 'moderator' && $_SESSION['admin_role'] !== 'root')) {
    header('Location: overview.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Function to format time
function formatTime($time) {
    return date('h:i A', strtotime($time));
}

// Get current time for auction calculations
date_default_timezone_set('Africa/Lagos');
$current_time = date('H:i:s');

// Fetch auction times
try {
    $stmt = $pdo->query("SELECT * FROM auction_time ORDER BY open_at ASC");
    $auction_times = $stmt->fetchAll();
    
    // Calculate next auction time
    $next_auction = null;
    $next_auction_time = null;
    foreach ($auction_times as $auction) {
        $auction_time = strtotime($auction['open_at']);
        $current_time_stamp = strtotime($current_time);
        
        if ($auction_time > $current_time_stamp) {
            $next_auction = $auction;
            $next_auction_time = date('H:i:s', $auction_time);
            break;
        }
    }
    
    // If no auction found for today, get first one for tomorrow
    if (!$next_auction && count($auction_times) > 0) {
        $next_auction = $auction_times[0];
        $next_auction_time = date('H:i:s', strtotime($next_auction['open_at'] . ' +1 day'));
    }
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Auction Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Timer Section */
        .timer-section {
            background: white;
            border-radius: 15px;
            padding: 40px;
            margin-bottom: 40px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            text-align: center;
        }
        
        .timer-section h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .timer-section h3 i {
            color: var(--primary);
        }
        
        .timer-large {
            font-family: 'Courier New', monospace;
            font-size: 64px;
            font-weight: 700;
            color: var(--primary-dark);
            background: linear-gradient(135deg, #f0f9ff, #e0f2fe);
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 8px 25px rgba(59, 130, 246, 0.2);
            margin: 30px 0;
            text-align: center;
        }
        
        .timer-info {
            display: flex;
            justify-content: center;
            gap: 40px;
            margin-top: 20px;
            flex-wrap: wrap;
        }
        
        .timer-info-item {
            text-align: center;
        }
        
        .timer-info-item h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--gray);
            margin-bottom: 8px;
        }
        
        .timer-info-item p {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
        }
        
        /* Auction Management Section */
        .management-section {
            background: white;
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 30px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .add-auction-btn {
            background: linear-gradient(90deg, var(--secondary), var(--secondary-dark));
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .add-auction-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .auction-list {
            margin-top: 30px;
        }
        
        .auction-item {
            background: var(--light);
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 15px;
            border-left: 4px solid var(--primary);
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .auction-item:hover {
            background: #f1f5f9;
            transform: translateX(5px);
        }
        
        .auction-item.active {
            background: linear-gradient(135deg, #dbeafe, #eff6ff);
            border-left-color: var(--secondary);
        }
        
        .auction-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .auction-details p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .auction-actions {
            display: flex;
            gap: 10px;
        }
        
        .edit-btn, .delete-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .edit-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-btn:hover {
            background: var(--primary-dark);
        }
        
        .delete-btn {
            background: var(--danger);
            color: white;
        }
        
        .delete-btn:hover {
            background: #dc2626;
        }
        
        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 2000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .modal {
            transform: translateY(0);
        }
        
        .modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .modal h3 i {
            color: var(--primary);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 16px;
            font-size: 16px;
            border: 2px solid var(--gray-light);
            border-radius: 8px;
            color: var(--dark);
            transition: all 0.3s ease;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        
        .save-btn, .cancel-btn {
            flex: 1;
            padding: 14px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
        }
        
        .save-btn {
            background: linear-gradient(90deg, var(--secondary), var(--secondary-dark));
            color: white;
        }
        
        .save-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .cancel-btn {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .cancel-btn:hover {
            background: #d1d5db;
        }
        
        /* Success Modal */
        .success-modal {
            position: fixed;
            top: 20px;
            right: 20px;
            background: var(--success);
            color: white;
            padding: 16px 24px;
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(16, 185, 129, 0.3);
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 3000;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
        }
        
        .success-modal.active {
            opacity: 1;
            transform: translateX(0);
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .timer-large {
                font-size: 48px;
                padding: 20px;
            }
            
            .timer-section {
                padding: 30px 20px;
            }
            
            .management-section {
                padding: 30px 20px;
            }
            
            .auction-item {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .auction-actions {
                width: 100%;
                justify-content: flex-end;
            }
        }
        
        @media (max-width: 480px) {
            .timer-large {
                font-size: 36px;
            }
            
            .timer-info {
                flex-direction: column;
                gap: 20px;
            }
            
            .modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu (AFTER the style tag) -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h2>Auction Management</h2>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                    <button class="add-auction-btn" id="addAuctionBtn">
                        <i class="fas fa-plus"></i> Add Auction Time
                    </button>
                </div>
            </div>
            
            <!-- Timer Section -->
            <div class="timer-section">
                <h3><i class="fas fa-gavel"></i> Next Auction Countdown</h3>
                <div class="timer-large" id="auctionTimer">
                    <?php if ($next_auction): ?>
                    <span id="hours">00</span>:<span id="minutes">00</span>:<span id="seconds">00</span>
                    <?php else: ?>
                    No Auction Scheduled
                    <?php endif; ?>
                </div>
                <div class="timer-info">
                    <?php if ($next_auction): ?>
                    <div class="timer-info-item">
                        <h4>Start Time</h4>
                        <p><?php echo formatTime($next_auction['open_at']); ?></p>
                    </div>
                    <div class="timer-info-item">
                        <h4>Duration</h4>
                        <p><?php echo $next_auction['close_after']; ?> minutes</p>
                    </div>
                    <div class="timer-info-item">
                        <h4>Status</h4>
                        <p style="color: var(--success);">Upcoming</p>
                    </div>
                    <?php else: ?>
                    <div class="timer-info-item">
                        <h4>No auction times configured</h4>
                        <p>Add auction times using the button above</p>
                    </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Auction Management Section -->
            <div class="management-section">
                <h3 class="section-title"><i class="fas fa-list"></i> Auction Schedule</h3>
                
                <div class="auction-list" id="auctionList">
                    <?php if (count($auction_times) > 0): ?>
                        <?php foreach ($auction_times as $index => $auction): 
                            $is_next = $next_auction && $next_auction['id'] == $auction['id'];
                        ?>
                        <div class="auction-item <?php echo $is_next ? 'active' : ''; ?>" data-id="<?php echo $auction['id']; ?>">
                            <div class="auction-details">
                                <h4>Auction <?php echo $index + 1; ?> - <?php echo formatTime($auction['open_at']); ?></h4>
                                <p>Duration: <?php echo $auction['close_after']; ?> minutes | ID: <?php echo $auction['id']; ?></p>
                                <p><small>Last updated: <?php echo date('M d, Y H:i', strtotime($auction['updated_at'])); ?></small></p>
                            </div>
                            <div class="auction-actions">
                                <button class="edit-btn" onclick="editAuction(<?php echo $auction['id']; ?>, '<?php echo $auction['open_at']; ?>', <?php echo $auction['close_after']; ?>)">
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                <button class="delete-btn" onclick="deleteAuction(<?php echo $auction['id']; ?>)">
                                    <i class="fas fa-trash"></i> Delete
                                </button>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div style="text-align: center; padding: 40px; color: var(--gray);">
                            <i class="fas fa-clock" style="font-size: 48px; margin-bottom: 20px; opacity: 0.5;"></i>
                            <h3>No Auction Times</h3>
                            <p>Click "Add Auction Time" to create your first auction schedule</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Auction Modal -->
    <div class="modal-overlay" id="auctionModal">
        <div class="modal">
            <h3><i class="fas fa-gavel"></i> <span id="modalTitle">Add Auction Time</span></h3>
            <form id="auctionForm">
                <input type="hidden" id="auctionId" value="">
                
                <div class="form-group">
                    <label for="openTime">Opening Time</label>
                    <input type="time" id="openTime" required step="300">
                    <small style="color: var(--gray); margin-top: 5px; display: block;">Select the time when auction opens</small>
                </div>
                
                <div class="form-group">
                    <label for="closeAfter">Duration (Minutes)</label>
                    <input type="number" id="closeAfter" min="1" max="1440" required placeholder="e.g., 60">
                    <small style="color: var(--gray); margin-top: 5px; display: block;">How long the auction stays open</small>
                </div>
                
                <div class="modal-actions">
                    <button type="submit" class="save-btn">
                        <i class="fas fa-save"></i> Save Auction
                    </button>
                    <button type="button" class="cancel-btn" id="cancelModalBtn">
                        <i class="fas fa-times"></i> Cancel
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Success Modal -->
    <div class="success-modal" id="successModal">
        <i class="fas fa-check-circle"></i>
        <span id="successMessage">Operation completed successfully!</span>
    </div>
    
    <script>
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        <?php if ($next_auction): ?>
        // Auction Timer Countdown
        function updateAuctionTimer() {
            const nextAuctionTime = new Date('<?php echo date("Y-m-d") . " " . $next_auction_time; ?>');
            const now = new Date();
            
            // If the auction time has passed for today, set it for tomorrow
            if (nextAuctionTime < now) {
                nextAuctionTime.setDate(nextAuctionTime.getDate() + 1);
            }
            
            const diff = nextAuctionTime - now;
            
            if (diff > 0) {
                const hours = Math.floor(diff / (1000 * 60 * 60));
                const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);
                
                document.getElementById('hours').textContent = hours.toString().padStart(2, '0');
                document.getElementById('minutes').textContent = minutes.toString().padStart(2, '0');
                document.getElementById('seconds').textContent = seconds.toString().padStart(2, '0');
            } else {
                document.getElementById('auctionTimer').innerHTML = '<span style="color: var(--success); font-size: 36px;">Auction Started!</span>';
            }
        }
        
        setInterval(updateAuctionTimer, 1000);
        updateAuctionTimer();
        <?php endif; ?>
        
        // Modal functionality
        const auctionModal = document.getElementById('auctionModal');
        const addAuctionBtn = document.getElementById('addAuctionBtn');
        const cancelModalBtn = document.getElementById('cancelModalBtn');
        const auctionForm = document.getElementById('auctionForm');
        const modalTitle = document.getElementById('modalTitle');
        const successModal = document.getElementById('successModal');
        
        // Show modal for adding auction
        addAuctionBtn.addEventListener('click', () => {
            modalTitle.textContent = 'Add Auction Time';
            document.getElementById('auctionId').value = '';
            document.getElementById('openTime').value = '';
            document.getElementById('closeAfter').value = '';
            auctionModal.classList.add('active');
        });
        
        // Edit auction function
        window.editAuction = function(id, openTime, closeAfter) {
            modalTitle.textContent = 'Edit Auction Time';
            document.getElementById('auctionId').value = id;
            
            // Convert time to HH:MM format for input
            const timeParts = openTime.split(':');
            const formattedTime = timeParts[0] + ':' + timeParts[1];
            document.getElementById('openTime').value = formattedTime;
            
            document.getElementById('closeAfter').value = closeAfter;
            auctionModal.classList.add('active');
        };
        
        // Close modal
        cancelModalBtn.addEventListener('click', () => {
            auctionModal.classList.remove('active');
        });
        
        // Close modal when clicking outside
        auctionModal.addEventListener('click', (e) => {
            if (e.target === auctionModal) {
                auctionModal.classList.remove('active');
            }
        });
        
        // Show success message
        function showSuccessMessage(message) {
            document.getElementById('successMessage').textContent = message;
            successModal.classList.add('active');
            
            setTimeout(() => {
                successModal.classList.remove('active');
            }, 2000);
        }
        
        // Form submission
        auctionForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const auctionId = document.getElementById('auctionId').value;
            const openTime = document.getElementById('openTime').value;
            const closeAfter = document.getElementById('closeAfter').value;
            
            // Validate inputs
            if (!openTime || !closeAfter) {
                showSuccessMessage('Please fill in all fields');
                return;
            }
            
            // Prepare form data
            const formData = new FormData();
            formData.append('action', auctionId ? 'update' : 'add');
            formData.append('auction_id', auctionId);
            formData.append('open_time', openTime);
            formData.append('close_after', closeAfter);
            
            try {
                const response = await fetch('b_auction.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    // Close modal
                    auctionModal.classList.remove('active');
                    
                    // Show success message
                    showSuccessMessage(data.message);
                    
                    // Reload page after 1.5 seconds to show updated data
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showSuccessMessage(data.message || 'Operation failed');
                }
            } catch (error) {
                console.error('Error:', error);
                showSuccessMessage('An error occurred. Please try again.');
            }
        });
        
        // Delete auction function
        window.deleteAuction = async function(id) {
            if (!confirm('Are you sure you want to delete this auction time?')) {
                return;
            }
            
            try {
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('auction_id', id);
                
                const response = await fetch('b_auction.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showSuccessMessage(data.message);
                    
                    // Reload page after 1.5 seconds to show updated data
                    setTimeout(() => {
                        location.reload();
                    }, 1500);
                } else {
                    showSuccessMessage(data.message || 'Delete failed');
                }
            } catch (error) {
                console.error('Error:', error);
                showSuccessMessage('An error occurred. Please try again.');
            }
        };
        
        // Auto-refresh page every 5 minutes to update timer
        setTimeout(() => {
            location.reload();
        }, 5 * 60 * 1000);
    </script>
</body>
</html>