<?php
// b_announcement.php
session_start();

// Set headers first
header('Content-Type: application/json; charset=utf-8');

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle GET requests
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
    
    try {
        switch ($action) {
            case 'get_announcement':
                $id = $_GET['id'] ?? 0;
                if (!$id) {
                    throw new Exception('Announcement ID is required');
                }
                
                $stmt = $pdo->prepare("SELECT * FROM announcement WHERE id = :id");
                $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                $stmt->execute();
                $announcement = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($announcement) {
                    echo json_encode(['success' => true, 'announcement' => $announcement]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Announcement not found']);
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
                break;
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Try form data if JSON fails
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    
    try {
        switch ($action) {
            case 'add_announcement':
                // Validate required fields
                $required_fields = ['type', 'text'];
                $missing_fields = [];
                foreach ($required_fields as $field) {
                    if (!isset($input[$field]) || trim($input[$field]) === '') {
                        $missing_fields[] = $field;
                    }
                }
                
                if (!empty($missing_fields)) {
                    throw new Exception('Missing required fields: ' . implode(', ', $missing_fields));
                }
                
                // Validate type
                $valid_types = ['alert', 'scrolling'];
                if (!in_array($input['type'], $valid_types)) {
                    throw new Exception('Invalid announcement type');
                }
                
                // Prepare SQL
                $sql = "INSERT INTO announcement (type, text, link, action_text, expired_at) 
                        VALUES (:type, :text, :link, :action_text, :expired_at)";
                
                $stmt = $pdo->prepare($sql);
                
                // Bind parameters
                $stmt->bindParam(':type', $input['type']);
                $stmt->bindParam(':text', $input['text']);
                
                $link = !empty($input['link']) ? $input['link'] : null;
                $stmt->bindParam(':link', $link);
                
                $action_text = !empty($input['action_text']) ? $input['action_text'] : null;
                $stmt->bindParam(':action_text', $action_text);
                
                // Handle expiration date
                if (!empty($input['expired_at'])) {
                    $expired_at = date('Y-m-d H:i:s', strtotime($input['expired_at']));
                } else {
                    $expired_at = null;
                }
                $stmt->bindParam(':expired_at', $expired_at);
                
                $result = $stmt->execute();
                
                if ($result) {
                    echo json_encode(['success' => true, 'message' => 'Announcement added successfully']);
                } else {
                    throw new Exception('Failed to add announcement');
                }
                break;
                
            case 'update_announcement':
                $id = $input['id'] ?? 0;
                if (!$id) {
                    throw new Exception('Announcement ID is required');
                }
                
                // Validate required fields
                $required_fields = ['type', 'text'];
                $missing_fields = [];
                foreach ($required_fields as $field) {
                    if (!isset($input[$field]) || trim($input[$field]) === '') {
                        $missing_fields[] = $field;
                    }
                }
                
                if (!empty($missing_fields)) {
                    throw new Exception('Missing required fields: ' . implode(', ', $missing_fields));
                }
                
                // Validate type
                $valid_types = ['alert', 'scrolling'];
                if (!in_array($input['type'], $valid_types)) {
                    throw new Exception('Invalid announcement type');
                }
                
                // Check if announcement exists
                $checkStmt = $pdo->prepare("SELECT id FROM announcement WHERE id = :id");
                $checkStmt->bindParam(':id', $id, PDO::PARAM_INT);
                $checkStmt->execute();
                
                if (!$checkStmt->fetch()) {
                    throw new Exception('Announcement not found');
                }
                
                // Prepare SQL
                $sql = "UPDATE announcement SET 
                        type = :type,
                        text = :text,
                        link = :link,
                        action_text = :action_text,
                        expired_at = :expired_at
                        WHERE id = :id";
                
                $stmt = $pdo->prepare($sql);
                
                // Bind parameters
                $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                $stmt->bindParam(':type', $input['type']);
                $stmt->bindParam(':text', $input['text']);
                
                $link = !empty($input['link']) ? $input['link'] : null;
                $stmt->bindParam(':link', $link);
                
                $action_text = !empty($input['action_text']) ? $input['action_text'] : null;
                $stmt->bindParam(':action_text', $action_text);
                
                // Handle expiration date
                if (!empty($input['expired_at'])) {
                    $expired_at = date('Y-m-d H:i:s', strtotime($input['expired_at']));
                } else {
                    $expired_at = null;
                }
                $stmt->bindParam(':expired_at', $expired_at);
                
                $result = $stmt->execute();
                
                if ($result && $stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Announcement updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'No changes made']);
                }
                break;
                
            case 'delete_announcement':
                $id = $input['id'] ?? 0;
                if (!$id) {
                    throw new Exception('Announcement ID is required');
                }
                
                // Check if announcement exists
                $checkStmt = $pdo->prepare("SELECT id FROM announcement WHERE id = :id");
                $checkStmt->bindParam(':id', $id, PDO::PARAM_INT);
                $checkStmt->execute();
                
                if (!$checkStmt->fetch()) {
                    throw new Exception('Announcement not found');
                }
                
                $stmt = $pdo->prepare("DELETE FROM announcement WHERE id = :id");
                $stmt->bindParam(':id', $id, PDO::PARAM_INT);
                $result = $stmt->execute();
                
                if ($result) {
                    echo json_encode(['success' => true, 'message' => 'Announcement deleted successfully']);
                } else {
                    throw new Exception('Failed to delete announcement');
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action: ' . $action]);
                break;
        }
    } catch (PDOException $e) {
        // Log the error for debugging
        error_log("Database Error in b_announcement.php: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Database error occurred']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

echo json_encode(['success' => false, 'message' => 'Invalid request method. Use GET or POST.']);
exit;
?>