<?php
// b_auction.php
session_start();

// Check if user is logged in and has moderator/root role
if (!isset($_SESSION['admin_id']) || 
    !isset($_SESSION['admin_role']) || 
    ($_SESSION['admin_role'] !== 'moderator' && $_SESSION['admin_role'] !== 'root')) {
    
    echo json_encode([
        'success' => false,
        'message' => 'Access denied. Moderator or root privileges required.'
    ]);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method.'
    ]);
    exit;
}

// Get action
$action = $_POST['action'] ?? '';
$auction_id = $_POST['auction_id'] ?? '';

// Validate action
if (!in_array($action, ['add', 'update', 'delete'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid action.'
    ]);
    exit;
}

try {
    if ($action === 'add' || $action === 'update') {
        // Validate required fields
        $open_time = $_POST['open_time'] ?? '';
        $close_after = $_POST['close_after'] ?? '';
        
        if (empty($open_time) || empty($close_after)) {
            echo json_encode([
                'success' => false,
                'message' => 'All fields are required.'
            ]);
            exit;
        }
        
        // Validate time format (HH:MM)
        if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $open_time)) {
            echo json_encode([
                'success' => false,
                'message' => 'Invalid time format. Use HH:MM format.'
            ]);
            exit;
        }
        
        // Validate close_after (1-1440 minutes = 24 hours)
        $close_after = intval($close_after);
        if ($close_after < 1 || $close_after > 1440) {
            echo json_encode([
                'success' => false,
                'message' => 'Duration must be between 1 and 1440 minutes.'
            ]);
            exit;
        }
        
        // Format time to HH:MM:SS
        $open_time = $open_time . ':00';
        
        if ($action === 'add') {
            // Check if time already exists
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM auction_time WHERE open_at = ?");
            $stmt->execute([$open_time]);
            $result = $stmt->fetch();
            
            if ($result['count'] > 0) {
                echo json_encode([
                    'success' => false,
                    'message' => 'An auction already exists at this time.'
                ]);
                exit;
            }
            
            // Add new auction time
            $stmt = $pdo->prepare("INSERT INTO auction_time (open_at, close_after) VALUES (?, ?)");
            $stmt->execute([$open_time, $close_after]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Auction time added successfully!'
            ]);
            
        } elseif ($action === 'update') {
            // Validate auction_id
            if (empty($auction_id)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Auction ID is required for update.'
                ]);
                exit;
            }
            
            // Check if auction exists
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM auction_time WHERE id = ?");
            $stmt->execute([$auction_id]);
            $result = $stmt->fetch();
            
            if ($result['count'] === 0) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Auction not found.'
                ]);
                exit;
            }
            
            // Check if time already exists (excluding current auction)
            $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM auction_time WHERE open_at = ? AND id != ?");
            $stmt->execute([$open_time, $auction_id]);
            $result = $stmt->fetch();
            
            if ($result['count'] > 0) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Another auction already exists at this time.'
                ]);
                exit;
            }
            
            // Update auction time
            $stmt = $pdo->prepare("UPDATE auction_time SET open_at = ?, close_after = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?");
            $stmt->execute([$open_time, $close_after, $auction_id]);
            
            echo json_encode([
                'success' => true,
                'message' => 'Auction time updated successfully!'
            ]);
        }
        
    } elseif ($action === 'delete') {
        // Validate auction_id
        if (empty($auction_id)) {
            echo json_encode([
                'success' => false,
                'message' => 'Auction ID is required for deletion.'
            ]);
            exit;
        }
        
        // Check if auction exists
        $stmt = $pdo->prepare("SELECT COUNT(*) as count FROM auction_time WHERE id = ?");
        $stmt->execute([$auction_id]);
        $result = $stmt->fetch();
        
        if ($result['count'] === 0) {
            echo json_encode([
                'success' => false,
                'message' => 'Auction not found.'
            ]);
            exit;
        }
        
        // Delete auction time
        $stmt = $pdo->prepare("DELETE FROM auction_time WHERE id = ?");
        $stmt->execute([$auction_id]);
        
        echo json_encode([
            'success' => true,
            'message' => 'Auction time deleted successfully!'
        ]);
    }
    
} catch (PDOException $e) {
    // Log error for debugging
    error_log("Auction management error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'Database error. Please try again.'
    ]);
    exit;
}
?>