<?php
// b_kyc.php - Backend for KYC operations
session_start();
header('Content-Type: application/json');

// Check if user is logged in and is root admin
if (!isset($_SESSION['admin_id']) || !isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Get POST data
$action = $_POST['action'] ?? '';

try {
    switch ($action) {
        case 'get_users':
            getUsers($pdo);
            break;
            
        case 'approve':
            approveVerification($pdo);
            break;
            
        case 'reject':
            rejectVerification($pdo);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

// Get users based on status
function getUsers($pdo) {
    $status = $_POST['status'] ?? 'pending';
    $search = $_POST['search'] ?? '';
    
    try {
        // Get stats first
        $stats = getStats($pdo);
        
        if ($status === 'verified') {
            // Get verified users from users table
            $sql = "SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, 
                           u.phone_number, u.country, u.profile_url,
                           v.document_type, v.document_url, v.liveness_url, v.verified_at
                    FROM users u
                    LEFT JOIN verification v ON u.user_id = v.user_id AND v.status = 'verified'
                    WHERE u.verification = 'verified'";
            
            $params = [];
            
            if (!empty($search)) {
                $sql .= " AND (u.email LIKE ? OR u.first_name LIKE ?)";
                $searchTerm = "%$search%";
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            $sql .= " ORDER BY v.verified_at DESC, u.created_at DESC";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
        } else {
            // Get pending verifications
            $sql = "SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, 
                           u.phone_number, u.country, u.profile_url,
                           v.id as verification_id, v.document_type, v.document_url, 
                           v.liveness_url, v.status, v.created_at as verification_date
                    FROM users u
                    INNER JOIN verification v ON u.user_id = v.user_id
                    WHERE v.status = 'pending'";
            
            $params = [];
            
            if (!empty($search)) {
                $sql .= " AND (u.email LIKE ? OR u.first_name LIKE ?)";
                $searchTerm = "%$search%";
                $params[] = $searchTerm;
                $params[] = $searchTerm;
            }
            
            $sql .= " ORDER BY v.created_at DESC";
            
            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        }
        
        echo json_encode([
            'success' => true,
            'users' => $users,
            'count' => count($users),
            'stats' => $stats
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Error fetching users: ' . $e->getMessage()]);
    }
}

// Approve verification
function approveVerification($pdo) {
    $userId = $_POST['user_id'] ?? 0;
    $verificationId = $_POST['verification_id'] ?? 0;
    
    if (!$userId || !$verificationId) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        return;
    }
    
    try {
        $pdo->beginTransaction();
        
        // Update verification table
        $stmt = $pdo->prepare("UPDATE verification SET status = 'verified', verified_at = NOW() WHERE id = ? AND user_id = ?");
        $stmt->execute([$verificationId, $userId]);
        
        if ($stmt->rowCount() === 0) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Verification record not found']);
            return;
        }
        
        // Update user table
        $stmt = $pdo->prepare("UPDATE users SET verification = 'verified' WHERE user_id = ?");
        $stmt->execute([$userId]);
        
        $pdo->commit();
        
        // Get updated stats
        $stats = getStats($pdo);
        
        echo json_encode([
            'success' => true,
            'message' => 'Verification approved successfully',
            'stats' => $stats
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Error approving verification: ' . $e->getMessage()]);
    }
}

// Reject verification
function rejectVerification($pdo) {
    $userId = $_POST['user_id'] ?? 0;
    $verificationId = $_POST['verification_id'] ?? 0;
    $rejectionReason = $_POST['rejection_reason'] ?? '';
    
    if (!$userId || !$verificationId) {
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        return;
    }
    
    if (empty($rejectionReason)) {
        echo json_encode(['success' => false, 'message' => 'Rejection reason is required']);
        return;
    }
    
    try {
        // Update verification table only (don't change users table)
        $stmt = $pdo->prepare("UPDATE verification SET status = 'rejected', rejection_reason = ? WHERE id = ? AND user_id = ?");
        $stmt->execute([$rejectionReason, $verificationId, $userId]);
        
        if ($stmt->rowCount() === 0) {
            echo json_encode(['success' => false, 'message' => 'Verification record not found']);
            return;
        }
        
        // Get updated stats
        $stats = getStats($pdo);
        
        echo json_encode([
            'success' => true,
            'message' => 'Verification rejected successfully',
            'stats' => $stats
        ]);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Error rejecting verification: ' . $e->getMessage()]);
    }
}

// Get statistics
function getStats($pdo) {
    try {
        // Get verified users count
        $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM users WHERE verification = 'verified'");
        $stmt->execute();
        $verified = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Get pending verification count
        $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM verification WHERE status = 'pending'");
        $stmt->execute();
        $pending = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return [
            'verified' => $verified['total'] ?? 0,
            'pending' => $pending['total'] ?? 0
        ];
        
    } catch (Exception $e) {
        return ['verified' => 0, 'pending' => 0];
    }
}
?>