<?php
// b_profile.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Function to send API request silently
function sendRefundApiRequest($buyer_user_id) {
    $url = 'https://biddersconnectsafrica.com/auto_refund.php';
    $data = [
        'user_id' => $buyer_user_id,
        'message' => 'users need refund'
    ];
    
    $options = [
        'http' => [
            'header'  => "Content-Type: application/json\r\n",
            'method'  => 'POST',
            'content' => json_encode($data),
            'timeout' => 5, // 5 second timeout
            'ignore_errors' => true // Don't throw errors on HTTP errors
        ]
    ];
    
    $context = stream_context_create($options);
    @file_get_contents($url, false, $context); // Silent API call with @ to suppress errors
}

// Handle GET requests
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
    
    try {
        switch ($action) {
            case 'get_bank_account':
                $bank_id = $_GET['bank_id'] ?? 0;
                if (!$bank_id) {
                    throw new Exception('Bank ID is required');
                }
                
                $stmt = $pdo->prepare("SELECT * FROM bank_accounts WHERE id = :bank_id");
                $stmt->bindParam(':bank_id', $bank_id, PDO::PARAM_INT);
                $stmt->execute();
                $bank = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($bank) {
                    echo json_encode(['success' => true, 'bank' => $bank]);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Bank account not found']);
                }
                break;
                
            case 'get_buyer_orders':
                $buy_order_id = $_GET['buy_order_id'] ?? 0;
                if (!$buy_order_id) {
                    throw new Exception('Buy Order ID is required');
                }
                
                $stmt = $pdo->prepare("
                    SELECT o.*, 
                           so.user_id as seller_user_id,
                           CONCAT(u.first_name, ' ', u.last_name) as seller_name
                    FROM orders o
                    JOIN seller_orders so ON o.seller_id = so.id
                    JOIN users u ON so.user_id = u.user_id
                    WHERE o.buyer_id = :buy_order_id
                    ORDER BY o.created_at DESC
                ");
                $stmt->bindParam(':buy_order_id', $buy_order_id, PDO::PARAM_INT);
                $stmt->execute();
                $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'orders' => $orders]);
                break;
                
            case 'get_seller_orders':
                $sell_order_id = $_GET['sell_order_id'] ?? 0;
                if (!$sell_order_id) {
                    throw new Exception('Sell Order ID is required');
                }
                
                $stmt = $pdo->prepare("
                    SELECT o.*, 
                           bo.user_id as buyer_user_id,
                           CONCAT(u.first_name, ' ', u.last_name) as buyer_name
                    FROM orders o
                    JOIN buyer_orders bo ON o.buyer_id = bo.id
                    JOIN users u ON bo.user_id = u.user_id
                    WHERE o.seller_id = :sell_order_id
                    ORDER BY o.created_at DESC
                ");
                $stmt->bindParam(':sell_order_id', $sell_order_id, PDO::PARAM_INT);
                $stmt->execute();
                $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'orders' => $orders]);
                break;
                
            case 'load_more_orders':
                $user_id = $_GET['user_id'] ?? 0;
                $type = $_GET['type'] ?? '';
                $offset = $_GET['offset'] ?? 5;
                
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                if ($type === 'buyer') {
                    $stmt = $pdo->prepare("
                        SELECT * FROM buyer_orders 
                        WHERE user_id = :user_id 
                        ORDER BY created_at DESC 
                        LIMIT :offset, 10
                    ");
                } else if ($type === 'seller') {
                    $stmt = $pdo->prepare("
                        SELECT * FROM seller_orders 
                        WHERE user_id = :user_id 
                        ORDER BY created_at DESC 
                        LIMIT :offset, 10
                    ");
                } else {
                    throw new Exception('Invalid order type');
                }
                
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->bindParam(':offset', $offset, PDO::PARAM_INT);
                $stmt->execute();
                $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'orders' => $orders]);
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
                break;
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Try form data if JSON fails
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    
    try {
        switch ($action) {
            case 'update_profile':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                // Update user profile
                $stmt = $pdo->prepare("
                    UPDATE users SET 
                        first_name = :first_name,
                        last_name = :last_name,
                        username = :username,
                        email = :email,
                        phone_number = :phone_number,
                        dob = :dob,
                        country = :country,
                        account_status = :account_status,
                        verification = :verification,
                        profile_url = :profile_url
                    WHERE user_id = :user_id
                ");
                
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->bindParam(':first_name', $input['first_name']);
                $stmt->bindParam(':last_name', $input['last_name']);
                $stmt->bindParam(':username', $input['username']);
                $stmt->bindParam(':email', $input['email']);
                $stmt->bindParam(':phone_number', $input['phone_number']);
                $stmt->bindParam(':dob', $input['dob']);
                $stmt->bindParam(':country', $input['country']);
                $stmt->bindParam(':account_status', $input['account_status']);
                $stmt->bindParam(':verification', $input['verification']);
                $stmt->bindParam(':profile_url', $input['profile_url'] ?: null);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Profile updated successfully']);
                break;
                
            case 'suspend':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                $stmt = $pdo->prepare("UPDATE users SET account_status = 'suspended' WHERE user_id = :user_id");
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'User suspended successfully']);
                break;
                
            case 'activate':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                $stmt = $pdo->prepare("UPDATE users SET account_status = 'active' WHERE user_id = :user_id");
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'User activated successfully']);
                break;
                
            case 'add_bank_account':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                $stmt = $pdo->prepare("
                    INSERT INTO bank_accounts (user_id, bank_name, account_number, account_name, bank_code) 
                    VALUES (:user_id, :bank_name, :account_number, :account_name, :bank_code)
                ");
                
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->bindParam(':bank_name', $input['bank_name']);
                $stmt->bindParam(':account_number', $input['account_number']);
                $stmt->bindParam(':account_name', $input['account_name']);
                $stmt->bindParam(':bank_code', $input['bank_code']);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Bank account added successfully']);
                break;
                
            case 'update_bank_account':
                $bank_id = $input['bank_id'] ?? 0;
                if (!$bank_id) {
                    throw new Exception('Bank ID is required');
                }
                
                $stmt = $pdo->prepare("
                    UPDATE bank_accounts SET 
                        bank_name = :bank_name,
                        account_number = :account_number,
                        account_name = :account_name,
                        bank_code = :bank_code
                    WHERE id = :bank_id AND user_id = :user_id
                ");
                
                $stmt->bindParam(':bank_id', $bank_id, PDO::PARAM_INT);
                $stmt->bindParam(':user_id', $input['user_id'], PDO::PARAM_INT);
                $stmt->bindParam(':bank_name', $input['bank_name']);
                $stmt->bindParam(':account_number', $input['account_number']);
                $stmt->bindParam(':account_name', $input['account_name']);
                $stmt->bindParam(':bank_code', $input['bank_code']);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Bank account updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Bank account not found or unauthorized']);
                }
                break;
                
            case 'delete_bank_account':
                $bank_id = $input['bank_id'] ?? 0;
                if (!$bank_id) {
                    throw new Exception('Bank ID is required');
                }
                
                $stmt = $pdo->prepare("DELETE FROM bank_accounts WHERE id = :bank_id AND user_id = :user_id");
                $stmt->bindParam(':bank_id', $bank_id, PDO::PARAM_INT);
                $stmt->bindParam(':user_id', $input['user_id'], PDO::PARAM_INT);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Bank account deleted successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Bank account not found or unauthorized']);
                }
                break;
                
            case 'approve_testimony':
                $testimony_id = $input['testimony_id'] ?? 0;
                if (!$testimony_id) {
                    throw new Exception('Testimony ID is required');
                }
                
                $stmt = $pdo->prepare("UPDATE testimony SET status = 'approved' WHERE id = :testimony_id");
                $stmt->bindParam(':testimony_id', $testimony_id, PDO::PARAM_INT);
                $stmt->execute();
                
                if ($stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Testimony approved successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'Testimony not found']);
                }
                break;
                
            case 'toggle_seller_order_status':
                $order_id = $input['order_id'] ?? 0;
                if (!$order_id) {
                    throw new Exception('Order ID is required');
                }
                
                // Get current status
                $stmt = $pdo->prepare("SELECT status FROM seller_orders WHERE id = :order_id");
                $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
                $stmt->execute();
                $order = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$order) {
                    echo json_encode(['success' => false, 'message' => 'Order not found']);
                    return;
                }
                
                // Toggle status
                $new_status = ($order['status'] === 'pause') ? 'in progress' : 'pause';
                
                $stmt = $pdo->prepare("UPDATE seller_orders SET status = :status WHERE id = :order_id");
                $stmt->bindParam(':status', $new_status);
                $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
                $stmt->execute();
                
                echo json_encode(['success' => true, 'message' => 'Order status updated', 'new_status' => $new_status]);
                break;
                
            case 'approve_payment':
                $order_match_id = $input['order_id'] ?? 0;
                $buyer_id = $input['buyer_id'] ?? 0;
                
                if (!$order_match_id || !$buyer_id) {
                    echo json_encode(['success' => false, 'message' => 'Order ID and Buyer ID required']);
                    return;
                }
                
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // First verify that this order exists
                    $stmt = $pdo->prepare("
                        SELECT o.id, o.seller_id, o.status 
                        FROM orders o
                        WHERE o.id = :order_match_id 
                        AND o.buyer_id = :buyer_id
                    ");
                    $stmt->bindParam(':order_match_id', $order_match_id);
                    $stmt->bindParam(':buyer_id', $buyer_id);
                    $stmt->execute();
                    $order_info = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$order_info) {
                        $pdo->rollBack();
                        echo json_encode(['success' => false, 'message' => 'Order not found']);
                        return;
                    }
                    
                    if ($order_info['status'] === 'approved') {
                        $pdo->rollBack();
                        echo json_encode(['success' => false, 'message' => 'Payment already approved']);
                        return;
                    }
                    
                    $seller_id = $order_info['seller_id'];
                    
                    // Update the order status to approved
                    $stmt = $pdo->prepare("
                        UPDATE orders 
                        SET status = 'approved', 
                            updated_at = CURRENT_TIMESTAMP 
                        WHERE id = :order_match_id 
                        AND buyer_id = :buyer_id
                    ");
                    $stmt->bindParam(':order_match_id', $order_match_id);
                    $stmt->bindParam(':buyer_id', $buyer_id);
                    $stmt->execute();
                    
                    if ($stmt->rowCount() === 0) {
                        $pdo->rollBack();
                        echo json_encode(['success' => false, 'message' => 'Failed to approve payment']);
                        return;
                    }
                    
                    // Check if this approved order is the last pending approval for this buyer_id
                    $stmt = $pdo->prepare("
                        SELECT COUNT(*) as pending_count 
                        FROM orders 
                        WHERE buyer_id = :buyer_id 
                        AND status = 'not approved'
                    ");
                    $stmt->bindParam(':buyer_id', $buyer_id);
                    $stmt->execute();
                    $pending_count = $stmt->fetch(PDO::FETCH_ASSOC)['pending_count'];
                    
                    // If no more pending approvals for this buyer_id (this was the last one)
                    if ($pending_count == 0) {
                        // Calculate maturity date: 14 days from now
                        $mature_at = date('Y-m-d H:i:s', strtotime('+14 days'));
                        
                        // Get buyer details to check if this is their first/only buy order
                        $stmt = $pdo->prepare("
                            SELECT bo.user_id, bo.amount 
                            FROM buyer_orders bo 
                            WHERE bo.id = :buyer_id
                        ");
                        $stmt->bindParam(':buyer_id', $buyer_id);
                        $stmt->execute();
                        $buyer_details = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        $buyer_user_id = $buyer_details['user_id'];
                        $buyer_amount = floatval($buyer_details['amount']);
                        
                        // Mark the buy_id in the buyer_orders table as approved and update maturity date
                        $stmt = $pdo->prepare("
                            UPDATE buyer_orders 
                            SET status = 'approved',
                                mature_at = :mature_at,
                                updated_at = CURRENT_TIMESTAMP
                            WHERE id = :buyer_id
                        ");
                        $stmt->bindParam(':buyer_id', $buyer_id);
                        $stmt->bindParam(':mature_at', $mature_at);
                        $stmt->execute();
                        
                        // Check if this buy_id exists as primary_buy_id in any recommit record
                        $stmt = $pdo->prepare("
                            SELECT id, status, buy_id 
                            FROM recommit 
                            WHERE primary_buy_id = :buyer_id
                            ORDER BY created_at DESC
                            LIMIT 1
                        ");
                        $stmt->bindParam(':buyer_id', $buyer_id);
                        $stmt->execute();
                        $recommit_record = $stmt->fetch(PDO::FETCH_ASSOC);
                        
                        // If recommit record exists where this buy_id is the primary_buy_id, mark that recommit record as successful
                        if ($recommit_record) {
                            $stmt = $pdo->prepare("
                                UPDATE recommit 
                                SET status = 'successful',
                                    updated_at = CURRENT_TIMESTAMP
                                WHERE id = :recommit_id
                            ");
                            $stmt->bindParam(':recommit_id', $recommit_record['id']);
                            $stmt->execute();
                        }
                        
                        // Check if this is the only buy order for this buyer (new user check)
                        $stmt = $pdo->prepare("
                            SELECT COUNT(*) as total_buy_orders 
                            FROM buyer_orders 
                            WHERE user_id = :buyer_user_id 
                            AND status = 'approved'
                        ");
                        $stmt->bindParam(':buyer_user_id', $buyer_user_id);
                        $stmt->execute();
                        $total_approved_orders = $stmt->fetch(PDO::FETCH_ASSOC)['total_buy_orders'];
                        
                        // If this is the first approved buy order for this user (new user)
                        if ($total_approved_orders == 1) {
                            // Check if this user is in referrals table as referred_user_id
                            $stmt = $pdo->prepare("
                                SELECT id, referrer_user_id, commission 
                                FROM referrals 
                                WHERE referred_user_id = :buyer_user_id
                            ");
                            $stmt->bindParam(':buyer_user_id', $buyer_user_id);
                            $stmt->execute();
                            $referral_record = $stmt->fetch(PDO::FETCH_ASSOC);
                            
                            // If referral record exists and commission is 'pending'
                            if ($referral_record && $referral_record['commission'] == 'pending') {
                                $referral_id = $referral_record['id'];
                                $referrer_user_id = $referral_record['referrer_user_id'];
                                
                                // Mark commission as 'paid'
                                $stmt = $pdo->prepare("
                                    UPDATE referrals 
                                    SET commission = 'paid' 
                                    WHERE id = :referral_id
                                ");
                                $stmt->bindParam(':referral_id', $referral_id);
                                $stmt->execute();
                                
                                // Calculate 5% commission
                                $commission_amount = $buyer_amount * 0.05;
                                
                                // Credit the commission to referrer's balance in referral_link table
                                $stmt = $pdo->prepare("
                                    UPDATE referral_link 
                                    SET balance = balance + :commission_amount 
                                    WHERE user_id = :referrer_user_id
                                ");
                                $stmt->bindParam(':commission_amount', $commission_amount);
                                $stmt->bindParam(':referrer_user_id', $referrer_user_id);
                                $stmt->execute();
                            }
                            
                            // Check security deposit for this buyer
                            $stmt = $pdo->prepare("
                                SELECT id, status 
                                FROM security_deposit 
                                WHERE user_id = :buyer_user_id 
                                ORDER BY created_at DESC 
                                LIMIT 1
                            ");
                            $stmt->bindParam(':buyer_user_id', $buyer_user_id);
                            $stmt->execute();
                            $security_deposit = $stmt->fetch(PDO::FETCH_ASSOC);
                            
                            // If security deposit record exists and status is 'paid' (not 'refunded')
                            if ($security_deposit && $security_deposit['status'] == 'paid') {
                                // Send API request for refund
                                sendRefundApiRequest($buyer_user_id);
                            }
                        }
                    }
                    
                    // Check if this is the last buyer for this seller order and amount_remaining is 0
                    $stmt = $pdo->prepare("
                        SELECT so.amount_remaining, 
                               (SELECT COUNT(*) FROM orders o2 WHERE o2.seller_id = :seller_id AND o2.status = 'not approved') as pending_seller_orders
                        FROM seller_orders so
                        WHERE so.id = :seller_id
                    ");
                    $stmt->bindParam(':seller_id', $seller_id);
                    $stmt->execute();
                    $seller_info = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if ($seller_info) {
                        $amount_remaining = floatval($seller_info['amount_remaining']);
                        $pending_seller_orders = intval($seller_info['pending_seller_orders']);
                        
                        // If amount remaining is 0 and there are no pending orders for this seller, mark as sold
                        if ($amount_remaining == 0 && $pending_seller_orders == 0) {
                            $stmt = $pdo->prepare("
                                UPDATE seller_orders 
                                SET status = 'sold',
                                    updated_at = CURRENT_TIMESTAMP
                                WHERE id = :seller_id
                            ");
                            $stmt->bindParam(':seller_id', $seller_id);
                            $stmt->execute();
                        }
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    echo json_encode([
                        'success' => true,
                        'message' => 'Payment approved successfully'
                    ]);
                    
                } catch (Exception $e) {
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
                break;
        }
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

echo json_encode(['success' => false, 'message' => 'Invalid request method']);
?>