<?php
// b_testimony.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    echo json_encode([
        'success' => false,
        'message' => 'Not authenticated'
    ]);
    exit;
}

// Include database connection
require_once 'php/db.php';

header('Content-Type: application/json');

// Get action
$action = $_POST['action'] ?? '';

// Handle different actions
switch ($action) {
    case 'get_testimonies':
        getTestimonies();
        break;
    case 'approve':
    case 'reject':
        updateTestimonyStatus();
        break;
    default:
        echo json_encode([
            'success' => false,
            'message' => 'Invalid action'
        ]);
        exit;
}

// Get testimonies based on status
function getTestimonies() {
    global $pdo;
    
    $status = $_POST['status'] ?? 'pending';
    
    // Validate status
    $valid_statuses = ['pending', 'approved', 'rejected', 'waiting'];
    if (!in_array($status, $valid_statuses)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid status'
        ]);
        exit;
    }
    
    try {
        // Get all testimonies with user details for the specified status
        $sql = "
            SELECT 
                t.*,
                o.amount,
                u.first_name,
                u.last_name,
                u.profile_url,
                CONCAT(u.first_name, ' ', u.last_name) as user_name,
                so.status as seller_status
            FROM testimony t
            LEFT JOIN orders o ON t.order_id = o.id
            LEFT JOIN seller_orders so ON o.seller_id = so.id
            LEFT JOIN users u ON so.user_id = u.user_id
            WHERE t.status = ?
            ORDER BY t.created_at DESC
        ";
        
        $stmt = $pdo->prepare($sql);
        $stmt->execute([$status]);
        $testimonies = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get statistics
        $statsStmt = $pdo->prepare("
            SELECT 
                SUM(CASE WHEN status = 'pending' THEN 1 ELSE 0 END) as pending,
                SUM(CASE WHEN status = 'approved' THEN 1 ELSE 0 END) as approved,
                SUM(CASE WHEN status = 'rejected' THEN 1 ELSE 0 END) as rejected,
                SUM(CASE WHEN status = 'waiting' THEN 1 ELSE 0 END) as waiting
            FROM testimony
        ");
        $statsStmt->execute();
        $stats = $statsStmt->fetch(PDO::FETCH_ASSOC);
        
        echo json_encode([
            'success' => true,
            'testimonies' => $testimonies,
            'stats' => $stats
        ]);
        
    } catch (PDOException $e) {
        error_log("Get testimonies error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
        exit;
    }
}

// Update testimony status (approve/reject)
function updateTestimonyStatus() {
    global $pdo;
    
    $action = $_POST['action'];
    $testimony_id = $_POST['testimony_id'] ?? '';
    
    // Validate testimony ID
    if (empty($testimony_id) || !is_numeric($testimony_id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid testimony ID'
        ]);
        exit;
    }
    
    try {
        // Check if testimony exists
        $checkStmt = $pdo->prepare("
            SELECT t.*, o.payment_proof, o.seller_id, u.first_name, u.last_name 
            FROM testimony t
            LEFT JOIN orders o ON t.order_id = o.id
            LEFT JOIN seller_orders so ON o.seller_id = so.id
            LEFT JOIN users u ON so.user_id = u.user_id
            WHERE t.id = ?
        ");
        $checkStmt->execute([$testimony_id]);
        $testimony = $checkStmt->fetch();
        
        if (!$testimony) {
            echo json_encode([
                'success' => false,
                'message' => 'Testimony not found'
            ]);
            exit;
        }
        
        // Check if testimony is already processed (only for pending)
        if ($testimony['status'] !== 'pending') {
            echo json_encode([
                'success' => false,
                'message' => 'Testimony has already been processed'
            ]);
            exit;
        }
        
        // Update testimony status
        $new_status = $action === 'approve' ? 'approved' : 'rejected';
        
        $updateStmt = $pdo->prepare("UPDATE testimony SET status = ?, updated_at = NOW() WHERE id = ?");
        $updateStmt->execute([$new_status, $testimony_id]);
        
        // If approved, send payment proof to Telegram group
        if ($action === 'approve' && !empty($testimony['payment_proof'])) {
            $userName = $testimony['first_name'] . ' ' . $testimony['last_name'];
            sendToTelegram($testimony['payment_proof'], $userName);
        }
        
        echo json_encode([
            'success' => true,
            'message' => 'Testimony ' . ($action === 'approve' ? 'approved' : 'rejected') . ' successfully'
        ]);
        
    } catch (PDOException $e) {
        error_log("Testimony update error: " . $e->getMessage());
        echo json_encode([
            'success' => false,
            'message' => 'Database error'
        ]);
        exit;
    }
}

// Send payment proof to Telegram
function sendToTelegram($paymentProof, $userName) {
    $botToken = '8590468032:AAF62n2m4-4qpdLoPEEvtZaML1fQMbU57BM';
    $chatId = '-1003630239668';
    
    $caption = $userName . " has successfully sold up his coin";
    
    // Send photo with caption
    $telegramUrl = "https://api.telegram.org/bot{$botToken}/sendPhoto";
    $postData = [
        'chat_id' => $chatId,
        'photo' => $paymentProof,
        'caption' => $caption
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $telegramUrl);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, $postData);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_exec($ch);
    curl_close($ch);
}
?>