<?php
// b_update_users.php
session_start();

// Set headers first
header('Content-Type: application/json; charset=utf-8');

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Try form data if JSON fails
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    
    try {
        switch ($action) {
            case 'update_profile':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                // Validate required fields
                $required_fields = ['first_name', 'last_name', 'username', 'email', 'phone_number', 'dob', 'country', 'account_status', 'verification'];
                $missing_fields = [];
                foreach ($required_fields as $field) {
                    if (!isset($input[$field]) || trim($input[$field]) === '') {
                        $missing_fields[] = $field;
                    }
                }
                
                if (!empty($missing_fields)) {
                    throw new Exception('Missing required fields: ' . implode(', ', $missing_fields));
                }
                
                // First, check if user exists
                $checkUserStmt = $pdo->prepare("SELECT user_id FROM users WHERE user_id = :user_id");
                $checkUserStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $checkUserStmt->execute();
                
                if (!$checkUserStmt->fetch()) {
                    throw new Exception('User not found with ID: ' . $user_id);
                }
                
                // Check if username already exists (excluding current user)
                $checkStmt = $pdo->prepare("SELECT user_id FROM users WHERE username = :username AND user_id != :user_id");
                $checkStmt->bindParam(':username', $input['username']);
                $checkStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $checkStmt->execute();
                
                if ($checkStmt->fetch()) {
                    throw new Exception('Username already exists. Please choose a different username.');
                }
                
                // Check if email already exists (excluding current user)
                $checkStmt = $pdo->prepare("SELECT user_id FROM users WHERE email = :email AND user_id != :user_id");
                $checkStmt->bindParam(':email', $input['email']);
                $checkStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $checkStmt->execute();
                
                if ($checkStmt->fetch()) {
                    throw new Exception('Email already exists. Please use a different email.');
                }
                
                // Update user profile
                $stmt = $pdo->prepare("
                    UPDATE users SET 
                        first_name = :first_name,
                        last_name = :last_name,
                        username = :username,
                        email = :email,
                        phone_number = :phone_number,
                        dob = :dob,
                        country = :country,
                        account_status = :account_status,
                        verification = :verification,
                        profile_url = :profile_url
                    WHERE user_id = :user_id
                ");
                
                $profile_url = !empty($input['profile_url']) ? $input['profile_url'] : null;
                
                $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $stmt->bindParam(':first_name', $input['first_name']);
                $stmt->bindParam(':last_name', $input['last_name']);
                $stmt->bindParam(':username', $input['username']);
                $stmt->bindParam(':email', $input['email']);
                $stmt->bindParam(':phone_number', $input['phone_number']);
                $stmt->bindParam(':dob', $input['dob']);
                $stmt->bindParam(':country', $input['country']);
                $stmt->bindParam(':account_status', $input['account_status']);
                $stmt->bindParam(':verification', $input['verification']);
                $stmt->bindParam(':profile_url', $profile_url);
                
                $result = $stmt->execute();
                
                if ($result && $stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Profile updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'No changes made. User may already have these values.']);
                }
                break;
                
            case 'suspend':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                // Check if user exists
                $checkStmt = $pdo->prepare("SELECT user_id, account_status FROM users WHERE user_id = :user_id");
                $checkStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $checkStmt->execute();
                $user = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$user) {
                    throw new Exception('User not found with ID: ' . $user_id);
                }
                
                if ($user['account_status'] === 'suspended') {
                    echo json_encode(['success' => false, 'message' => 'User is already suspended']);
                    break;
                }
                
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Suspend the user
                    $stmt = $pdo->prepare("UPDATE users SET account_status = 'suspended' WHERE user_id = :user_id");
                    $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    // Check if user has any records in seller_orders table
                    $checkSellerOrdersStmt = $pdo->prepare("
                        SELECT id, match_status, status 
                        FROM seller_orders 
                        WHERE user_id = :user_id
                    ");
                    $checkSellerOrdersStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $checkSellerOrdersStmt->execute();
                    $seller_orders = $checkSellerOrdersStmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $paused_orders = 0;
                    
                    if (!empty($seller_orders)) {
                        foreach ($seller_orders as $order) {
                            // Check if match_status is pending
                            if ($order['match_status'] === 'pending') {
                                // Mark the status as pause
                                $updateStmt = $pdo->prepare("
                                    UPDATE seller_orders 
                                    SET status = 'pause' 
                                    WHERE id = :order_id 
                                    AND user_id = :user_id 
                                    AND match_status = 'pending'
                                ");
                                $updateStmt->bindParam(':order_id', $order['id'], PDO::PARAM_INT);
                                $updateStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                                $updateStmt->execute();
                                
                                if ($updateStmt->rowCount() > 0) {
                                    $paused_orders++;
                                }
                            }
                            // If match_status is 'completed', do nothing
                        }
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    $message = 'User suspended successfully';
                    if ($paused_orders > 0) {
                        $message .= " and $paused_orders seller order(s) with pending match status have been paused";
                    }
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => $message,
                        'paused_orders' => $paused_orders
                    ]);
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            case 'activate':
                $user_id = $input['user_id'] ?? 0;
                if (!$user_id) {
                    throw new Exception('User ID is required');
                }
                
                // Check if user exists
                $checkStmt = $pdo->prepare("SELECT user_id, account_status FROM users WHERE user_id = :user_id");
                $checkStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                $checkStmt->execute();
                $user = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$user) {
                    throw new Exception('User not found with ID: ' . $user_id);
                }
                
                if ($user['account_status'] === 'active') {
                    echo json_encode(['success' => false, 'message' => 'User is already active']);
                    break;
                }
                
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Activate the user
                    $stmt = $pdo->prepare("UPDATE users SET account_status = 'active' WHERE user_id = :user_id");
                    $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    // Check if user has any paused seller orders
                    $checkPausedOrdersStmt = $pdo->prepare("
                        SELECT id, match_status 
                        FROM seller_orders 
                        WHERE user_id = :user_id 
                        AND status = 'pause'
                    ");
                    $checkPausedOrdersStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $checkPausedOrdersStmt->execute();
                    $paused_orders = $checkPausedOrdersStmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $resumed_orders = 0;
                    
                    if (!empty($paused_orders)) {
                        foreach ($paused_orders as $order) {
                            // Only resume orders with pending match status
                            if ($order['match_status'] === 'pending') {
                                // Change status back to 'in progress'
                                $updateStmt = $pdo->prepare("
                                    UPDATE seller_orders 
                                    SET status = 'in progress' 
                                    WHERE id = :order_id 
                                    AND user_id = :user_id 
                                    AND match_status = 'pending'
                                    AND status = 'pause'
                                ");
                                $updateStmt->bindParam(':order_id', $order['id'], PDO::PARAM_INT);
                                $updateStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                                $updateStmt->execute();
                                
                                if ($updateStmt->rowCount() > 0) {
                                    $resumed_orders++;
                                }
                            }
                        }
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    $message = 'User activated successfully';
                    if ($resumed_orders > 0) {
                        $message .= " and $resumed_orders paused seller order(s) have been resumed";
                    }
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => $message,
                        'resumed_orders' => $resumed_orders
                    ]);
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action: ' . $action]);
                break;
        }
    } catch (PDOException $e) {
        // Log the error for debugging
        error_log("Database Error in b_update_users.php: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Database error occurred']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

echo json_encode(['success' => false, 'message' => 'Invalid request method. Use POST.']);
exit;
?>