<?php
// b_users.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Get users for listing (GET request)
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    try {
        // Get pagination parameters
        $page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 10;
        $offset = ($page - 1) * $limit;
        $search = isset($_GET['search']) ? trim($_GET['search']) : '';
        $status = isset($_GET['status']) ? trim($_GET['status']) : 'all';
        
        // Build query with search and status conditions
        $whereClauses = [];
        $params = [];
        
        if ($search) {
            $whereClauses[] = "(u.first_name LIKE :search OR u.email LIKE :search)";
            $params[':search'] = "%{$search}%";
        }
        
        if ($status !== 'all') {
            $whereClauses[] = "u.account_status = :status";
            $params[':status'] = $status;
        }
        
        // Combine where clauses
        $whereClause = '';
        if (!empty($whereClauses)) {
            $whereClause = 'WHERE ' . implode(' AND ', $whereClauses);
        }
        
        // Get total count for pagination
        $countQuery = "SELECT COUNT(*) as total FROM users u";
        if ($whereClause) {
            $countQuery .= " " . $whereClause;
        }
        
        $countStmt = $pdo->prepare($countQuery);
        foreach ($params as $key => $value) {
            $countStmt->bindValue($key, $value);
        }
        $countStmt->execute();
        $totalCount = $countStmt->fetch(PDO::FETCH_ASSOC)['total'];
        
        // Get users with pagination and KYC info
        $query = "
            SELECT 
                u.user_id,
                u.username,
                u.first_name,
                u.last_name,
                u.email,
                u.phone_number,
                u.profile_url,
                u.account_status,
                u.verification,
                u.created_at,
                COALESCE(k.email_status, 'not verified') as email_verified,
                COALESCE(k.phone_status, 'not verified') as phone_verified
            FROM users u
            LEFT JOIN basic_kyc k ON u.user_id = k.user_id
        ";
        
        if ($whereClause) {
            $query .= " " . $whereClause;
        }
        
        $query .= " ORDER BY u.created_at DESC LIMIT :limit OFFSET :offset";
        
        $stmt = $pdo->prepare($query);
        
        // Bind all parameters
        foreach ($params as $key => $value) {
            $stmt->bindValue($key, $value);
        }
        $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
        $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
        $stmt->execute();
        $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Get statistics (only without filters or with 'all' status)
        $stats = [];
        
        if (!$search && $status === 'all') {
            // Total users
            $stats['total'] = $totalCount;
            
            // Active users
            $activeStmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE account_status = 'active'");
            $stats['active'] = $activeStmt->fetch(PDO::FETCH_ASSOC)['count'];
            
            // Verified users
            $verifiedStmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE verification = 'verified'");
            $stats['verified'] = $verifiedStmt->fetch(PDO::FETCH_ASSOC)['count'];
            
            // Suspended users
            $suspendedStmt = $pdo->query("SELECT COUNT(*) as count FROM users WHERE account_status = 'suspended'");
            $stats['suspended'] = $suspendedStmt->fetch(PDO::FETCH_ASSOC)['count'];
        } else {
            // For filtered results, use filtered counts
            $stats['total'] = $totalCount;
            $stats['active'] = 0;
            $stats['verified'] = 0;
            $stats['suspended'] = 0;
            
            // Calculate filtered stats
            foreach ($users as $user) {
                if ($user['account_status'] === 'active') $stats['active']++;
                if ($user['account_status'] === 'suspended') $stats['suspended']++;
                if ($user['verification'] === 'verified') $stats['verified']++;
            }
        }
        
        // Check if there are more users
        $hasMore = ($offset + $limit) < $totalCount;
        
        header('Content-Type: application/json');
        echo json_encode([
            'success' => true,
            'users' => $users,
            'stats' => $stats,
            'hasMore' => $hasMore,
            'page' => $page,
            'total' => $totalCount,
            'search' => $search,
            'status' => $status
        ]);
        
    } catch (PDOException $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

// Handle user actions (POST request)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Try form data if JSON fails
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    $userId = $input['user_id'] ?? 0;
    
    if (empty($action) || $userId <= 0) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => 'Invalid request parameters'
        ]);
        exit;
    }
    
    try {
        switch ($action) {
            case 'suspend':
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Check if user exists and current status
                    $checkStmt = $pdo->prepare("SELECT user_id, account_status FROM users WHERE user_id = :user_id");
                    $checkStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $checkStmt->execute();
                    $user = $checkStmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$user) {
                        $pdo->rollBack();
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'message' => 'User not found'
                        ]);
                        exit;
                    }
                    
                    if ($user['account_status'] === 'suspended') {
                        $pdo->rollBack();
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'message' => 'User is already suspended'
                        ]);
                        exit;
                    }
                    
                    // Suspend the user
                    $stmt = $pdo->prepare("UPDATE users SET account_status = 'suspended' WHERE user_id = :user_id");
                    $stmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    // Check if user has any records in seller_orders table
                    $checkSellerOrdersStmt = $pdo->prepare("
                        SELECT id, match_status 
                        FROM seller_orders 
                        WHERE user_id = :user_id
                    ");
                    $checkSellerOrdersStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $checkSellerOrdersStmt->execute();
                    $seller_orders = $checkSellerOrdersStmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $paused_orders = 0;
                    
                    if (!empty($seller_orders)) {
                        foreach ($seller_orders as $order) {
                            // Check if match_status is pending
                            if ($order['match_status'] === 'pending') {
                                // Mark the status as pause
                                $updateStmt = $pdo->prepare("
                                    UPDATE seller_orders 
                                    SET status = 'pause' 
                                    WHERE id = :order_id 
                                    AND user_id = :user_id 
                                    AND match_status = 'pending'
                                ");
                                $updateStmt->bindParam(':order_id', $order['id'], PDO::PARAM_INT);
                                $updateStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                                $updateStmt->execute();
                                
                                if ($updateStmt->rowCount() > 0) {
                                    $paused_orders++;
                                }
                            }
                            // If match_status is 'completed', do nothing
                        }
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    $message = 'User suspended successfully';
                    if ($paused_orders > 0) {
                        $message .= " and $paused_orders seller order(s) with pending match status have been paused";
                    }
                    
                    header('Content-Type: application/json');
                    echo json_encode([
                        'success' => true,
                        'message' => $message,
                        'paused_orders' => $paused_orders
                    ]);
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            case 'activate':
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Check if user exists and current status
                    $checkStmt = $pdo->prepare("SELECT user_id, account_status FROM users WHERE user_id = :user_id");
                    $checkStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $checkStmt->execute();
                    $user = $checkStmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$user) {
                        $pdo->rollBack();
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'message' => 'User not found'
                        ]);
                        exit;
                    }
                    
                    if ($user['account_status'] === 'active') {
                        $pdo->rollBack();
                        header('Content-Type: application/json');
                        echo json_encode([
                            'success' => false,
                            'message' => 'User is already active'
                        ]);
                        exit;
                    }
                    
                    // Activate the user
                    $stmt = $pdo->prepare("UPDATE users SET account_status = 'active' WHERE user_id = :user_id");
                    $stmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $stmt->execute();
                    
                    // Check if user has any paused seller orders
                    $checkPausedOrdersStmt = $pdo->prepare("
                        SELECT id, match_status 
                        FROM seller_orders 
                        WHERE user_id = :user_id 
                        AND status = 'pause'
                    ");
                    $checkPausedOrdersStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                    $checkPausedOrdersStmt->execute();
                    $paused_orders = $checkPausedOrdersStmt->fetchAll(PDO::FETCH_ASSOC);
                    
                    $resumed_orders = 0;
                    
                    if (!empty($paused_orders)) {
                        foreach ($paused_orders as $order) {
                            // Only resume orders with pending match status
                            if ($order['match_status'] === 'pending') {
                                // Change status back to 'in progress'
                                $updateStmt = $pdo->prepare("
                                    UPDATE seller_orders 
                                    SET status = 'in progress' 
                                    WHERE id = :order_id 
                                    AND user_id = :user_id 
                                    AND match_status = 'pending'
                                    AND status = 'pause'
                                ");
                                $updateStmt->bindParam(':order_id', $order['id'], PDO::PARAM_INT);
                                $updateStmt->bindParam(':user_id', $userId, PDO::PARAM_INT);
                                $updateStmt->execute();
                                
                                if ($updateStmt->rowCount() > 0) {
                                    $resumed_orders++;
                                }
                            }
                        }
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    $message = 'User activated successfully';
                    if ($resumed_orders > 0) {
                        $message .= " and $resumed_orders paused seller order(s) have been resumed";
                    }
                    
                    header('Content-Type: application/json');
                    echo json_encode([
                        'success' => true,
                        'message' => $message,
                        'resumed_orders' => $resumed_orders
                    ]);
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            default:
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'message' => 'Invalid action'
                ]);
                break;
        }
        
    } catch (PDOException $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode([
            'success' => false,
            'message' => $e->getMessage()
        ]);
    }
}
?>