<?php
// bonus.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Bonus Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Search Container */
        .search-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .search-box {
            position: relative;
            margin-bottom: 20px;
        }
        
        .search-input {
            width: 100%;
            padding: 15px 20px 15px 50px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 16px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
            background: var(--light);
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .search-icon {
            position: absolute;
            left: 20px;
            top: 50%;
            transform: translateY(-50%);
            color: var(--gray);
            font-size: 18px;
        }
        
        .search-hint {
            font-size: 14px;
            color: var(--gray);
            margin-top: 10px;
            text-align: center;
        }
        
        /* Results Container */
        .results-container {
            display: none;
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .results-container.active {
            display: block;
        }
        
        .results-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .results-header h3 {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .results-count {
            font-size: 14px;
            color: var(--gray);
            background: var(--light);
            padding: 6px 12px;
            border-radius: 20px;
            font-weight: 500;
        }
        
        .user-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .user-card {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .user-card:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
            flex: 1;
        }
        
        .user-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .user-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .user-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 2px;
        }
        
        .user-stats {
            display: flex;
            gap: 15px;
            margin-top: 10px;
        }
        
        .user-stat {
            display: flex;
            flex-direction: column;
        }
        
        .user-stat label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .user-stat span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .user-actions {
            display: flex;
            gap: 10px;
        }
        
        .credit-user-btn {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .credit-user-btn:hover {
            background: linear-gradient(90deg, var(--secondary-dark), #047857);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
        }
        
        /* Modals */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .credit-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .credit-modal {
            transform: translateY(0);
        }
        
        .credit-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .user-display {
            background: var(--light);
            border-radius: 10px;
            padding: 20px;
            margin-bottom: 25px;
            text-align: center;
        }
        
        .user-display h4 {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .user-display p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-group label.required::after {
            content: ' *';
            color: var(--danger);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid var(--gray-light);
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .form-hint {
            font-size: 12px;
            color: var(--gray);
            margin-top: 5px;
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .form-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .form-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .form-btn.primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .form-btn.success {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .form-btn.success:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .form-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .form-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .form-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Loading Spinner */
        .spinner {
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Message Modal */
        .message-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .message-modal {
            transform: translateY(0);
        }
        
        .message-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .message-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .message-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .message-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .message-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .message-actions {
            display: flex;
            justify-content: center;
        }
        
        .message-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .message-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Confirmation Modal */
        .confirm-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .confirm-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .confirm-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .confirm-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .confirm-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .confirm-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .confirm-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .confirm-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .confirm-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .confirm-btn.secondary:hover {
            background: #d1d5db;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .user-card {
                flex-direction: column;
                align-items: flex-start;
                gap: 15px;
            }
            
            .user-actions {
                width: 100%;
                justify-content: flex-end;
            }
            
            .credit-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .message-modal, .confirm-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
        
        @media (max-width: 480px) {
            .form-actions, .confirm-actions {
                flex-direction: column;
            }
            
            .form-btn, .confirm-btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Bonus Management</h2>
                    <p>Search users and credit bonuses</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Search Container -->
            <div class="search-container">
                <div class="search-box">
                    <i class="fas fa-search search-icon"></i>
                    <input type="text" 
                           class="search-input" 
                           id="searchInput" 
                           placeholder="Search users by name, email, or phone number..."
                           autocomplete="off">
                </div>
                <p class="search-hint">Type at least 3 characters to search</p>
            </div>
            
            <!-- Results Container -->
            <div class="results-container" id="resultsContainer">
                <div class="results-header">
                    <h3>Search Results</h3>
                    <span class="results-count" id="resultsCount">0 users found</span>
                </div>
                <div class="user-list" id="userList">
                    <!-- Users will be loaded here via AJAX -->
                </div>
            </div>
        </div>
    </div>
    
    <!-- Credit User Modal -->
    <div class="modal-overlay" id="creditModal">
        <div class="credit-modal">
            <h3>Credit User</h3>
            <div class="user-display">
                <h4 id="selectedUserName"></h4>
                <p id="selectedUserEmail"></p>
            </div>
            <form id="creditForm">
                <input type="hidden" id="selectedUserId" name="user_id">
                
                <div class="form-group">
                    <label for="creditAmount" class="required">Amount to Credit</label>
                    <input type="text" 
                           class="form-control" 
                           id="creditAmount" 
                           name="amount" 
                           placeholder="Enter amount (e.g., 10,000)"
                           required>
                    <p class="form-hint">Maximum amount: ₦1,000,000. Use commas for thousands (e.g., 10,000)</p>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelCreditBtn">Cancel</button>
                    <button type="submit" class="form-btn success" id="submitCreditBtn">
                        <span id="submitBtnText">Credit User</span>
                        <div class="spinner" style="display: none;"></div>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Confirmation Modal -->
    <div class="modal-overlay" id="confirmModal">
        <div class="confirm-modal">
            <div class="confirm-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle">Confirm Credit</h3>
            <p id="confirmMessage">Are you sure you want to credit this amount to the user?</p>
            <div class="confirm-actions">
                <button class="confirm-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button class="confirm-btn primary" id="confirmActionBtn">Yes, Credit User</button>
            </div>
        </div>
    </div>
    
    <!-- Success/Error Message Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="message-modal">
            <div class="message-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="message-actions">
                <button class="message-btn" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    <script>
    // Global variables
    let selectedUser = null;
    let confirmCallback = null;
    let searchTimer = null;
    
    // Update current time
    function updateCurrentTime() {
        const now = new Date();
        const options = { 
            weekday: 'long', 
            year: 'numeric', 
            month: 'long', 
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit',
            second: '2-digit'
        };
        document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
    }
    
    setInterval(updateCurrentTime, 1000);
    updateCurrentTime();
    
    // Show message modal
    function showMessage(title, message, type = 'success') {
        const modal = document.getElementById('messageModal');
        const icon = document.getElementById('messageIcon');
        const titleEl = document.getElementById('messageTitle');
        const messageEl = document.getElementById('messageText');
        
        titleEl.textContent = title;
        messageEl.textContent = message;
        
        // Set icon based on type
        icon.className = 'message-icon ' + type;
        if (type === 'success') {
            icon.innerHTML = '<i class="fas fa-check-circle"></i>';
        } else if (type === 'error') {
            icon.innerHTML = '<i class="fas fa-times-circle"></i>';
        }
        
        modal.classList.add('active');
    }
    
    // Show confirmation modal
    function showConfirm(title, message, callback) {
        document.getElementById('confirmTitle').textContent = title;
        document.getElementById('confirmMessage').textContent = message;
        confirmCallback = callback;
        document.getElementById('confirmModal').classList.add('active');
    }
    
    // Format amount with commas
    function formatAmount(amount) {
        return amount.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
    }
    
    // Parse amount from formatted string
    function parseAmount(formattedAmount) {
        // Remove all commas and parse as float
        const numericString = formattedAmount.replace(/,/g, '');
        return parseFloat(numericString) || 0;
    }
    
    // Validate amount input
    function validateAmount(input) {
        let value = input.value.replace(/,/g, '');
        value = value.replace(/[^\d.]/g, '');
        
        // Allow only numbers and one decimal point
        if (value.split('.').length > 2) {
            value = value.substring(0, value.lastIndexOf('.'));
        }
        
        // Format with commas
        if (value) {
            const parts = value.split('.');
            parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, ",");
            input.value = parts.join('.');
        } else {
            input.value = '';
        }
        
        // Validate maximum amount
        const numericValue = parseFloat(value) || 0;
        if (numericValue > 1000000) {
            input.value = '1,000,000';
            showMessage('Warning', 'Maximum amount is ₦1,000,000', 'error');
            return false;
        }
        
        return true;
    }
    
    // Search users via AJAX
    async function searchUsers(query) {
        if (query.length < 3) {
            document.getElementById('resultsContainer').classList.remove('active');
            return;
        }
        
        try {
            const response = await fetch('b_bonus.php?action=search_users&query=' + encodeURIComponent(query));
            const data = await response.json();
            
            const resultsContainer = document.getElementById('resultsContainer');
            const userList = document.getElementById('userList');
            const resultsCount = document.getElementById('resultsCount');
            
            if (data.success && data.users.length > 0) {
                resultsContainer.classList.add('active');
                resultsCount.textContent = data.users.length + ' user' + (data.users.length !== 1 ? 's' : '') + ' found';
                
                let html = '';
                data.users.forEach(user => {
                    const initials = (user.first_name.charAt(0) + user.last_name.charAt(0)).toUpperCase();
                    const fullName = user.first_name + ' ' + user.last_name;
                    
                    html += `
                        <div class="user-card" id="user-${user.user_id}">
                            <div class="user-info">
                                <div class="user-avatar">
                                    ${user.profile_url ? 
                                        `<img src="${user.profile_url}" alt="${fullName}">` : 
                                        initials}
                                </div>
                                <div class="user-details">
                                    <h4>${fullName}</h4>
                                    <p>${user.email}</p>
                                    <p>${user.phone_number}</p>
                                    <div class="user-stats">
                                        <div class="user-stat">
                                            <label>Status</label>
                                            <span>${user.account_status}</span>
                                        </div>
                                        <div class="user-stat">
                                            <label>Verified</label>
                                            <span>${user.verification}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="user-actions">
                                <button class="credit-user-btn" onclick="openCreditModal(${user.user_id}, '${user.first_name}', '${user.email}')">
                                    <i class="fas fa-gift"></i> Credit User
                                </button>
                            </div>
                        </div>
                    `;
                });
                
                userList.innerHTML = html;
            } else {
                resultsContainer.classList.add('active');
                resultsCount.textContent = '0 users found';
                userList.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-search"></i>
                        <h3>No Users Found</h3>
                        <p>No users match your search criteria</p>
                    </div>
                `;
            }
        } catch (error) {
            console.error('Error searching users:', error);
            showMessage('Error', 'Error searching users', 'error');
        }
    }
    
    // Open credit modal
    function openCreditModal(userId, firstName, email) {
        selectedUser = { userId, firstName, email };
        
        document.getElementById('selectedUserId').value = userId;
        document.getElementById('selectedUserName').textContent = firstName;
        document.getElementById('selectedUserEmail').textContent = email;
        document.getElementById('creditAmount').value = '';
        
        document.getElementById('creditModal').classList.add('active');
    }
    
    // Search input handler with debounce
    document.getElementById('searchInput').addEventListener('input', function(e) {
        const query = e.target.value.trim();
        
        clearTimeout(searchTimer);
        
        if (query.length >= 3) {
            searchTimer = setTimeout(() => {
                searchUsers(query);
            }, 500);
        } else {
            document.getElementById('resultsContainer').classList.remove('active');
        }
    });
    
    // Amount input formatting
    document.getElementById('creditAmount').addEventListener('input', function(e) {
        validateAmount(this);
    });
    
    // Credit form submission
    document.getElementById('creditForm').addEventListener('submit', async function(e) {
        e.preventDefault();
        
        const amountInput = document.getElementById('creditAmount');
        const amount = parseAmount(amountInput.value);
        
        if (amount <= 0) {
            showMessage('Error', 'Please enter a valid amount', 'error');
            return;
        }
        
        if (amount > 1000000) {
            showMessage('Error', 'Maximum amount is ₦1,000,000', 'error');
            return;
        }
        
        // Show confirmation modal
        showConfirm(
            'Confirm Credit',
            `Are you sure you want to credit ₦${formatAmount(amount)} to ${selectedUser.firstName}?`,
            async () => {
                await submitCreditRequest(amount);
            }
        );
    });
    
    // Submit credit request
    async function submitCreditRequest(amount) {
        const submitBtn = document.getElementById('submitCreditBtn');
        const submitBtnText = document.getElementById('submitBtnText');
        const spinner = submitBtn.querySelector('.spinner');
        
        try {
            // Show loading state
            submitBtn.disabled = true;
            submitBtnText.textContent = 'Processing...';
            spinner.style.display = 'block';
            
            const response = await fetch('b_bonus.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'credit_user',
                    user_id: selectedUser.userId,
                    amount: amount
                })
            });
            
            const result = await response.json();
            
            if (result.success) {
                // Close modals
                document.getElementById('creditModal').classList.remove('active');
                document.getElementById('confirmModal').classList.remove('active');
                
                // Show success message
                showMessage('Success', result.message, 'success');
                
                // Clear search results
                document.getElementById('searchInput').value = '';
                document.getElementById('resultsContainer').classList.remove('active');
            } else {
                showMessage('Error', result.message || 'Failed to credit user', 'error');
            }
        } catch (error) {
            console.error('Error crediting user:', error);
            showMessage('Error', 'Error processing credit request', 'error');
        } finally {
            // Reset button state
            submitBtn.disabled = false;
            submitBtnText.textContent = 'Credit User';
            spinner.style.display = 'none';
        }
    }
    
    // Modal close buttons
    document.getElementById('cancelCreditBtn').addEventListener('click', () => {
        document.getElementById('creditModal').classList.remove('active');
    });
    
    document.getElementById('confirmCancelBtn').addEventListener('click', () => {
        document.getElementById('confirmModal').classList.remove('active');
        confirmCallback = null;
    });
    
    document.getElementById('confirmActionBtn').addEventListener('click', () => {
        document.getElementById('confirmModal').classList.remove('active');
        if (confirmCallback) {
            confirmCallback();
            confirmCallback = null;
        }
    });
    
    document.getElementById('messageOkBtn').addEventListener('click', () => {
        document.getElementById('messageModal').classList.remove('active');
    });
    
    // Close modals when clicking outside
    document.querySelectorAll('.modal-overlay').forEach(modal => {
        modal.addEventListener('click', (e) => {
            if (e.target === modal) {
                modal.classList.remove('active');
                if (modal.id === 'confirmModal') {
                    confirmCallback = null;
                }
            }
        });
    });
</script>
</body>
</html>