<?php
// device.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
    header('Location: auction.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle ban/unban action
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['device_id']) && isset($_POST['action'])) {
    try {
        $device_id = $_POST['device_id'];
        $action = $_POST['action'];
        
        if ($action === 'ban') {
            $stmt = $pdo->prepare("UPDATE device_link SET status = 'ban', updated_at = NOW() WHERE id = ?");
            $stmt->execute([$device_id]);
            $_SESSION['success_message'] = "Device banned successfully!";
        } elseif ($action === 'unban') {
            $stmt = $pdo->prepare("UPDATE device_link SET status = 'active', updated_at = NOW() WHERE id = ?");
            $stmt->execute([$device_id]);
            $_SESSION['success_message'] = "Device unbanned successfully!";
        }
        
        header('Location: device.php');
        exit;
        
    } catch (Exception $e) {
        $error = "Error processing request: " . $e->getMessage();
    }
}

// Get device statistics
try {
    // Get total devices count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM device_link");
    $stmt->execute();
    $totalDevices = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get active devices count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM device_link WHERE status = 'active'");
    $stmt->execute();
    $activeDevices = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get banned devices count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM device_link WHERE status = 'ban'");
    $stmt->execute();
    $bannedDevices = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get all devices with user information
    $stmt = $pdo->prepare("
        SELECT d.*, 
               u.user_id as user_id, 
               u.first_name, 
               u.last_name, 
               u.email,
               u.username
        FROM device_link d
        LEFT JOIN users u ON d.user_id = u.user_id
        ORDER BY d.created_at DESC
    ");
    $stmt->execute();
    $devices = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Error fetching data: " . $e->getMessage();
}

// Check for success message
$success_message = '';
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Device Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Stats Container */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 20px;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-icon {
            width: 70px;
            height: 70px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 30px;
        }
        
        .stat-icon.total {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .stat-icon.active {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .stat-icon.ban {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .stat-content {
            flex: 1;
        }
        
        .stat-title {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .stat-subtitle {
            font-size: 14px;
            color: var(--gray);
        }
        
        /* Devices Container */
        .devices-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .container-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .container-header h3 {
            font-size: 22px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .devices-count {
            background: var(--light);
            color: var(--dark);
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Devices Table */
        .devices-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .devices-table thead {
            background: var(--light);
        }
        
        .devices-table th {
            padding: 15px;
            text-align: left;
            font-weight: 600;
            color: var(--dark);
            border-bottom: 2px solid var(--gray-light);
        }
        
        .devices-table td {
            padding: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .devices-table tbody tr {
            transition: all 0.3s ease;
        }
        
        .devices-table tbody tr:hover {
            background: #f8fafc;
        }
        
        .device-info {
            display: flex;
            flex-direction: column;
            gap: 5px;
        }
        
        .device-name {
            font-weight: 600;
            color: var(--dark);
        }
        
        .device-id {
            font-size: 12px;
            color: var(--gray);
        }
        
        .user-info {
            display: flex;
            flex-direction: column;
            gap: 5px;
        }
        
        .user-name {
            font-weight: 600;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .user-email {
            font-size: 14px;
            color: var(--gray);
        }
        
        .user-id {
            font-size: 12px;
            color: var(--gray);
            background: var(--light);
            padding: 2px 8px;
            border-radius: 4px;
            display: inline-block;
        }
        
        .status-badge {
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            display: inline-block;
        }
        
        .status-active {
            background: #d1fae5;
            color: var(--secondary-dark);
        }
        
        .status-banned {
            background: #fee2e2;
            color: var(--danger);
        }
        
        .date-info {
            display: flex;
            flex-direction: column;
            gap: 3px;
        }
        
        .date-label {
            font-size: 12px;
            color: var(--gray);
        }
        
        .date-value {
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
        }
        
        .action-buttons {
            display: flex;
            gap: 8px;
        }
        
        .ban-btn {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 13px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
            white-space: nowrap;
        }
        
        .ban-btn:hover {
            background: linear-gradient(90deg, #dc2626, #b91c1c);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
        }
        
        .unban-btn {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 13px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
            white-space: nowrap;
        }
        
        .unban-btn:hover {
            background: linear-gradient(90deg, var(--secondary-dark), #047857);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(16, 185, 129, 0.3);
        }
        
        /* Action Forms */
        .action-form {
            display: inline;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
        }
        
        /* Success Message */
        .success-message {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
            animation: fadeIn 0.5s ease;
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Error Message */
        .error-message {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .devices-table {
                display: block;
                overflow-x: auto;
            }
            
            .devices-table th,
            .devices-table td {
                white-space: nowrap;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .devices-container {
                padding: 20px;
            }
            
            .container-header {
                flex-direction: column;
                gap: 10px;
                align-items: flex-start;
            }
            
            .devices-table {
                font-size: 14px;
            }
            
            .devices-table th,
            .devices-table td {
                padding: 10px;
            }
            
            .action-buttons {
                flex-direction: column;
                gap: 5px;
            }
            
            .ban-btn,
            .unban-btn {
                padding: 6px 12px;
                font-size: 12px;
            }
        }
        
        @media (max-width: 480px) {
            .stat-card {
                flex-direction: column;
                text-align: center;
                gap: 15px;
            }
            
            .user-info {
                font-size: 13px;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Device Management</h2>
                    <p>Monitor and manage user devices</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <?php if ($success_message): ?>
                <div class="success-message">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($error)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <!-- Stats Container -->
            <div class="stats-container">
                <div class="stat-card">
                    <div class="stat-icon total">
                        <i class="fas fa-server"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Total Devices</div>
                        <div class="stat-value"><?php echo $totalDevices['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">All registered devices</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon active">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Active Devices</div>
                        <div class="stat-value"><?php echo $activeDevices['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Currently active devices</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon ban">
                        <i class="fas fa-ban"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Banned Devices</div>
                        <div class="stat-value"><?php echo $bannedDevices['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Currently banned devices</div>
                    </div>
                </div>
            </div>
            
            <!-- Devices Container -->
            <div class="devices-container">
                <div class="container-header">
                    <h3>All Devices</h3>
                    <span class="devices-count"><?php echo count($devices ?? []); ?> devices</span>
                </div>
                
                <?php if (!empty($devices)): ?>
                    <table class="devices-table">
                        <thead>
                            <tr>
                                <th>Device Info</th>
                                <th>User Info</th>
                                <th>Status</th>
                                <th>Created</th>
                                <th>Updated</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($devices as $device): ?>
                                <tr>
                                    <td>
                                        <div class="device-info">
                                            <div class="device-name">
                                                <i class="fas fa-laptop"></i> <?php echo htmlspecialchars($device['device_name']); ?>
                                            </div>
                                            <div class="device-id">
                                                ID: <?php echo $device['id']; ?>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <?php if ($device['user_id']): ?>
                                            <div class="user-info">
                                                <div class="user-name">
                                                    <i class="fas fa-user"></i> 
                                                    <?php echo htmlspecialchars($device['first_name'] . ' ' . $device['last_name']); ?>
                                                </div>
                                                <div class="user-email">
                                                    <?php echo htmlspecialchars($device['email']); ?>
                                                </div>
                                                <div class="user-id">
                                                    User ID: <?php echo $device['user_id']; ?> | @<?php echo htmlspecialchars($device['username']); ?>
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <div class="user-info">
                                                <div class="user-name" style="color: var(--gray);">
                                                    <i class="fas fa-user-slash"></i> Not Linked
                                                </div>
                                            </div>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <?php if ($device['status'] === 'active'): ?>
                                            <span class="status-badge status-active">
                                                <i class="fas fa-check-circle"></i> Active
                                            </span>
                                        <?php else: ?>
                                            <span class="status-badge status-banned">
                                                <i class="fas fa-ban"></i> Banned
                                            </span>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <div class="date-info">
                                            <span class="date-label">Created</span>
                                            <span class="date-value"><?php echo date('M d, Y', strtotime($device['created_at'])); ?></span>
                                            <span class="date-label"><?php echo date('H:i', strtotime($device['created_at'])); ?></span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="date-info">
                                            <span class="date-label">Updated</span>
                                            <span class="date-value"><?php echo date('M d, Y', strtotime($device['updated_at'])); ?></span>
                                            <span class="date-label"><?php echo date('H:i', strtotime($device['updated_at'])); ?></span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="action-buttons">
                                            <?php if ($device['status'] === 'active'): ?>
                                                <form method="POST" class="action-form" onsubmit="return confirm('Are you sure you want to ban this device?')">
                                                    <input type="hidden" name="device_id" value="<?php echo $device['id']; ?>">
                                                    <input type="hidden" name="action" value="ban">
                                                    <button type="submit" class="ban-btn">
                                                        <i class="fas fa-ban"></i> Ban
                                                    </button>
                                                </form>
                                            <?php else: ?>
                                                <form method="POST" class="action-form" onsubmit="return confirm('Are you sure you want to unban this device?')">
                                                    <input type="hidden" name="device_id" value="<?php echo $device['id']; ?>">
                                                    <input type="hidden" name="action" value="unban">
                                                    <button type="submit" class="unban-btn">
                                                        <i class="fas fa-check-circle"></i> Unban
                                                    </button>
                                                </form>
                                            <?php endif; ?>
                                        </div>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-server"></i>
                        <h3>No Devices Found</h3>
                        <p>There are currently no registered devices</p>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script>
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
    </script>
</body>
</html>