<?php
// verify.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
    header('Location: auction.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Get verification stats
try {
    // Get total verified users count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM users WHERE verification = 'verified'");
    $stmt->execute();
    $verifiedUsers = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get pending verification count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM verification WHERE status = 'pending'");
    $stmt->execute();
    $pendingCount = $stmt->fetch(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Error fetching data: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verification Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 20px;
            transition: transform 0.3s ease;
            cursor: pointer;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-card.active {
            border: 2px solid var(--primary);
        }
        
        .stat-icon {
            width: 70px;
            height: 70px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 30px;
        }
        
        .stat-icon.verified {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .stat-icon.pending {
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .stat-content {
            flex: 1;
        }
        
        .stat-title {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .search-container {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .search-box {
            display: flex;
            gap: 15px;
        }
        
        .search-input {
            flex: 1;
            padding: 15px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        .search-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 0 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .search-btn:hover {
            background: var(--primary-dark);
        }
        
        .users-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .container-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .container-header h3 {
            font-size: 22px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .users-count {
            background: var(--light);
            color: var(--dark);
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        .users-list {
            display: flex;
            flex-direction: column;
            gap: 20px;
            min-height: 400px;
        }
        
        .user-card {
            background: var(--light);
            border-radius: 12px;
            padding: 25px;
            transition: all 0.3s ease;
            border-left: 4px solid var(--warning);
        }
        
        .user-card.verified-card {
            border-left-color: var(--success);
        }
        
        .user-card:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .user-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 20px;
        }
        
        .user-info {
            display: flex;
            align-items: flex-start;
            gap: 15px;
            flex: 1;
        }
        
        .user-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            overflow: hidden;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 20px;
            flex-shrink: 0;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .user-details {
            flex: 1;
        }
        
        .user-details h4 {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .user-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 8px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .user-details p i {
            width: 16px;
        }
        
        .user-meta {
            display: flex;
            gap: 20px;
            margin-top: 15px;
            flex-wrap: wrap;
        }
        
        .meta-item {
            display: flex;
            flex-direction: column;
            min-width: 120px;
        }
        
        .meta-label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .meta-value {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .documents-section {
            margin-top: 20px;
            padding-top: 20px;
            border-top: 1px solid var(--gray-light);
        }
        
        .documents-section h5 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .documents-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .document-item {
            background: white;
            border-radius: 10px;
            padding: 15px;
            border: 1px solid var(--gray-light);
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .document-item:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .document-preview {
            width: 100%;
            height: 150px;
            border-radius: 8px;
            overflow: hidden;
            background: var(--light);
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 10px;
        }
        
        .document-preview img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .document-preview.pdf {
            background: linear-gradient(135deg, #f56565, #e53e3e);
            color: white;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
        }
        
        .document-preview.pdf i {
            font-size: 48px;
            margin-bottom: 10px;
        }
        
        .document-info {
            text-align: center;
        }
        
        .document-type {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .document-action {
            font-size: 12px;
            color: var(--primary);
            font-weight: 500;
        }
        
        .user-actions {
            display: flex;
            gap: 10px;
            flex-shrink: 0;
        }
        
        .approve-btn {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .approve-btn:hover {
            background: linear-gradient(90deg, var(--secondary-dark), #047857);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .reject-btn {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .reject-btn:hover {
            background: linear-gradient(90deg, #dc2626, #b91c1c);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        .verified-badge {
            background: var(--success);
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: inline-flex;
            align-items: center;
            gap: 5px;
        }
        
        .review-date {
            background: var(--warning);
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .modal-content {
            background: white;
            border-radius: 15px;
            max-width: 800px;
            width: 100%;
            max-height: 90vh;
            overflow: hidden;
            position: relative;
            animation: modalSlideIn 0.3s ease;
        }
        
        .modal-content.small {
            max-width: 400px;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .modal-header {
            padding: 20px;
            border-bottom: 1px solid var(--gray-light);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h3 {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            color: var(--gray);
            cursor: pointer;
            padding: 5px;
            line-height: 1;
        }
        
        .modal-body {
            padding: 20px;
            max-height: calc(90vh - 140px);
            overflow-y: auto;
        }
        
        .pdf-viewer {
            width: 100%;
            height: 600px;
            border: none;
            border-radius: 8px;
        }
        
        .image-viewer {
            width: 100%;
            max-height: 600px;
            object-fit: contain;
            border-radius: 8px;
        }
        
        .rejection-form {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .rejection-form textarea {
            width: 100%;
            min-height: 150px;
            padding: 15px;
            border: 2px solid var(--gray-light);
            border-radius: 8px;
            font-family: inherit;
            font-size: 14px;
            resize: vertical;
            transition: border-color 0.3s ease;
        }
        
        .rejection-form textarea:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        .form-actions {
            display: flex;
            justify-content: flex-end;
            gap: 10px;
            margin-top: 20px;
        }
        
        .cancel-btn {
            background: var(--gray-light);
            color: var(--dark);
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .cancel-btn:hover {
            background: #d1d5db;
        }
        
        .confirm-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .confirm-btn:hover {
            background: var(--primary-dark);
        }
        
        .confirm-btn.danger {
            background: var(--danger);
        }
        
        .confirm-btn.danger:hover {
            background: #dc2626;
        }
        
        .message-modal-icon {
            font-size: 48px;
            text-align: center;
            margin-bottom: 20px;
        }
        
        .message-modal-icon.success {
            color: var(--success);
        }
        
        .message-modal-icon.error {
            color: var(--danger);
        }
        
        .message-modal-icon.warning {
            color: var(--warning);
        }
        
        .message-modal-text {
            text-align: center;
            font-size: 16px;
            margin-bottom: 20px;
            line-height: 1.6;
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
        }
        
        .loading-spinner {
            text-align: center;
            padding: 40px;
        }
        
        .loading-spinner i {
            font-size: 40px;
            color: var(--primary);
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .verified-date {
            background: var(--success);
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .search-box {
                flex-direction: column;
            }
            
            .user-header {
                flex-direction: column;
                gap: 15px;
            }
            
            .user-info {
                flex-direction: column;
                align-items: flex-start;
            }
            
            .user-actions {
                width: 100%;
                justify-content: flex-start;
            }
            
            .documents-grid {
                grid-template-columns: 1fr;
            }
            
            .users-container {
                padding: 20px;
            }
        }
        
        @media (max-width: 480px) {
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .user-meta {
                flex-direction: column;
                gap: 10px;
            }
            
            .user-actions {
                flex-direction: column;
                width: 100%;
            }
            
            .approve-btn, .reject-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Verification Management</h2>
                    <p>Monitor and manage user verifications</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Stats Container -->
            <div class="stats-container">
                <div class="stat-card <?php echo (!isset($_GET['tab']) || $_GET['tab'] == 'pending') ? 'active' : ''; ?>" id="pendingTab">
                    <div class="stat-icon pending">
                        <i class="fas fa-hourglass-half"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Pending Verification</div>
                        <div class="stat-value" id="pendingCount"><?php echo $pendingCount['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Awaiting review</div>
                    </div>
                </div>
                
                <div class="stat-card <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'verified') ? 'active' : ''; ?>" id="verifiedTab">
                    <div class="stat-icon verified">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Verified Users</div>
                        <div class="stat-value" id="verifiedCount"><?php echo $verifiedUsers['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Total verified accounts</div>
                    </div>
                </div>
            </div>
            
            <!-- Search Container -->
            <div class="search-container">
                <div class="search-box">
                    <input type="text" class="search-input" id="searchInput" placeholder="Search by email or first name..." value="">
                    <button class="search-btn" onclick="searchUsers()">
                        <i class="fas fa-search"></i> Search
                    </button>
                </div>
            </div>
            
            <!-- Users Container -->
            <div class="users-container">
                <div class="container-header">
                    <h3 id="usersTitle">Pending Verification</h3>
                    <span class="users-count" id="usersCount">0 users</span>
                </div>
                
                <div class="users-list" id="usersList">
                    <div class="loading-spinner">
                        <i class="fas fa-spinner fa-pulse"></i>
                        <p>Loading users...</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Document Preview Modal -->
    <div id="documentModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitle">Document Preview</h3>
                <button class="close-modal" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body">
                <iframe id="pdfViewer" class="pdf-viewer" style="display: none;"></iframe>
                <img id="imageViewer" class="image-viewer" style="display: none;">
            </div>
        </div>
    </div>
    
    <!-- Rejection Modal -->
    <div id="rejectionModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>Reject Verification</h3>
                <button class="close-modal" onclick="closeRejectionModal()">&times;</button>
            </div>
            <div class="modal-body">
                <form id="rejectionForm" class="rejection-form">
                    <input type="hidden" name="user_id" id="rejectUserId">
                    <input type="hidden" name="verification_id" id="rejectVerificationId">
                    <div id="rejectUserName" style="margin-bottom: 15px; font-size: 16px; font-weight: 600;"></div>
                    <div>
                        <label for="rejection_reason" style="display: block; margin-bottom: 8px; font-weight: 500;">Reason for rejection:</label>
                        <textarea name="rejection_reason" id="rejection_reason" required placeholder="Please provide a reason for rejecting this verification..."></textarea>
                    </div>
                    <div class="form-actions">
                        <button type="button" class="cancel-btn" onclick="closeRejectionModal()">Cancel</button>
                        <button type="button" class="reject-btn" onclick="submitRejection()">Reject Verification</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Message Modal -->
    <div id="messageModal" class="modal">
        <div class="modal-content small">
            <div class="modal-header">
                <h3 id="messageModalTitle">Message</h3>
                <button class="close-modal" onclick="closeMessageModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div id="messageModalIcon" class="message-modal-icon"></div>
                <div id="messageModalText" class="message-modal-text"></div>
                <div style="text-align: center;">
                    <button class="confirm-btn" onclick="closeMessageModal()">OK</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Confirm Modal -->
    <div id="confirmModal" class="modal">
        <div class="modal-content small">
            <div class="modal-header">
                <h3 id="confirmModalTitle">Confirm Action</h3>
                <button class="close-modal" onclick="closeConfirmModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div id="confirmModalIcon" class="message-modal-icon warning">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div id="confirmModalText" class="message-modal-text"></div>
                <div style="display: flex; justify-content: center; gap: 10px;">
                    <button class="cancel-btn" onclick="closeConfirmModal()">Cancel</button>
                    <button class="confirm-btn" id="confirmActionBtn">Confirm</button>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Current state
        let currentTab = 'pending';
        let currentSearch = '';
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Load users on page load
        document.addEventListener('DOMContentLoaded', function() {
            loadUsers('pending', '');
            
            // Add click events to tabs
            document.getElementById('pendingTab').addEventListener('click', function() {
                setActiveTab('pending');
                loadUsers('pending', currentSearch);
            });
            
            document.getElementById('verifiedTab').addEventListener('click', function() {
                setActiveTab('verified');
                loadUsers('verified', currentSearch);
            });
        });
        
        // Set active tab
        function setActiveTab(tab) {
            currentTab = tab;
            const pendingTab = document.getElementById('pendingTab');
            const verifiedTab = document.getElementById('verifiedTab');
            
            if (tab === 'pending') {
                pendingTab.classList.add('active');
                verifiedTab.classList.remove('active');
                document.getElementById('usersTitle').textContent = 'Pending Verification';
            } else {
                verifiedTab.classList.add('active');
                pendingTab.classList.remove('active');
                document.getElementById('usersTitle').textContent = 'Verified Users';
            }
        }
        
        // Search users
        function searchUsers() {
            currentSearch = document.getElementById('searchInput').value.trim();
            loadUsers(currentTab, currentSearch);
        }
        
        // Load users via AJAX
        function loadUsers(status, search) {
            const usersList = document.getElementById('usersList');
            usersList.innerHTML = '<div class="loading-spinner"><i class="fas fa-spinner fa-pulse"></i><p>Loading users...</p></div>';
            
            const formData = new FormData();
            formData.append('action', 'get_users');
            formData.append('status', status);
            formData.append('search', search);
            
            fetch('b_kyc.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    displayUsers(data.users, status);
                    document.getElementById('usersCount').textContent = data.count + ' users';
                    
                    // Update counts
                    if (data.stats) {
                        document.getElementById('pendingCount').textContent = data.stats.pending;
                        document.getElementById('verifiedCount').textContent = data.stats.verified;
                    }
                } else {
                    showMessage('Error', data.message || 'Failed to load users', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showMessage('Error', 'An error occurred while loading users', 'error');
            });
        }
        
        // Display users
        function displayUsers(users, status) {
            const usersList = document.getElementById('usersList');
            
            if (!users || users.length === 0) {
                usersList.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-users-slash"></i>
                        <h3>No Users Found</h3>
                        <p>There are no ${status === 'pending' ? 'pending verifications' : 'verified users'} to display</p>
                    </div>
                `;
                return;
            }
            
            let html = '';
            
            users.forEach(user => {
                const isVerified = status === 'verified';
                const cardClass = isVerified ? 'verified-card' : '';
                
                html += `
                    <div class="user-card ${cardClass}">
                        <div class="user-header">
                            <div class="user-info">
                                <div class="user-avatar">
                                    ${user.profile_url ? 
                                        `<img src="${escapeHtml(user.profile_url)}" alt="${escapeHtml(user.first_name)}">` : 
                                        escapeHtml(user.first_name.charAt(0) + user.last_name.charAt(0))
                                    }
                                </div>
                                <div class="user-details">
                                    <h4>${escapeHtml(user.first_name)} ${escapeHtml(user.last_name)}</h4>
                                    <p>
                                        <i class="fas fa-envelope"></i>
                                        ${escapeHtml(user.email)}
                                    </p>
                                    <p>
                                        <i class="fas fa-phone"></i>
                                        ${escapeHtml(user.phone_number || 'N/A')}
                                    </p>
                                    <p>
                                        <i class="fas fa-globe"></i>
                                        ${escapeHtml(user.country || 'N/A')}
                                    </p>
                                    <div class="user-meta">
                                        <div class="meta-item">
                                            <span class="meta-label">Username</span>
                                            <span class="meta-value">${escapeHtml(user.username)}</span>
                                        </div>
                                        ${!isVerified ? `
                                            <div class="meta-item">
                                                <span class="meta-label">Verification ID</span>
                                                <span class="meta-value">#${user.verification_id}</span>
                                            </div>
                                            <div class="meta-item">
                                                <span class="meta-label">Document Type</span>
                                                <span class="meta-value">${escapeHtml(user.document_type || 'N/A')}</span>
                                            </div>
                                        ` : ''}
                                    </div>
                                </div>
                            </div>
                            ${!isVerified ? `
                                <div class="user-actions">
                                    <button class="approve-btn" onclick="confirmApprove(${user.user_id}, ${user.verification_id})">
                                        <i class="fas fa-check-circle"></i> Approve
                                    </button>
                                    <button class="reject-btn" onclick="openRejectionModal(${user.user_id}, ${user.verification_id}, '${escapeHtml(user.first_name)} ${escapeHtml(user.last_name)}')">
                                        <i class="fas fa-times-circle"></i> Reject
                                    </button>
                                </div>
                            ` : ''}
                        </div>
                        
                        <!-- Documents Section -->
                        ${(user.document_url || user.liveness_url) ? `
                            <div class="documents-section">
                                <h5><i class="fas fa-file-alt"></i> ${isVerified ? 'Verified Documents' : 'Verification Documents'}</h5>
                                <div class="documents-grid">
                                    ${user.document_url ? `
                                        ${createDocumentHtml(user.document_url, 'ID Document', isVerified)}
                                    ` : ''}
                                    ${user.liveness_url ? `
                                        ${createDocumentHtml(user.liveness_url, 'Liveness Check', isVerified)}
                                    ` : ''}
                                </div>
                            </div>
                        ` : ''}
                        
                        <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 15px;">
                            ${!isVerified ? `
                                <div class="review-date">
                                    <i class="fas fa-clock"></i> 
                                    ${formatDate(user.verification_date)}
                                </div>
                                <div style="font-size: 12px; color: var(--warning);">
                                    <i class="fas fa-exclamation-triangle"></i> Awaiting Review
                                </div>
                            ` : `
                                <div class="verified-date">
                                    <i class="fas fa-check-circle"></i> 
                                    Verified: ${formatDate(user.verified_at)}
                                </div>
                                <div style="font-size: 12px; color: var(--success);">
                                    <i class="fas fa-check"></i> Verified
                                </div>
                            `}
                        </div>
                    </div>
                `;
            });
            
            usersList.innerHTML = html;
        }
        
        // Create document HTML
        function createDocumentHtml(url, type, isVerified) {
            const isPdf = url.toLowerCase().endsWith('.pdf');
            
            return `
                <div class="document-item" onclick="openDocumentModal('${escapeHtml(url)}', '${isPdf ? 'pdf' : 'image'}', '${type}')">
                    <div class="document-preview ${isPdf ? 'pdf' : ''}">
                        ${isPdf ? 
                            '<i class="fas fa-file-pdf"></i>' : 
                            `<img src="${escapeHtml(url)}" alt="${type}">`
                        }
                    </div>
                    <div class="document-info">
                        <div class="document-type">${type}</div>
                        <div class="document-action">Click to view</div>
                    </div>
                </div>
            `;
        }
        
        // Escape HTML
        function escapeHtml(text) {
            if (!text) return '';
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        }
        
        // Format date
        function formatDate(dateString) {
            if (!dateString) return 'N/A';
            const date = new Date(dateString);
            return date.toLocaleDateString('en-US', { 
                month: 'short', 
                day: 'numeric', 
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
        }
        
        // Confirm approve
        function confirmApprove(userId, verificationId) {
            showConfirm(
                'Approve Verification',
                'Are you sure you want to approve this verification?',
                function() {
                    approveVerification(userId, verificationId);
                }
            );
        }
        
        // Approve verification
        function approveVerification(userId, verificationId) {
            const formData = new FormData();
            formData.append('action', 'approve');
            formData.append('user_id', userId);
            formData.append('verification_id', verificationId);
            
            fetch('b_kyc.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    loadUsers(currentTab, currentSearch);
                } else {
                    showMessage('Error', data.message || 'Failed to approve verification', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showMessage('Error', 'An error occurred while approving', 'error');
            });
        }
        
        // Open rejection modal
        function openRejectionModal(userId, verificationId, userName) {
            document.getElementById('rejectUserId').value = userId;
            document.getElementById('rejectVerificationId').value = verificationId;
            document.getElementById('rejectUserName').textContent = `Reject verification for: ${userName}`;
            document.getElementById('rejection_reason').value = '';
            document.getElementById('rejectionModal').style.display = 'flex';
        }
        
        // Close rejection modal
        function closeRejectionModal() {
            document.getElementById('rejectionModal').style.display = 'none';
        }
        
        // Submit rejection
        function submitRejection() {
            const reason = document.getElementById('rejection_reason').value.trim();
            
            if (!reason) {
                showMessage('Error', 'Please provide a reason for rejection', 'error');
                return;
            }
            
            const userId = document.getElementById('rejectUserId').value;
            const verificationId = document.getElementById('rejectVerificationId').value;
            
            const formData = new FormData();
            formData.append('action', 'reject');
            formData.append('user_id', userId);
            formData.append('verification_id', verificationId);
            formData.append('rejection_reason', reason);
            
            fetch('b_kyc.php', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                closeRejectionModal();
                
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    loadUsers(currentTab, currentSearch);
                } else {
                    showMessage('Error', data.message || 'Failed to reject verification', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                closeRejectionModal();
                showMessage('Error', 'An error occurred while rejecting', 'error');
            });
        }
        
        // Document Modal Functions
        function openDocumentModal(url, type, title) {
            const modal = document.getElementById('documentModal');
            const modalTitle = document.getElementById('modalTitle');
            const pdfViewer = document.getElementById('pdfViewer');
            const imageViewer = document.getElementById('imageViewer');
            
            modalTitle.textContent = title;
            
            if (type === 'pdf') {
                pdfViewer.src = url;
                pdfViewer.style.display = 'block';
                imageViewer.style.display = 'none';
            } else {
                imageViewer.src = url;
                imageViewer.style.display = 'block';
                pdfViewer.style.display = 'none';
            }
            
            modal.style.display = 'flex';
        }
        
        function closeModal() {
            document.getElementById('documentModal').style.display = 'none';
            document.getElementById('pdfViewer').src = '';
            document.getElementById('imageViewer').src = '';
        }
        
        // Message Modal Functions
        function showMessage(title, message, type = 'info') {
            const modal = document.getElementById('messageModal');
            const modalTitle = document.getElementById('messageModalTitle');
            const modalIcon = document.getElementById('messageModalIcon');
            const modalText = document.getElementById('messageModalText');
            
            modalTitle.textContent = title;
            modalText.textContent = message;
            
            let iconHtml = '';
            if (type === 'success') {
                iconHtml = '<i class="fas fa-check-circle" style="color: var(--success);"></i>';
            } else if (type === 'error') {
                iconHtml = '<i class="fas fa-times-circle" style="color: var(--danger);"></i>';
            } else {
                iconHtml = '<i class="fas fa-info-circle" style="color: var(--primary);"></i>';
            }
            
            modalIcon.innerHTML = iconHtml;
            modal.style.display = 'flex';
        }
        
        function closeMessageModal() {
            document.getElementById('messageModal').style.display = 'none';
        }
        
        // Confirm Modal Functions
        function showConfirm(title, message, onConfirm) {
            const modal = document.getElementById('confirmModal');
            const modalTitle = document.getElementById('confirmModalTitle');
            const modalText = document.getElementById('confirmModalText');
            const confirmBtn = document.getElementById('confirmActionBtn');
            
            modalTitle.textContent = title;
            modalText.textContent = message;
            
            // Remove previous event listener
            const newConfirmBtn = confirmBtn.cloneNode(true);
            confirmBtn.parentNode.replaceChild(newConfirmBtn, confirmBtn);
            
            // Add new event listener
            newConfirmBtn.addEventListener('click', function() {
                closeConfirmModal();
                onConfirm();
            });
            
            modal.style.display = 'flex';
        }
        
        function closeConfirmModal() {
            document.getElementById('confirmModal').style.display = 'none';
        }
        
        // Close modals when clicking outside
        window.onclick = function(event) {
            const documentModal = document.getElementById('documentModal');
            const rejectionModal = document.getElementById('rejectionModal');
            const messageModal = document.getElementById('messageModal');
            const confirmModal = document.getElementById('confirmModal');
            
            if (event.target === documentModal) {
                closeModal();
            }
            if (event.target === rejectionModal) {
                closeRejectionModal();
            }
            if (event.target === messageModal) {
                closeMessageModal();
            }
            if (event.target === confirmModal) {
                closeConfirmModal();
            }
        }
        
        // Close modals with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeModal();
                closeRejectionModal();
                closeMessageModal();
                closeConfirmModal();
            }
        });
    </script>
</body>
</html>