<?php
// link.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
    header('Location: auction.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle add primary account
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        if (isset($_POST['add_primary']) && isset($_POST['email'])) {
            $email = trim($_POST['email']);
            
            // Check if user exists
            $stmt = $pdo->prepare("SELECT user_id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                // Check if primary account already exists for this user
                $stmt = $pdo->prepare("SELECT id FROM primary_account WHERE user_id = ?");
                $stmt->execute([$user['user_id']]);
                $existing = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$existing) {
                    // Generate unique account code
                    $account_code = 'PRIM-' . strtoupper(uniqid());
                    
                    // Insert primary account
                    $stmt = $pdo->prepare("INSERT INTO primary_account (user_id, account_code) VALUES (?, ?)");
                    $stmt->execute([$user['user_id'], $account_code]);
                    
                    $_SESSION['success_message'] = "Primary account created successfully! Account Code: " . $account_code;
                } else {
                    $_SESSION['error_message'] = "This user already has a primary account!";
                }
            } else {
                $_SESSION['error_message'] = "User with this email not found!";
            }
        }
        
        // Handle link account
        elseif (isset($_POST['link_account']) && isset($_POST['primary_id']) && isset($_POST['email'])) {
            $primary_id = $_POST['primary_id'];
            $email = trim($_POST['email']);
            
            // Check if user exists
            $stmt = $pdo->prepare("SELECT user_id FROM users WHERE email = ?");
            $stmt->execute([$email]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($user) {
                // Check if account is already linked
                $stmt = $pdo->prepare("SELECT id FROM link_accounts WHERE primary_id = ? AND user_id = ?");
                $stmt->execute([$primary_id, $user['user_id']]);
                $existing = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if (!$existing) {
                    // Generate link code
                    $link_code = 'LINK-' . strtoupper(uniqid());
                    
                    // Insert linked account
                    $stmt = $pdo->prepare("INSERT INTO link_accounts (user_id, primary_id, link_code) VALUES (?, ?, ?)");
                    $stmt->execute([$user['user_id'], $primary_id, $link_code]);
                    
                    $_SESSION['success_message'] = "Account linked successfully! Link Code: " . $link_code;
                } else {
                    $_SESSION['error_message'] = "This account is already linked!";
                }
            } else {
                $_SESSION['error_message'] = "User with this email not found!";
            }
        }
        
        // Handle unlink account
        elseif (isset($_POST['unlink_account']) && isset($_POST['link_id'])) {
            $link_id = $_POST['link_id'];
            
            $stmt = $pdo->prepare("DELETE FROM link_accounts WHERE id = ?");
            $stmt->execute([$link_id]);
            
            $_SESSION['success_message'] = "Account unlinked successfully!";
        }
        
        header('Location: links.php');
        exit;
        
    } catch (Exception $e) {
        $_SESSION['error_message'] = "Error processing request: " . $e->getMessage();
        header('Location: link.php');
        exit;
    }
}

// Get statistics and data
try {
    // Get primary accounts count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM primary_account");
    $stmt->execute();
    $primaryCount = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get linked accounts count
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM link_accounts");
    $stmt->execute();
    $linkedCount = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get all primary accounts with user details
    $stmt = $pdo->prepare("
        SELECT p.*, 
               u.user_id, 
               u.username, 
               u.first_name, 
               u.last_name, 
               u.email,
               u.profile_url,
               COUNT(l.id) as linked_count
        FROM primary_account p
        INNER JOIN users u ON p.user_id = u.user_id
        LEFT JOIN link_accounts l ON p.id = l.primary_id
        GROUP BY p.id
        ORDER BY p.created_at DESC
    ");
    $stmt->execute();
    $primaryAccounts = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Error fetching data: " . $e->getMessage();
}

// Check for messages
$success_message = '';
$error_message = '';
if (isset($_SESSION['success_message'])) {
    $success_message = $_SESSION['success_message'];
    unset($_SESSION['success_message']);
}
if (isset($_SESSION['error_message'])) {
    $error_message = $_SESSION['error_message'];
    unset($_SESSION['error_message']);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Account Linking - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Stats Container */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 20px;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-icon {
            width: 70px;
            height: 70px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 30px;
        }
        
        .stat-icon.primary {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .stat-icon.linked {
            background: linear-gradient(135deg, var(--secondary), var(--secondary-dark));
            color: white;
        }
        
        .stat-content {
            flex: 1;
        }
        
        .stat-title {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .stat-subtitle {
            font-size: 14px;
            color: var(--gray);
        }
        
        /* Add Primary Account Form */
        .add-primary-form {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            margin-bottom: 30px;
        }
        
        .form-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .form-row {
            display: flex;
            gap: 15px;
            align-items: flex-end;
        }
        
        .form-group {
            flex: 1;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-group input {
            width: 100%;
            padding: 12px 15px;
            border: 2px solid var(--gray-light);
            border-radius: 8px;
            font-size: 14px;
            transition: all 0.3s ease;
        }
        
        .form-group input:focus {
            outline: none;
            border-color: var(--primary);
        }
        
        .submit-btn {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .submit-btn:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Primary Accounts Container */
        .primary-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .container-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .container-header h3 {
            font-size: 22px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .accounts-count {
            background: var(--light);
            color: var(--dark);
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Primary Accounts List */
        .accounts-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .account-card {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary);
            cursor: pointer;
        }
        
        .account-card:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .account-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .account-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .account-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            overflow: hidden;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
        }
        
        .account-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .account-details {
            flex: 1;
        }
        
        .account-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .account-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 3px;
        }
        
        .account-meta {
            display: flex;
            gap: 20px;
            margin-top: 10px;
        }
        
        .meta-item {
            display: flex;
            flex-direction: column;
        }
        
        .meta-label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .meta-value {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .linked-badge {
            background: var(--secondary);
            color: white;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        /* Modals */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: rgba(0, 0, 0, 0.8);
            z-index: 1000;
            align-items: center;
            justify-content: center;
            padding: 20px;
        }
        
        .modal-content {
            background: white;
            border-radius: 15px;
            max-width: 800px;
            width: 100%;
            max-height: 90vh;
            overflow: hidden;
            position: relative;
            animation: modalSlideIn 0.3s ease;
        }
        
        @keyframes modalSlideIn {
            from {
                opacity: 0;
                transform: translateY(-50px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        .modal-header {
            padding: 20px;
            border-bottom: 1px solid var(--gray-light);
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .modal-header h3 {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .close-modal {
            background: none;
            border: none;
            font-size: 24px;
            color: var(--gray);
            cursor: pointer;
            padding: 5px;
            line-height: 1;
        }
        
        .modal-body {
            padding: 20px;
            max-height: calc(90vh - 140px);
            overflow-y: auto;
        }
        
        /* Linked Accounts List */
        .linked-accounts-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .linked-account-card {
            background: white;
            border: 1px solid var(--gray-light);
            border-radius: 10px;
            padding: 15px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .linked-account-card:hover {
            background: var(--light);
            transform: translateY(-2px);
        }
        
        .linked-account-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .linked-account-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            overflow: hidden;
            background: linear-gradient(135deg, var(--secondary), var(--secondary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 14px;
            flex-shrink: 0;
        }
        
        .linked-account-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .linked-account-details {
            flex: 1;
        }
        
        .linked-account-details h5 {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 3px;
        }
        
        .linked-account-details p {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 2px;
        }
        
        .linked-account-code {
            font-size: 11px;
            color: var(--primary);
            background: var(--light);
            padding: 2px 8px;
            border-radius: 4px;
        }
        
        .unlink-btn {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
            white-space: nowrap;
        }
        
        .unlink-btn:hover {
            background: linear-gradient(90deg, #dc2626, #b91c1c);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(239, 68, 68, 0.3);
        }
        
        .add-link-form {
            background: var(--light);
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
        }
        
        .add-link-form h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        /* Success Message */
        .success-message {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
            animation: fadeIn 0.5s ease;
        }
        
        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Error Message */
        .error-message {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--gray-light);
        }
        
        .empty-state h4 {
            font-size: 18px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 14px;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .form-row {
                flex-direction: column;
                gap: 15px;
            }
            
            .submit-btn {
                width: 100%;
                justify-content: center;
            }
            
            .primary-container {
                padding: 20px;
            }
            
            .container-header {
                flex-direction: column;
                gap: 10px;
                align-items: flex-start;
            }
            
            .account-header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .account-info {
                width: 100%;
            }
            
            .linked-account-card {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .unlink-btn {
                width: 100%;
                justify-content: center;
            }
        }
        
        @media (max-width: 480px) {
            .stat-card {
                flex-direction: column;
                text-align: center;
                gap: 15px;
            }
            
            .account-meta {
                flex-direction: column;
                gap: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Account Linking</h2>
                    <p>Manage primary accounts and linked accounts</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <?php if ($success_message): ?>
                <div class="success-message">
                    <i class="fas fa-check-circle"></i> <?php echo htmlspecialchars($success_message); ?>
                </div>
            <?php endif; ?>
            
            <?php if ($error_message): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error_message); ?>
                </div>
            <?php endif; ?>
            
            <?php if (isset($error)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-circle"></i> <?php echo htmlspecialchars($error); ?>
                </div>
            <?php endif; ?>
            
            <!-- Stats Container -->
            <div class="stats-container">
                <div class="stat-card">
                    <div class="stat-icon primary">
                        <i class="fas fa-user-shield"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Primary Accounts</div>
                        <div class="stat-value"><?php echo $primaryCount['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Total primary accounts</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon linked">
                        <i class="fas fa-link"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Linked Accounts</div>
                        <div class="stat-value"><?php echo $linkedCount['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Total linked accounts</div>
                    </div>
                </div>
            </div>
            
            <!-- Add Primary Account Form -->
            <div class="add-primary-form">
                <div class="form-title">
                    <i class="fas fa-plus-circle"></i> Add Primary Account
                </div>
                <form method="POST">
                    <div class="form-row">
                        <div class="form-group">
                            <label for="email">User Email</label>
                            <input type="email" name="email" id="email" placeholder="Enter user email address" required>
                        </div>
                        <button type="submit" name="add_primary" class="submit-btn">
                            <i class="fas fa-user-plus"></i> Create Primary Account
                        </button>
                    </div>
                </form>
            </div>
            
            <!-- Primary Accounts Container -->
            <div class="primary-container">
                <div class="container-header">
                    <h3>Primary Accounts</h3>
                    <span class="accounts-count"><?php echo count($primaryAccounts ?? []); ?> accounts</span>
                </div>
                
                <div class="accounts-list">
                    <?php if (!empty($primaryAccounts)): ?>
                        <?php foreach ($primaryAccounts as $account): ?>
                            <div class="account-card" onclick="openLinkedAccountsModal(<?php echo $account['id']; ?>)">
                                <div class="account-header">
                                    <div class="account-info">
                                        <div class="account-avatar">
                                            <?php if (!empty($account['profile_url'])): ?>
                                                <img src="<?php echo htmlspecialchars($account['profile_url']); ?>" 
                                                     alt="<?php echo htmlspecialchars($account['first_name'] . ' ' . $account['last_name']); ?>">
                                            <?php else: ?>
                                                <?php echo strtoupper(substr($account['first_name'], 0, 1) . substr($account['last_name'], 0, 1)); ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="account-details">
                                            <h4>
                                                <?php echo htmlspecialchars($account['first_name'] . ' ' . $account['last_name']); ?>
                                                <span class="linked-badge">
                                                    <i class="fas fa-link"></i> <?php echo $account['linked_count']; ?> linked
                                                </span>
                                            </h4>
                                            <p>
                                                <i class="fas fa-envelope"></i> <?php echo htmlspecialchars($account['email']); ?>
                                            </p>
                                            <p>
                                                <i class="fas fa-user"></i> @<?php echo htmlspecialchars($account['username']); ?>
                                            </p>
                                            <div class="account-meta">
                                                <div class="meta-item">
                                                    <span class="meta-label">Account Code</span>
                                                    <span class="meta-value"><?php echo htmlspecialchars($account['account_code']); ?></span>
                                                </div>
                                                <div class="meta-item">
                                                    <span class="meta-label">User ID</span>
                                                    <span class="meta-value"><?php echo $account['user_id']; ?></span>
                                                </div>
                                                <div class="meta-item">
                                                    <span class="meta-label">Created</span>
                                                    <span class="meta-value"><?php echo date('M d, Y', strtotime($account['created_at'])); ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div style="color: var(--primary); font-size: 24px;">
                                        <i class="fas fa-chevron-right"></i>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-user-shield"></i>
                            <h4>No Primary Accounts</h4>
                            <p>Add your first primary account using the form above</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Linked Accounts Modal -->
    <div id="linkedAccountsModal" class="modal">
        <div class="modal-content">
            <div class="modal-header">
                <h3 id="modalTitle">Linked Accounts</h3>
                <button class="close-modal" onclick="closeModal()">&times;</button>
            </div>
            <div class="modal-body">
                <div id="linkedAccountsList" class="linked-accounts-list">
                    <!-- Linked accounts will be loaded here -->
                </div>
                
                <!-- Add Link Account Form -->
                <div class="add-link-form">
                    <h4><i class="fas fa-link"></i> Link New Account</h4>
                    <form id="linkAccountForm" method="POST">
                        <input type="hidden" name="primary_id" id="modalPrimaryId">
                        <div class="form-row">
                            <div class="form-group">
                                <label for="link_email">User Email</label>
                                <input type="email" name="email" id="link_email" placeholder="Enter user email to link" required>
                            </div>
                            <button type="submit" name="link_account" class="submit-btn">
                                <i class="fas fa-link"></i> Link Account
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Unlink Form (hidden) -->
    <form id="unlinkForm" method="POST" style="display: none;">
        <input type="hidden" name="link_id" id="unlinkId">
        <input type="hidden" name="unlink_account" value="1">
    </form>
    
    <script>
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Open linked accounts modal
        function openLinkedAccountsModal(primaryId) {
            const modal = document.getElementById('linkedAccountsModal');
            const modalTitle = document.getElementById('modalTitle');
            const linkedAccountsList = document.getElementById('linkedAccountsList');
            const modalPrimaryId = document.getElementById('modalPrimaryId');
            
            modalPrimaryId.value = primaryId;
            
            // Load linked accounts via AJAX
            fetch(`get_linked_accounts.php?primary_id=${primaryId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        modalTitle.textContent = `Linked Accounts - ${data.primary_name}`;
                        
                        if (data.accounts.length > 0) {
                            let html = '';
                            data.accounts.forEach(account => {
                                html += `
                                    <div class="linked-account-card">
                                        <div class="linked-account-info">
                                            <div class="linked-account-avatar">
                                                ${account.profile_url ? 
                                                    `<img src="${account.profile_url}" alt="${account.first_name} ${account.last_name}">` : 
                                                    `${account.first_name.charAt(0).toUpperCase()}${account.last_name.charAt(0).toUpperCase()}`
                                                }
                                            </div>
                                            <div class="linked-account-details">
                                                <h5>${account.first_name} ${account.last_name}</h5>
                                                <p><i class="fas fa-envelope"></i> ${account.email}</p>
                                                <p><i class="fas fa-user"></i> @${account.username}</p>
                                                <div class="linked-account-code">${account.link_code}</div>
                                            </div>
                                        </div>
                                        <button class="unlink-btn" onclick="unlinkAccount(${account.id})">
                                            <i class="fas fa-unlink"></i> Unlink
                                        </button>
                                    </div>
                                `;
                            });
                            linkedAccountsList.innerHTML = html;
                        } else {
                            linkedAccountsList.innerHTML = `
                                <div class="empty-state">
                                    <i class="fas fa-unlink"></i>
                                    <h4>No Linked Accounts</h4>
                                    <p>This primary account has no linked accounts yet</p>
                                </div>
                            `;
                        }
                    } else {
                        linkedAccountsList.innerHTML = `
                            <div class="error-message" style="margin-bottom: 20px;">
                                ${data.message}
                            </div>
                        `;
                    }
                })
                .catch(error => {
                    linkedAccountsList.innerHTML = `
                        <div class="error-message" style="margin-bottom: 20px;">
                            Error loading linked accounts
                        </div>
                    `;
                });
            
            modal.style.display = 'flex';
        }
        
        // Close modal
        function closeModal() {
            document.getElementById('linkedAccountsModal').style.display = 'none';
        }
        
        // Unlink account
        function unlinkAccount(linkId) {
            if (confirm('Are you sure you want to unlink this account?')) {
                document.getElementById('unlinkId').value = linkId;
                document.getElementById('unlinkForm').submit();
            }
        }
        
        // Close modal when clicking outside
        window.onclick = function(event) {
            const modal = document.getElementById('linkedAccountsModal');
            if (event.target === modal) {
                closeModal();
            }
        }
        
        // Close modal with Escape key
        document.addEventListener('keydown', function(event) {
            if (event.key === 'Escape') {
                closeModal();
            }
        });
    </script>
</body>
</html>