<?php
// order.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle form submission for deleting seller order
$message = '';
$message_type = ''; // success or error

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['delete_order_id'])) {
    $order_id = trim($_POST['delete_order_id']);
    
    if (empty($order_id)) {
        $message = 'Please enter an order ID';
        $message_type = 'error';
    } else {
        try {
            // Check if the seller order exists
            $checkStmt = $pdo->prepare("SELECT id FROM seller_orders WHERE id = ?");
            $checkStmt->execute([$order_id]);
            
            if ($checkStmt->rowCount() === 0) {
                $message = 'Seller order with ID ' . $order_id . ' does not exist';
                $message_type = 'error';
            } else {
                // Delete the seller order
                $deleteStmt = $pdo->prepare("DELETE FROM seller_orders WHERE id = ?");
                $deleteStmt->execute([$order_id]);
                
                if ($deleteStmt->rowCount() > 0) {
                    $message = 'Seller order #' . $order_id . ' deleted successfully';
                    $message_type = 'success';
                    
                    // Refresh the page after successful deletion
                    header("Refresh: 2");
                } else {
                    $message = 'Failed to delete seller order #' . $order_id;
                    $message_type = 'error';
                }
            }
        } catch (PDOException $e) {
            $message = 'Database error: ' . $e->getMessage();
            $message_type = 'error';
        }
    }
}

// Fetch buyer and seller orders with user information
try {
    // Get total pending buyer orders count and sum
    $buyerTotalStmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_count,
            COALESCE(SUM(amount), 0) as total_amount
        FROM buyer_orders 
        WHERE status = 'pending'
    ");
    $buyerTotalStmt->execute();
    $buyerTotals = $buyerTotalStmt->fetch(PDO::FETCH_ASSOC);
    $buyer_total_count = $buyerTotals['total_count'] ?? 0;
    $buyer_total_amount = $buyerTotals['total_amount'] ?? 0;
    
    // Get total pending seller orders count and sum
    $sellerTotalStmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_count,
            COALESCE(SUM(amount), 0) as total_amount
        FROM seller_orders 
        WHERE status = 'in progress'
    ");
    $sellerTotalStmt->execute();
    $sellerTotals = $sellerTotalStmt->fetch(PDO::FETCH_ASSOC);
    $seller_total_count = $sellerTotals['total_count'] ?? 0;
    $seller_total_amount = $sellerTotals['total_amount'] ?? 0;
    
    // Get pending buyer orders 
    $buyerStmt = $pdo->prepare("
        SELECT bo.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, 
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at as user_created_at
        FROM buyer_orders bo 
        JOIN users u ON bo.user_id = u.user_id 
        WHERE bo.status = 'pending'
        ORDER BY bo.created_at DESC 
        
    ");
    $buyerStmt->execute();
    $buyer_orders = $buyerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get pending seller orders 
    $sellerStmt = $pdo->prepare("
        SELECT so.*, u.username, u.first_name, u.last_name, u.email, u.phone_number,
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at as user_created_at
        FROM seller_orders so 
        JOIN users u ON so.user_id = u.user_id 
        WHERE so.status = 'in progress'
        ORDER BY so.created_at DESC 
       
    ");
    $sellerStmt->execute();
    $seller_orders = $sellerStmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Orders Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --danger-dark: #dc2626;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        /* Search Containers */
        .search-containers {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .search-container {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .search-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .search-header h3 {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .search-form {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .search-input-group {
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .search-input {
            flex: 1;
            padding: 12px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .search-btn {
            padding: 12px 30px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            width: 100%;
            justify-content: center;
        }
        
        .search-btn:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .search-btn.danger {
            background: linear-gradient(90deg, var(--danger), var(--danger-dark));
        }
        
        .search-btn.danger:hover {
            background: linear-gradient(90deg, var(--danger-dark), #b91c1c);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        .form-group {
            margin-bottom: 15px;
        }
        
        .form-group label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
        }
        
        .form-group select {
            width: 100%;
            padding: 12px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            background: white;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        /* Message Alert */
        .message-alert {
            background: white;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            border-left: 4px solid var(--primary);
            display: flex;
            align-items: center;
            gap: 15px;
            animation: slideIn 0.3s ease-out;
        }
        
        .message-alert.error {
            border-left-color: var(--danger);
        }
        
        .message-alert.success {
            border-left-color: var(--secondary);
        }
        
        .message-icon {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 18px;
            color: white;
            flex-shrink: 0;
        }
        
        .message-icon.error {
            background: linear-gradient(135deg, var(--danger), var(--danger-dark));
        }
        
        .message-icon.success {
            background: linear-gradient(135deg, var(--secondary), var(--secondary-dark));
        }
        
        .message-content {
            flex: 1;
        }
        
        .message-content h4 {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 5px;
        }
        
        .message-content p {
            font-size: 14px;
            color: var(--gray);
        }
        
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(-20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        
        /* Summary Cards */
        .summary-cards {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .summary-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .summary-card.buyer {
            border-top: 4px solid var(--primary);
        }
        
        .summary-card.seller {
            border-top: 4px solid var(--secondary);
        }
        
        .summary-header {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 20px;
        }
        
        .summary-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 24px;
            color: white;
        }
        
        .summary-icon.buyer {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
        }
        
        .summary-icon.seller {
            background: linear-gradient(135deg, var(--secondary), var(--secondary-dark));
        }
        
        .summary-header h3 {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .summary-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 20px;
        }
        
        .summary-item {
            display: flex;
            flex-direction: column;
        }
        
        .summary-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .summary-item .count {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .summary-item .amount {
            font-size: 28px;
            font-weight: 700;
            color: var(--primary);
        }
        
        /* Orders Container */
        .orders-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .orders-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
            padding-bottom: 10px;
        }
        
        .order-tab {
            padding: 12px 24px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .order-tab.active {
            background: var(--primary);
            color: white;
        }
        
        .order-tab:hover:not(.active) {
            background: #d1d5db;
        }
        
        .orders-content {
            min-height: 400px;
        }
        
        .orders-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .order-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
        }
        
        .order-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .order-id {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .order-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-pending {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-in_progress {
            background: #d1fae5;
            color: #065f46;
        }
        
        .order-user {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .user-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
            overflow: hidden;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .user-info h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .user-info p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .order-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .order-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .order-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .order-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .amount-value {
            color: var(--primary);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 14px;
        }
        
        /* Modals */
        .modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 1000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal.active {
            opacity: 1;
            visibility: visible;
        }
        
        .modal-content {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal.active .modal-content {
            transform: translateY(0);
        }
        
        .modal-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .modal-content h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .modal-form .form-group {
            margin-bottom: 20px;
        }
        
        .modal-form .form-group label {
            display: block;
            margin-bottom: 8px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
        }
        
        .modal-form .form-group input,
        .modal-form .form-group select {
            width: 100%;
            padding: 12px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .modal-form .form-group input:focus,
        .modal-form .form-group select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .modal-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .modal-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .modal-btn.primary:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .modal-btn.cancel {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .modal-btn.cancel:hover {
            background: #d1d5db;
        }
        
        /* AJAX Response Message */
        .ajax-message {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            border-radius: 10px;
            padding: 15px 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.1);
            display: flex;
            align-items: center;
            gap: 10px;
            z-index: 1001;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            border-left: 4px solid var(--primary);
        }
        
        .ajax-message.show {
            transform: translateX(0);
        }
        
        .ajax-message.success {
            border-left-color: var(--secondary);
        }
        
        .ajax-message.error {
            border-left-color: var(--danger);
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .search-containers {
                grid-template-columns: 1fr;
            }
            
            .summary-cards {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .orders-tabs {
                flex-wrap: wrap;
            }
            
            .order-details {
                grid-template-columns: 1fr;
            }
            
            .summary-details {
                grid-template-columns: 1fr;
                gap: 15px;
            }
            
            .modal-actions {
                flex-direction: column;
            }
            
            .modal-btn {
                width: 100%;
            }
        }
        
        @media (max-width: 480px) {
            .order-user {
                flex-direction: column;
                text-align: center;
                gap: 10px;
            }
            
            .order-header {
                flex-direction: column;
                gap: 10px;
                text-align: center;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Orders Management</h2>
                    <p>View and manage pending orders</p>
                </div>
            </div>
            
            <!-- Search Containers -->
            <div class="search-containers">
                <!-- Manage Order Container -->
                <div class="search-container">
                    <div class="search-header">
                        <h3>Manage Order</h3>
                    </div>
                    <form method="GET" class="search-form" id="manageOrderForm">
                        <div class="form-group">
                            <label for="order_type">Order Type</label>
                            <select id="order_type" name="order_type" class="search-input" required>
                                <option value="">Select Order Type</option>
                                <option value="buyer">Buyer Order</option>
                                <option value="seller">Seller Order</option>
                            </select>
                        </div>
                        <div class="search-input-group">
                            <input type="number" name="order_id" id="order_id" class="search-input" 
                                   placeholder="Enter Order ID" min="1" required>
                        </div>
                        <button type="submit" class="search-btn" id="manageOrderBtn">
                            <i class="fas fa-edit"></i> Manage Order
                        </button>
                    </form>
                </div>
                
                <!-- Delete Seller Order Container -->
                <div class="search-container">
                    <div class="search-header">
                        <h3>Delete Seller Order</h3>
                    </div>
                    <form method="POST" class="search-form" id="deleteForm">
                        <div class="search-input-group">
                            <input type="number" name="delete_order_id" class="search-input" 
                                   placeholder="Enter Seller Order ID to delete" min="1" required>
                        </div>
                        <button type="submit" class="search-btn danger">
                            <i class="fas fa-trash-alt"></i> Delete Order
                        </button>
                    </form>
                </div>
            </div>
            
            <!-- Message Alert -->
            <?php if ($message): ?>
                <div class="message-alert <?php echo $message_type; ?>">
                    <div class="message-icon <?php echo $message_type; ?>">
                        <i class="fas fa-<?php echo $message_type === 'success' ? 'check-circle' : 'exclamation-circle'; ?>"></i>
                    </div>
                    <div class="message-content">
                        <h4><?php echo $message_type === 'success' ? 'Success' : 'Error'; ?></h4>
                        <p><?php echo htmlspecialchars($message); ?></p>
                    </div>
                </div>
            <?php endif; ?>
            
            <!-- Summary Cards -->
            <div class="summary-cards">
                <div class="summary-card buyer">
                    <div class="summary-header">
                        <div class="summary-icon buyer">
                            <i class="fas fa-shopping-cart"></i>
                        </div>
                        <h3>Pending Buyer Orders</h3>
                    </div>
                    <div class="summary-details">
                        <div class="summary-item">
                            <label>Total Orders</label>
                            <div class="count"><?php echo number_format($buyer_total_count); ?></div>
                        </div>
                        <div class="summary-item">
                            <label>Total Amount</label>
                            <div class="amount">₦<?php echo number_format($buyer_total_amount, 2); ?></div>
                        </div>
                    </div>
                </div>
                
                <div class="summary-card seller">
                    <div class="summary-header">
                        <div class="summary-icon seller">
                            <i class="fas fa-store"></i>
                        </div>
                        <h3>Pending Seller Orders</h3>
                    </div>
                    <div class="summary-details">
                        <div class="summary-item">
                            <label>Total Orders</label>
                            <div class="count"><?php echo number_format($seller_total_count); ?></div>
                        </div>
                        <div class="summary-item">
                            <label>Total Amount</label>
                            <div class="amount">₦<?php echo number_format($seller_total_amount, 2); ?></div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Orders Container -->
            <div class="orders-container">
                <div class="orders-tabs">
                    <button class="order-tab active" id="buyerTab">
                        Buyer Orders (<?php echo count($buyer_orders); ?>)
                    </button>
                    <button class="order-tab" id="sellerTab">
                        Seller Orders (<?php echo count($seller_orders); ?>)
                    </button>
                </div>
                
                <div class="orders-content">
                    <!-- Buyer Orders -->
                    <div class="orders-list" id="buyerOrders">
                        <?php if (empty($buyer_orders)): ?>
                            <div class="empty-state">
                                <i class="fas fa-shopping-cart"></i>
                                <h3>No Pending Buyer Orders</h3>
                                <p>There are no pending buyer orders at the moment</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($buyer_orders as $order): ?>
                                <div class="order-item">
                                    <div class="order-header">
                                        <div class="order-id">Buy Order #<?php echo $order['id']; ?></div>
                                        <span class="order-status status-pending">Pending</span>
                                    </div>
                                    <div class="order-user">
                                        <div class="user-avatar">
                                            <?php if ($order['profile_url']): ?>
                                                <img src="<?php echo htmlspecialchars($order['profile_url']); ?>" 
                                                     alt="<?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?>"
                                                     onerror="this.onerror=null; this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect width=%22100%22 height=%22100%22 fill=%22%233b82f6%22/><text x=%2250%22 y=%2255%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22white%22 font-size=%2230%22><?php echo strtoupper(substr($order['first_name'], 0, 1) . substr($order['last_name'], 0, 1)); ?></text></svg>'">
                                            <?php else: ?>
                                                <?php echo strtoupper(substr($order['first_name'], 0, 1) . substr($order['last_name'], 0, 1)); ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="user-info">
                                            <h4><?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?></h4>
                                            <p><?php echo htmlspecialchars($order['email']); ?></p>
                                        </div>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Order Amount</label>
                                            <span class="amount-value">₦<?php echo number_format($order['amount'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining Amount</label>
                                            <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Interest</label>
                                            <span>₦<?php echo number_format($order['interest'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Match Status</label>
                                            <span><?php echo ucfirst($order['match_status']); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Created Date</label>
                                            <span><?php echo date('M j, Y', strtotime($order['created_at'])); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>User Status</label>
                                            <span><?php echo ucfirst($order['account_status']); ?></span>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Seller Orders -->
                    <div class="orders-list" id="sellerOrders" style="display: none;">
                        <?php if (empty($seller_orders)): ?>
                            <div class="empty-state">
                                <i class="fas fa-store"></i>
                                <h3>No Pending Seller Orders</h3>
                                <p>There are no pending seller orders at the moment</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($seller_orders as $order): ?>
                                <div class="order-item">
                                    <div class="order-header">
                                        <div class="order-id">Sell Order #<?php echo $order['id']; ?></div>
                                        <span class="order-status status-in_progress">In Progress</span>
                                    </div>
                                    <div class="order-user">
                                        <div class="user-avatar">
                                            <?php if ($order['profile_url']): ?>
                                                <img src="<?php echo htmlspecialchars($order['profile_url']); ?>" 
                                                     alt="<?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?>"
                                                     onerror="this.onerror=null; this.src='data:image/svg+xml,<svg xmlns=%22http://www.w3.org/2000/svg%22 viewBox=%220 0 100 100%22><rect width=%22100%22 height=%22100%22 fill=%22%2310b981%22/><text x=%2250%22 y=%2255%22 text-anchor=%22middle%22 dy=%22.3em%22 fill=%22white%22 font-size=%2230%22><?php echo strtoupper(substr($order['first_name'], 0, 1) . substr($order['last_name'], 0, 1)); ?></text></svg>'">
                                            <?php else: ?>
                                                <?php echo strtoupper(substr($order['first_name'], 0, 1) . substr($order['last_name'], 0, 1)); ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="user-info">
                                            <h4><?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?></h4>
                                            <p><?php echo htmlspecialchars($order['email']); ?></p>
                                        </div>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Order Amount</label>
                                            <span class="amount-value">₦<?php echo number_format($order['amount'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining Amount</label>
                                            <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Match Status</label>
                                            <span><?php echo ucfirst($order['match_status']); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Created Date</label>
                                            <span><?php echo date('M j, Y', strtotime($order['created_at'])); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>User Status</label>
                                            <span><?php echo ucfirst($order['account_status']); ?></span>
                                        </div>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Manage Order Modal -->
    <div class="modal" id="manageOrderModal">
        <div class="modal-content">
            <div class="modal-icon">
                <i class="fas fa-edit"></i>
            </div>
            <h3>Manage Order</h3>
            <form class="modal-form" id="orderManagementForm">
                <input type="hidden" id="modal_order_type" name="order_type">
                <input type="hidden" id="modal_order_id" name="order_id">
                
                <div class="form-group">
                    <label for="amount">Amount (Optional)</label>
                    <input type="number" id="amount" name="amount" step="0.01" min="0" placeholder="Enter amount to credit/debit">
                </div>
                
                <div class="form-group">
                    <label for="transaction_type">Transaction Type</label>
                    <select id="transaction_type" name="transaction_type">
                        <option value="credit">Credit</option>
                        <option value="debit">Debit</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="status">Status</label>
                    <select id="status" name="status">
                        <!-- Buyer order status options -->
                        <option value="approved">Approved</option>
                        <option value="not_approved">Not Approved</option>
                        <option value="pending">Pending</option>
                        <option value="pause">Pause</option>
                        <option value="sold">Sold</option>
                    </select>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="modal-btn cancel" id="cancelManageBtn">Cancel</button>
                    <button type="submit" class="modal-btn primary">Update Order</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Confirmation Modal for Delete -->
    <div class="modal" id="confirmationModal">
        <div class="modal-content">
            <div class="modal-icon" style="background: linear-gradient(135deg, var(--warning), #f59e0b);">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3>Confirm Deletion</h3>
            <p>Are you sure you want to delete seller order #<span id="orderIdToDelete">0</span>? This action cannot be undone.</p>
            <div class="modal-actions">
                <button class="modal-btn cancel" id="cancelDeleteBtn">Cancel</button>
                <button class="modal-btn primary" style="background: linear-gradient(90deg, var(--danger), var(--danger-dark));" id="confirmDeleteBtn">Delete Order</button>
            </div>
        </div>
    </div>
    
    <!-- AJAX Response Message -->
    <div class="ajax-message" id="ajaxMessage">
        <i class="fas fa-check-circle" id="ajaxIcon"></i>
        <span id="ajaxText"></span>
    </div>

    <script>
        // Tab switching functionality
        document.addEventListener('DOMContentLoaded', function() {
            const buyerTab = document.getElementById('buyerTab');
            const sellerTab = document.getElementById('sellerTab');
            const buyerOrders = document.getElementById('buyerOrders');
            const sellerOrders = document.getElementById('sellerOrders');
            
            buyerTab.addEventListener('click', function() {
                buyerTab.classList.add('active');
                sellerTab.classList.remove('active');
                buyerOrders.style.display = 'block';
                sellerOrders.style.display = 'none';
            });
            
            sellerTab.addEventListener('click', function() {
                sellerTab.classList.add('active');
                buyerTab.classList.remove('active');
                sellerOrders.style.display = 'block';
                buyerOrders.style.display = 'none';
            });
            
            // Manage Order Modal
            const manageOrderForm = document.getElementById('manageOrderForm');
            const manageOrderModal = document.getElementById('manageOrderModal');
            const orderManagementForm = document.getElementById('orderManagementForm');
            const cancelManageBtn = document.getElementById('cancelManageBtn');
            
            // Status select element
            const statusSelect = document.getElementById('status');
            
            manageOrderForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const orderType = document.getElementById('order_type').value;
                const orderId = document.getElementById('order_id').value.trim();
                
                if (!orderType || !orderId) {
                    showAjaxMessage('Please select order type and enter order ID', 'error');
                    return;
                }
                
                // Update hidden fields in modal form
                document.getElementById('modal_order_type').value = orderType;
                document.getElementById('modal_order_id').value = orderId;
                
                // Update status options based on order type
                if (orderType === 'buyer') {
                    statusSelect.innerHTML = `
                        <option value="approved">Approved</option>
                        <option value="not_approved">Not Approved</option>
                        <option value="pending">Pending</option>
                        <option value="pause">Pause</option>
                        <option value="sold">Sold</option>
                    `;
                } else if (orderType === 'seller') {
                    statusSelect.innerHTML = `
                        <option value="in progress">In Progress</option>
                        <option value="sold">Sold</option>
                        <option value="pause">Pause</option>
                    `;
                }
                
                // Reset form fields
                document.getElementById('amount').value = '';
                document.getElementById('transaction_type').value = 'credit';
                
                // Show modal
                manageOrderModal.classList.add('active');
            });
            
            // Handle order management form submission
            orderManagementForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(this);
                
                // Show loading state
                const submitBtn = orderManagementForm.querySelector('button[type="submit"]');
                const originalText = submitBtn.textContent;
                submitBtn.textContent = 'Processing...';
                submitBtn.disabled = true;
                
                // Send AJAX request
                fetch('b_set.php', {
                    method: 'POST',
                    body: formData
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showAjaxMessage(data.message, 'success');
                        setTimeout(() => {
                            window.location.reload();
                        }, 1500);
                    } else {
                        showAjaxMessage(data.message, 'error');
                        submitBtn.textContent = originalText;
                        submitBtn.disabled = false;
                    }
                })
                .catch(error => {
                    showAjaxMessage('An error occurred. Please try again.', 'error');
                    submitBtn.textContent = originalText;
                    submitBtn.disabled = false;
                    console.error('Error:', error);
                });
            });
            
            // Cancel manage order
            cancelManageBtn.addEventListener('click', function() {
                manageOrderModal.classList.remove('active');
            });
            
            // Confirmation modal for delete
            const deleteForm = document.getElementById('deleteForm');
            const confirmationModal = document.getElementById('confirmationModal');
            const orderIdToDelete = document.getElementById('orderIdToDelete');
            const confirmDeleteBtn = document.getElementById('confirmDeleteBtn');
            const cancelDeleteBtn = document.getElementById('cancelDeleteBtn');
            
            deleteForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const orderIdInput = deleteForm.querySelector('input[name="delete_order_id"]');
                const orderId = orderIdInput.value.trim();
                
                if (!orderId) {
                    showAjaxMessage('Please enter a seller order ID', 'error');
                    return;
                }
                
                // Show confirmation modal
                orderIdToDelete.textContent = orderId;
                confirmationModal.classList.add('active');
            });
            
            // Handle delete confirmation
            confirmDeleteBtn.addEventListener('click', function() {
                // Submit the form
                deleteForm.submit();
            });
            
            // Handle cancel
            cancelDeleteBtn.addEventListener('click', function() {
                confirmationModal.classList.remove('active');
            });
            
            // Close modals when clicking outside
            [manageOrderModal, confirmationModal].forEach(modal => {
                modal.addEventListener('click', function(e) {
                    if (e.target === modal) {
                        modal.classList.remove('active');
                    }
                });
            });
            
            // Function to show AJAX messages
            function showAjaxMessage(message, type) {
                const ajaxMessage = document.getElementById('ajaxMessage');
                const ajaxIcon = document.getElementById('ajaxIcon');
                const ajaxText = document.getElementById('ajaxText');
                
                ajaxMessage.className = 'ajax-message ' + type;
                ajaxText.textContent = message;
                
                if (type === 'success') {
                    ajaxIcon.className = 'fas fa-check-circle';
                } else {
                    ajaxIcon.className = 'fas fa-exclamation-circle';
                }
                
                ajaxMessage.classList.add('show');
                
                setTimeout(() => {
                    ajaxMessage.classList.remove('show');
                }, 5000);
            }
        });
    </script>
</body>
</html>