<?php
// order.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
    header('Location: auction.php');
    exit;
}

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}

// Include database connection
require_once 'php/db.php';

// Fetch initial buyer and seller orders (first 20 for better initial view)
try {
    // Get buyer orders (initial load - 20 orders)
    $buyerStmt = $pdo->prepare("
        SELECT bo.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, 
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at as user_created_at
        FROM buyer_orders bo 
        JOIN users u ON bo.user_id = u.user_id 
        WHERE bo.status != 'completed'
        ORDER BY bo.created_at DESC 
        LIMIT 20
    ");
    $buyerStmt->execute();
    $buyer_orders = $buyerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get seller orders (initial load - 20 orders)
    $sellerStmt = $pdo->prepare("
        SELECT so.*, u.username, u.first_name, u.last_name, u.email, u.phone_number,
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at as user_created_at
        FROM seller_orders so 
        JOIN users u ON so.user_id = u.user_id 
        WHERE so.status != 'completed'
        ORDER BY so.created_at DESC 
        LIMIT 20
    ");
    $sellerStmt->execute();
    $seller_orders = $sellerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get total counts
    $buyerCountStmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM buyer_orders 
        WHERE status != 'completed'
    ");
    $buyerCountStmt->execute();
    $buyer_total = $buyerCountStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $sellerCountStmt = $pdo->prepare("
        SELECT COUNT(*) as total 
        FROM seller_orders 
        WHERE status != 'completed'
    ");
    $sellerCountStmt->execute();
    $seller_total = $sellerCountStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get latest IDs for polling
    $latestBuyerId = !empty($buyer_orders) ? $buyer_orders[0]['id'] : 0;
    $latestSellerId = !empty($seller_orders) ? $seller_orders[0]['id'] : 0;
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Orders Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
            --header-height: 80px;
            --search-height: 180px;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 0;
            transition: all 0.3s ease;
            height: 100vh;
            overflow-y: auto;
            position: relative;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        /* Sticky Header */
        .header-section {
            position: sticky;
            top: 0;
            z-index: 1000;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            padding: 20px 30px 10px 30px;
            transition: all 0.3s ease;
        }
        
        .header-section.scrolled {
            padding: 10px 30px;
            background: rgba(248, 250, 252, 0.95);
            backdrop-filter: blur(10px);
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.1);
        }
        
        .header-section.scrolled .search-container {
            opacity: 0;
            visibility: hidden;
            height: 0;
            margin: 0;
            padding: 0;
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .header-section.scrolled .header {
            margin-bottom: 0;
            padding-bottom: 10px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
            transition: all 0.3s ease;
        }
        
        .header-section.scrolled .header {
            margin-bottom: 0;
            padding-bottom: 10px;
            border-bottom: 1px solid rgba(229, 231, 235, 0.5);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header p {
            color: var(--gray);
            font-size: 14px;
            margin-top: 4px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Search Container - Fades out on scroll */
        .search-container {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
            transform-origin: top;
        }
        
        .search-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .search-header h3 {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .search-form {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .search-input-group {
            flex: 1;
            display: flex;
            gap: 10px;
        }
        
        .search-input {
            flex: 1;
            padding: 12px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .search-select {
            padding: 12px 20px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            background: white;
            color: var(--dark);
            cursor: pointer;
            min-width: 150px;
            transition: all 0.3s ease;
        }
        
        .search-select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .search-btn {
            padding: 12px 30px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .search-btn:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .search-results {
            margin-top: 20px;
            display: none;
        }
        
        .search-results.active {
            display: block;
        }
        
        .search-result-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
        }
        
        .search-result-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .no-results {
            text-align: center;
            padding: 40px;
            color: var(--gray);
        }
        
        .no-results i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--gray-light);
        }
        
        .no-results h4 {
            font-size: 18px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        /* Content Section */
        .content-section {
            padding: 0 30px 30px 30px;
        }
        
        .orders-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            position: relative;
        }
        
        .orders-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
            padding-bottom: 10px;
            position: sticky;
            top: var(--header-height);
            background: white;
            z-index: 100;
        }
        
        .order-tab {
            padding: 10px 20px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            background: var(--gray-light);
            color: var(--gray);
            position: relative;
        }
        
        .order-tab.active {
            background: var(--primary);
            color: white;
        }
        
        .order-tab:hover:not(.active) {
            background: #d1d5db;
        }
        
        .new-orders-badge {
            position: absolute;
            top: -8px;
            right: -8px;
            background: var(--danger);
            color: white;
            font-size: 11px;
            font-weight: 700;
            min-width: 20px;
            height: 20px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            padding: 0 6px;
            animation: pulse 2s infinite;
            cursor: pointer;
            z-index: 10;
            border: 2px solid white;
        }
        
        .new-orders-badge:hover {
            transform: scale(1.1);
        }
        
        @keyframes pulse {
            0% {
                box-shadow: 0 0 0 0 rgba(239, 68, 68, 0.7);
            }
            70% {
                box-shadow: 0 0 0 6px rgba(239, 68, 68, 0);
            }
            100% {
                box-shadow: 0 0 0 0 rgba(239, 68, 68, 0);
            }
        }
        
        .orders-content {
            min-height: 400px;
            position: relative;
        }
        
        .orders-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .order-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
            cursor: pointer;
            position: relative;
            border: 1px solid transparent;
        }
        
        .order-item.new-order {
            animation: highlight 2s ease-in-out;
            border-left: 4px solid var(--primary);
        }
        
        @keyframes highlight {
            0% {
                background: #dbeafe;
                transform: translateX(-10px);
            }
            100% {
                background: var(--light);
                transform: translateX(0);
            }
        }
        
        .order-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
            border-color: var(--primary-light);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .order-id {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .order-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-pending {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-approved {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-not_approved {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .status-pause {
            background: #d1d5db;
            color: #4b5563;
        }
        
        .status-sold {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .status-in_progress {
            background: #f3e8ff;
            color: #7c3aed;
        }
        
        .status-completed {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-cancelled {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .order-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .order-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .order-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .order-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .order-user {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 10px;
        }
        
        .user-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 14px;
            flex-shrink: 0;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .user-info h4 {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 2px;
        }
        
        .user-info p {
            font-size: 12px;
            color: var(--gray);
        }
        
        .order-actions {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }
        
        .action-btn {
            padding: 8px 16px;
            border-radius: 6px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
            transition: all 0.3s ease;
        }
        
        .action-btn.refund {
            background: var(--danger);
            color: white;
        }
        
        .action-btn.refund:hover {
            background: #dc2626;
            transform: translateY(-1px);
        }
        
        .action-btn.pause {
            background: var(--warning);
            color: white;
        }
        
        .action-btn.pause:hover {
            background: #d97706;
            transform: translateY(-1px);
        }
        
        .action-btn.resume {
            background: var(--success);
            color: white;
        }
        
        .action-btn.resume:hover {
            background: #059669;
            transform: translateY(-1px);
        }
        
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 14px;
        }
        
        .loading-spinner {
            display: inline-block;
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        .infinite-loader {
            text-align: center;
            padding: 30px;
            color: var(--gray);
            display: flex;
            flex-direction: column;
            align-items: center;
            gap: 10px;
        }
        
        .infinite-loader.hidden {
            display: none;
        }
        
        .infinite-spinner {
            width: 40px;
            height: 40px;
            border: 4px solid var(--gray-light);
            border-radius: 50%;
            border-top-color: var(--primary);
            animation: spin 1s ease-in-out infinite;
        }
        
        .no-more-orders {
            text-align: center;
            padding: 30px;
            color: var(--gray);
            font-size: 14px;
            border-top: 1px solid var(--gray-light);
            margin-top: 20px;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Scroll to top button */
        .scroll-top {
            position: fixed;
            bottom: 30px;
            right: 30px;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: var(--primary);
            color: white;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            box-shadow: 0 4px 15px rgba(59, 130, 246, 0.3);
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            z-index: 2000;
            border: none;
        }
        
        .scroll-top.visible {
            opacity: 1;
            visibility: visible;
        }
        
        .scroll-top:hover {
            transform: translateY(-5px);
            background: var(--primary-dark);
        }
        
        /* Modal styles (keeping existing modal styles) */
        .profile-header {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 30px;
        }
        
        .profile-avatar-container {
            position: relative;
        }
        
        .profile-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 36px;
            flex-shrink: 0;
        }
        
        .profile-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .profile-verified-badge {
            position: absolute;
            bottom: 5px;
            right: 5px;
            background: var(--primary);
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            border: 3px solid white;
        }
        
        .profile-info {
            flex: 1;
        }
        
        .profile-name {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .profile-name h3 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .profile-status {
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-active {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-suspended {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-closed {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .profile-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .profile-detail-item {
            display: flex;
            flex-direction: column;
        }
        
        .profile-detail-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .profile-detail-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .verified-icon {
            color: var(--primary);
            font-size: 12px;
        }
        
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .order-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 900px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .order-modal {
            transform: translateY(0);
        }
        
        .order-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .order-details-modal {
            margin-bottom: 25px;
        }
        
        .order-detail-group {
            margin-bottom: 20px;
        }
        
        .order-detail-group h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .order-detail-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .order-detail-item {
            display: flex;
            flex-direction: column;
        }
        
        .order-detail-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .order-detail-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .bank-details {
            background: var(--light);
            border-radius: 10px;
            padding: 20px;
            margin-top: 15px;
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .modal-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .modal-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .modal-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .modal-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .modal-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .modal-btn.success {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .modal-btn.success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .modal-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        .modal-btn.danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        .receipt-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .receipt-image {
            width: 100%;
            max-height: 500px;
            object-fit: contain;
            border-radius: 10px;
            margin: 20px 0;
        }
        
        .alert-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .alert-modal {
            transform: translateY(0);
        }
        
        .alert-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .alert-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-icon.info {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .alert-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .alert-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .alert-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .alert-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .confirmation-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .confirmation-modal {
            transform: translateY(0);
        }
        
        .confirmation-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--warning), #f59e0b);
            color: white;
        }
        
        .confirmation-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .confirmation-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .confirmation-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .confirmation-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .confirmation-btn.confirm {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .confirmation-btn.confirm:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .confirmation-btn.cancel {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .confirmation-btn.cancel:hover {
            background: #d1d5db;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .profile-header {
                flex-direction: column;
                text-align: center;
            }
        }
        
        @media (max-width: 768px) {
            .header-section {
                padding: 15px 20px 5px 20px;
            }
            
            .content-section {
                padding: 0 20px 20px 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .search-form {
                flex-direction: column;
                align-items: stretch;
            }
            
            .search-input-group {
                flex-direction: column;
            }
            
            .orders-tabs {
                flex-wrap: wrap;
            }
            
            .order-modal, .receipt-modal, .alert-modal, .confirmation-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .modal-actions, .alert-actions, .confirmation-actions {
                flex-direction: column;
            }
            
            .modal-btn, .alert-btn, .confirmation-btn {
                width: 100%;
            }
            
            .profile-details {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 480px) {
            .order-details {
                grid-template-columns: 1fr;
            }
            
            .order-detail-row {
                grid-template-columns: 1fr;
            }
            
            .order-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content" id="mainContent">
            <!-- Sticky Header Section -->
            <div class="header-section" id="headerSection">
                <div class="header">
                    <div>
                        <h2>Orders Management</h2>
                        <p>Manage buyer and seller orders</p>
                    </div>
                    <div class="header-actions">
                        <div class="time-display">
                            <i class="fas fa-clock"></i>
                            <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                        </div>
                    </div>
                </div>
                
                <!-- Search Container - Fades out on scroll -->
                <div class="search-container" id="searchContainer">
                    <div class="search-header">
                        <h3>Search Orders</h3>
                    </div>
                    <form id="searchForm" class="search-form">
                        <div class="search-input-group">
                            <input type="number" id="searchInput" class="search-input" placeholder="Enter Order ID" min="1">
                            <select id="searchType" class="search-select">
                                <option value="buyer">Buyer Orders</option>
                                <option value="seller">Seller Orders</option>
                            </select>
                        </div>
                        <button type="submit" class="search-btn">
                            <i class="fas fa-search"></i> Search
                        </button>
                    </form>
                    <div class="search-results" id="searchResults"></div>
                </div>
            </div>
            
            <!-- Content Section -->
            <div class="content-section">
                <!-- Orders Container -->
                <div class="orders-container">
                    <div class="orders-tabs">
                        <button class="order-tab active" data-type="buyer" data-page="1" data-total="<?php echo $buyer_total; ?>" data-has-more="<?php echo $buyer_total > 20 ? 'true' : 'false'; ?>" id="buyerTab">
                            Buyer Orders (<?php echo count($buyer_orders); ?>/<?php echo $buyer_total; ?>)
                        </button>
                        <button class="order-tab" data-type="seller" data-page="1" data-total="<?php echo $seller_total; ?>" data-has-more="<?php echo $seller_total > 20 ? 'true' : 'false'; ?>" id="sellerTab">
                            Seller Orders (<?php echo count($seller_orders); ?>/<?php echo $seller_total; ?>)
                        </button>
                    </div>
                    
                    <div class="orders-content">
                        <!-- Buyer Orders -->
                        <div class="orders-list" id="buyerOrders">
                            <?php if (empty($buyer_orders)): ?>
                                <div class="empty-state">
                                    <i class="fas fa-shopping-cart"></i>
                                    <h3>No Buyer Orders</h3>
                                    <p>There are no pending buyer orders</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($buyer_orders as $order): ?>
                                    <div class="order-item" data-id="<?php echo $order['id']; ?>" onclick="viewBuyerOrder(<?php echo $order['id']; ?>)">
                                        <div class="order-header">
                                            <div class="order-id">Buy Order #<?php echo $order['id']; ?></div>
                                            <span class="order-status status-<?php echo $order['status']; ?>">
                                                <?php echo str_replace('_', ' ', ucfirst($order['status'])); ?>
                                            </span>
                                        </div>
                                        <div class="order-user">
                                            <div class="user-avatar">
                                                <?php if ($order['profile_url']): ?>
                                                    <img src="<?php echo htmlspecialchars($order['profile_url']); ?>" alt="<?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?>">
                                                <?php else: ?>
                                                    <?php echo getUserInitials($order['first_name'] . ' ' . $order['last_name']); ?>
                                                <?php endif; ?>
                                            </div>
                                            <div class="user-info">
                                                <h4><?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?></h4>
                                                <p><?php echo htmlspecialchars($order['email']); ?></p>
                                            </div>
                                        </div>
                                        <div class="order-details">
                                            <div class="order-info-item">
                                                <label>Amount</label>
                                                <span>₦<?php echo number_format($order['amount'], 2); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Remaining</label>
                                                <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Interest</label>
                                                <span>₦<?php echo number_format($order['interest'], 2); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Match Status</label>
                                                <span><?php echo ucfirst($order['match_status']); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Created</label>
                                                <span><?php echo date('M j, Y', strtotime($order['created_at'])); ?></span>
                                            </div>
                                        </div>
                                        <?php if ($order['status'] === 'pending'): ?>
                                            <div class="order-actions">
                                                <button class="action-btn refund" onclick="event.stopPropagation(); showRefundConfirmation(<?php echo $order['id']; ?>, '<?php echo htmlspecialchars($order['first_name']); ?>')">
                                                    <i class="fas fa-undo"></i> Refund
                                                </button>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                        
                        <!-- Seller Orders -->
                        <div class="orders-list" id="sellerOrders" style="display: none;">
                            <?php if (empty($seller_orders)): ?>
                                <div class="empty-state">
                                    <i class="fas fa-shopping-cart"></i>
                                    <h3>No Seller Orders</h3>
                                    <p>There are no pending seller orders</p>
                                </div>
                            <?php else: ?>
                                <?php foreach ($seller_orders as $order): ?>
                                    <div class="order-item" data-id="<?php echo $order['id']; ?>" onclick="viewSellerOrder(<?php echo $order['id']; ?>)">
                                        <div class="order-header">
                                            <div class="order-id">Sell Order #<?php echo $order['id']; ?></div>
                                            <span class="order-status status-<?php echo $order['status']; ?>">
                                                <?php echo str_replace('_', ' ', ucfirst($order['status'])); ?>
                                            </span>
                                        </div>
                                        <div class="order-user">
                                            <div class="user-avatar">
                                                <?php if ($order['profile_url']): ?>
                                                    <img src="<?php echo htmlspecialchars($order['profile_url']); ?>" alt="<?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?>">
                                                <?php else: ?>
                                                    <?php echo getUserInitials($order['first_name'] . ' ' . $order['last_name']); ?>
                                                <?php endif; ?>
                                            </div>
                                            <div class="user-info">
                                                <h4><?php echo htmlspecialchars($order['first_name'] . ' ' . $order['last_name']); ?></h4>
                                                <p><?php echo htmlspecialchars($order['email']); ?></p>
                                            </div>
                                        </div>
                                        <div class="order-details">
                                            <div class="order-info-item">
                                                <label>Amount</label>
                                                <span>₦<?php echo number_format($order['amount'], 2); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Remaining</label>
                                                <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Match Status</label>
                                                <span><?php echo ucfirst($order['match_status']); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Status</label>
                                                <span><?php echo str_replace('_', ' ', ucfirst($order['status'])); ?></span>
                                            </div>
                                            <div class="order-info-item">
                                                <label>Created</label>
                                                <span><?php echo date('M j, Y', strtotime($order['created_at'])); ?></span>
                                            </div>
                                        </div>
                                        <?php if ($order['match_status'] === 'pending'): ?>
                                            <div class="order-actions">
                                                <?php if ($order['status'] === 'pause'): ?>
                                                    <button class="action-btn resume" onclick="event.stopPropagation(); showToggleConfirmation(<?php echo $order['id']; ?>, 'resume')">
                                                        <i class="fas fa-play"></i> Resume
                                                    </button>
                                                <?php else: ?>
                                                    <button class="action-btn pause" onclick="event.stopPropagation(); showToggleConfirmation(<?php echo $order['id']; ?>, 'pause')">
                                                        <i class="fas fa-pause"></i> Pause
                                                    </button>
                                                <?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </div>
                    </div>
                    
                    <!-- Infinite Scroll Loader -->
                    <div class="infinite-loader" id="infiniteLoader">
                        <div class="infinite-spinner"></div>
                        <p>Loading more orders...</p>
                    </div>
                    
                    <!-- No More Orders Indicator -->
                    <div class="no-more-orders hidden" id="noMoreOrders">
                        <i class="fas fa-check-circle"></i> No more orders to load
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Scroll to Top Button -->
    <button class="scroll-top" id="scrollTop" onclick="scrollToTop()">
        <i class="fas fa-arrow-up"></i>
    </button>
    
    <!-- Buyer Order Details Modal -->
    <div class="modal-overlay" id="buyerOrderModal">
        <div class="order-modal">
            <h3>Buyer Order Details</h3>
            <div class="order-details-modal" id="buyerOrderDetails">
                <!-- Buyer order details will be loaded here -->
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" id="closeBuyerModalBtn">Close</button>
                <button type="button" class="modal-btn success" id="approvePaymentBtn" style="display: none;">
                    <i class="fas fa-check"></i> Approve Payment
                </button>
            </div>
        </div>
    </div>
    
    <!-- Seller Order Details Modal -->
    <div class="modal-overlay" id="sellerOrderModal">
        <div class="order-modal">
            <h3>Seller Order Details</h3>
            <div class="order-details-modal" id="sellerOrderDetails">
                <!-- Seller order details will be loaded here -->
            </div>
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" id="closeSellerModalBtn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Receipt Modal -->
    <div class="modal-overlay" id="receiptModal">
        <div class="receipt-modal">
            <h3>Payment Receipt</h3>
            <img id="receiptImage" src="" alt="Payment Receipt" class="receipt-image">
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" id="closeReceiptBtn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Confirmation Modal -->
    <div class="modal-overlay" id="confirmationModal">
        <div class="confirmation-modal">
            <div class="confirmation-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmationTitle">Confirm Action</h3>
            <p id="confirmationText">Are you sure you want to perform this action?</p>
            <div class="confirmation-actions">
                <button class="confirmation-btn cancel" id="confirmationCancelBtn">Cancel</button>
                <button class="confirmation-btn confirm" id="confirmationConfirmBtn">Confirm</button>
            </div>
        </div>
    </div>
    
    <!-- Message Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="alert-modal">
            <div class="alert-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="alert-actions">
                <button class="alert-btn primary" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <script>
        // ============================================
        // INFINITE SCROLL CONFIGURATION
        // ============================================
        const INFINITE_SCROLL_CONFIG = {
            buyer: {
                page: 1,
                hasMore: <?php echo $buyer_total > 20 ? 'true' : 'false'; ?>,
                isLoading: false,
                totalItems: <?php echo $buyer_total; ?>,
                loadedCount: <?php echo count($buyer_orders); ?>
            },
            seller: {
                page: 1,
                hasMore: <?php echo $seller_total > 20 ? 'true' : 'false'; ?>,
                isLoading: false,
                totalItems: <?php echo $seller_total; ?>,
                loadedCount: <?php echo count($seller_orders); ?>
            }
        };
        
        let currentTabType = 'buyer';
        let isLoading = false;
        let isPolling = true;
        let lastScrollTop = 0;
        
        // Polling variables
        let lastBuyerId = <?php echo $latestBuyerId; ?>;
        let lastSellerId = <?php echo $latestSellerId; ?>;
        let pollingInterval = 5000;
        let newBuyerOrders = [];
        let newSellerOrders = [];
        let newBuyerCount = 0;
        let newSellerCount = 0;
        
        // Persisted badge counts
        let persistedNewBuyerCount = localStorage.getItem('newBuyerCount') ? parseInt(localStorage.getItem('newBuyerCount')) : 0;
        let persistedNewSellerCount = localStorage.getItem('newSellerCount') ? parseInt(localStorage.getItem('newSellerCount')) : 0;
        newBuyerCount = persistedNewBuyerCount;
        newSellerCount = persistedNewSellerCount;
        
        // ============================================
        // SCROLL HANDLER FOR HEADER FADE
        // ============================================
        function handleHeaderScroll() {
            const mainContent = document.getElementById('mainContent');
            const headerSection = document.getElementById('headerSection');
            const scrollTop = mainContent.scrollTop;
            const scrollThreshold = 50;
            
            if (scrollTop > scrollThreshold) {
                headerSection.classList.add('scrolled');
            } else {
                headerSection.classList.remove('scrolled');
            }
            
            // Show/hide scroll to top button
            const scrollTopBtn = document.getElementById('scrollTop');
            if (scrollTop > 300) {
                scrollTopBtn.classList.add('visible');
            } else {
                scrollTopBtn.classList.remove('visible');
            }
            
            lastScrollTop = scrollTop;
        }
        
        function scrollToTop() {
            const mainContent = document.getElementById('mainContent');
            mainContent.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        }
        
        // ============================================
        // TIME UPDATE FUNCTION
        // ============================================
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // ============================================
        // MODAL FUNCTIONS
        // ============================================
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            icon.className = 'alert-icon ' + type;
            switch(type) {
                case 'success':
                    icon.innerHTML = '<i class="fas fa-check-circle"></i>';
                    break;
                case 'error':
                    icon.innerHTML = '<i class="fas fa-times-circle"></i>';
                    break;
                case 'warning':
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    break;
                case 'info':
                    icon.innerHTML = '<i class="fas fa-info-circle"></i>';
                    break;
            }
            
            modal.classList.add('active');
        }
        
        function showConfirmationModal(title, message, onConfirm) {
            const modal = document.getElementById('confirmationModal');
            const titleEl = document.getElementById('confirmationTitle');
            const messageEl = document.getElementById('confirmationText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            const confirmBtn = document.getElementById('confirmationConfirmBtn');
            confirmBtn.onclick = onConfirm;
            
            modal.classList.add('active');
        }
        
        function showRefundConfirmation(orderId, userName) {
            event.stopPropagation();
            showConfirmationModal(
                'Confirm Refund',
                `Are you sure you want to refund ${userName}'s buyer order? This action cannot be undone.`,
                () => {
                    document.getElementById('confirmationModal').classList.remove('active');
                    refundBuyerOrder(orderId, userName);
                }
            );
        }
        
        function showToggleConfirmation(orderId, action) {
            event.stopPropagation();
            const actionText = action === 'pause' ? 'pause' : 'resume';
            const confirmText = action === 'pause' 
                ? 'Are you sure you want to pause this seller order?' 
                : 'Are you sure you want to resume this seller order?';
            
            showConfirmationModal(
                `Confirm ${actionText.charAt(0).toUpperCase() + actionText.slice(1)}`,
                confirmText,
                () => {
                    document.getElementById('confirmationModal').classList.remove('active');
                    toggleSellerOrderStatus(orderId, action);
                }
            );
        }
        
        // ============================================
        // BADGE FUNCTIONS
        // ============================================
        function saveBadgeCounts() {
            localStorage.setItem('newBuyerCount', newBuyerCount);
            localStorage.setItem('newSellerCount', newSellerCount);
        }
        
        function clearBuyerBadge() {
            newBuyerCount = 0;
            newBuyerOrders = [];
            saveBadgeCounts();
            updateNewOrdersBadges();
        }
        
        function clearSellerBadge() {
            newSellerCount = 0;
            newSellerOrders = [];
            saveBadgeCounts();
            updateNewOrdersBadges();
        }
        
        function updateNewOrdersBadges() {
            const buyerTab = document.getElementById('buyerTab');
            const sellerTab = document.getElementById('sellerTab');
            
            const existingBuyerBadge = buyerTab.querySelector('.new-orders-badge');
            const existingSellerBadge = sellerTab.querySelector('.new-orders-badge');
            
            if (existingBuyerBadge) existingBuyerBadge.remove();
            if (existingSellerBadge) existingSellerBadge.remove();
            
            if (newBuyerCount > 0) {
                const badge = document.createElement('span');
                badge.className = 'new-orders-badge';
                badge.textContent = newBuyerCount > 99 ? '99+' : newBuyerCount;
                badge.title = `${newBuyerCount} new buyer order${newBuyerCount > 1 ? 's' : ''}`;
                badge.onclick = (e) => {
                    e.stopPropagation();
                    clearBuyerBadge();
                    if (currentTabType !== 'buyer') {
                        document.getElementById('buyerTab').click();
                    } else {
                        addNewBuyerOrders();
                    }
                };
                buyerTab.appendChild(badge);
                buyerTab.style.position = 'relative';
            }
            
            if (newSellerCount > 0) {
                const badge = document.createElement('span');
                badge.className = 'new-orders-badge';
                badge.textContent = newSellerCount > 99 ? '99+' : newSellerCount;
                badge.title = `${newSellerCount} new seller order${newSellerCount > 1 ? 's' : ''}`;
                badge.onclick = (e) => {
                    e.stopPropagation();
                    clearSellerBadge();
                    if (currentTabType !== 'seller') {
                        document.getElementById('sellerTab').click();
                    } else {
                        addNewSellerOrders();
                    }
                };
                sellerTab.appendChild(badge);
                sellerTab.style.position = 'relative';
            }
        }
        
        // ============================================
        // CREATE ORDER HTML
        // ============================================
        function createBuyerOrderHtml(order) {
            const initials = order.first_name ? 
                (order.first_name[0].toUpperCase() + (order.last_name ? order.last_name[0].toUpperCase() : '')) : 
                '??';
            
            return `
                <div class="order-item" data-id="${order.id}" onclick="viewBuyerOrder(${order.id})">
                    <div class="order-header">
                        <div class="order-id">Buy Order #${order.id}</div>
                        <span class="order-status status-${order.status}">
                            ${order.status ? order.status.replace('_', ' ') : ''}
                        </span>
                    </div>
                    <div class="order-user">
                        <div class="user-avatar">
                            ${order.profile_url ? 
                                `<img src="${order.profile_url}" alt="${order.first_name} ${order.last_name}">` : 
                                initials
                            }
                        </div>
                        <div class="user-info">
                            <h4>${order.first_name || ''} ${order.last_name || ''}</h4>
                            <p>${order.email || ''}</p>
                        </div>
                    </div>
                    <div class="order-details">
                        <div class="order-info-item">
                            <label>Amount</label>
                            <span>₦${parseFloat(order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Remaining</label>
                            <span>₦${parseFloat(order.amount_remaining || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Interest</label>
                            <span>₦${parseFloat(order.interest || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Match Status</label>
                            <span>${order.match_status || ''}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Created</label>
                            <span>${order.created_at ? new Date(order.created_at).toLocaleDateString('en-US', {month: 'short', day: 'numeric', year: 'numeric'}) : ''}</span>
                        </div>
                    </div>
                    ${order.status === 'pending' ? `
                        <div class="order-actions">
                            <button class="action-btn refund" onclick="event.stopPropagation(); showRefundConfirmation(${order.id}, '${order.first_name || ''}')">
                                <i class="fas fa-undo"></i> Refund
                            </button>
                        </div>
                    ` : ''}
                </div>
            `;
        }
        
        function createSellerOrderHtml(order) {
            const initials = order.first_name ? 
                (order.first_name[0].toUpperCase() + (order.last_name ? order.last_name[0].toUpperCase() : '')) : 
                '??';
            
            return `
                <div class="order-item" data-id="${order.id}" onclick="viewSellerOrder(${order.id})">
                    <div class="order-header">
                        <div class="order-id">Sell Order #${order.id}</div>
                        <span class="order-status status-${order.status}">
                            ${order.status ? order.status.replace('_', ' ') : ''}
                        </span>
                    </div>
                    <div class="order-user">
                        <div class="user-avatar">
                            ${order.profile_url ? 
                                `<img src="${order.profile_url}" alt="${order.first_name} ${order.last_name}">` : 
                                initials
                            }
                        </div>
                        <div class="user-info">
                            <h4>${order.first_name || ''} ${order.last_name || ''}</h4>
                            <p>${order.email || ''}</p>
                        </div>
                    </div>
                    <div class="order-details">
                        <div class="order-info-item">
                            <label>Amount</label>
                            <span>₦${parseFloat(order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Remaining</label>
                            <span>₦${parseFloat(order.amount_remaining || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Match Status</label>
                            <span>${order.match_status || ''}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Status</label>
                            <span>${order.status ? order.status.replace('_', ' ') : ''}</span>
                        </div>
                        <div class="order-info-item">
                            <label>Created</label>
                            <span>${order.created_at ? new Date(order.created_at).toLocaleDateString('en-US', {month: 'short', day: 'numeric', year: 'numeric'}) : ''}</span>
                        </div>
                    </div>
                    ${order.match_status === 'pending' ? `
                        <div class="order-actions">
                            ${order.status === 'pause' ? `
                                <button class="action-btn resume" onclick="event.stopPropagation(); showToggleConfirmation(${order.id}, 'resume')">
                                    <i class="fas fa-play"></i> Resume
                                </button>
                            ` : `
                                <button class="action-btn pause" onclick="event.stopPropagation(); showToggleConfirmation(${order.id}, 'pause')">
                                    <i class="fas fa-pause"></i> Pause
                                </button>
                            `}
                        </div>
                    ` : ''}
                </div>
            `;
        }
        
        // ============================================
        // ADD NEW ORDERS FROM POLLING
        // ============================================
        function addNewBuyerOrders() {
            if (newBuyerOrders.length === 0) return;
            
            const container = document.getElementById('buyerOrders');
            const emptyState = container.querySelector('.empty-state');
            
            if (emptyState) {
                emptyState.remove();
            }
            
            newBuyerOrders.forEach(order => {
                if (container.querySelector(`.order-item[data-id="${order.id}"]`)) {
                    return;
                }
                
                const orderHtml = createBuyerOrderHtml(order);
                if (container.firstChild) {
                    container.insertAdjacentHTML('afterbegin', orderHtml);
                } else {
                    container.innerHTML = orderHtml;
                }
                
                const newElement = container.querySelector(`.order-item[data-id="${order.id}"]`);
                if (newElement) {
                    newElement.classList.add('new-order');
                    setTimeout(() => {
                        newElement.classList.remove('new-order');
                    }, 2000);
                }
            });
            
            clearBuyerBadge();
            updateTabCount('buyer');
            
            INFINITE_SCROLL_CONFIG.buyer.loadedCount = container.querySelectorAll('.order-item').length;
            INFINITE_SCROLL_CONFIG.buyer.hasMore = INFINITE_SCROLL_CONFIG.buyer.loadedCount < INFINITE_SCROLL_CONFIG.buyer.totalItems;
        }
        
        function addNewSellerOrders() {
            if (newSellerOrders.length === 0) return;
            
            const container = document.getElementById('sellerOrders');
            const emptyState = container.querySelector('.empty-state');
            
            if (emptyState) {
                emptyState.remove();
            }
            
            newSellerOrders.forEach(order => {
                if (container.querySelector(`.order-item[data-id="${order.id}"]`)) {
                    return;
                }
                
                const orderHtml = createSellerOrderHtml(order);
                if (container.firstChild) {
                    container.insertAdjacentHTML('afterbegin', orderHtml);
                } else {
                    container.innerHTML = orderHtml;
                }
                
                const newElement = container.querySelector(`.order-item[data-id="${order.id}"]`);
                if (newElement) {
                    newElement.classList.add('new-order');
                    setTimeout(() => {
                        newElement.classList.remove('new-order');
                    }, 2000);
                }
            });
            
            clearSellerBadge();
            updateTabCount('seller');
            
            INFINITE_SCROLL_CONFIG.seller.loadedCount = container.querySelectorAll('.order-item').length;
            INFINITE_SCROLL_CONFIG.seller.hasMore = INFINITE_SCROLL_CONFIG.seller.loadedCount < INFINITE_SCROLL_CONFIG.seller.totalItems;
        }
        
        // ============================================
        // UPDATE TAB COUNT
        // ============================================
        function updateTabCount(type) {
            const tab = document.getElementById(`${type}Tab`);
            const container = document.getElementById(`${type}Orders`);
            const totalItems = container.querySelectorAll('.order-item').length;
            const totalDb = INFINITE_SCROLL_CONFIG[type].totalItems;
            
            tab.textContent = `${type === 'buyer' ? 'Buyer' : 'Seller'} Orders (${totalItems}/${totalDb})`;
        }
        
        // ============================================
        // INFINITE SCROLL - LOAD MORE ORDERS
        // ============================================
        async function loadMoreOrders() {
            const config = INFINITE_SCROLL_CONFIG[currentTabType];
            
            if (config.isLoading || !config.hasMore) {
                return;
            }
            
            config.isLoading = true;
            isLoading = true;
            
            const loader = document.getElementById('infiniteLoader');
            loader.classList.remove('hidden');
            
            try {
                const container = document.getElementById(`${currentTabType}Orders`);
                const nextPage = config.page + 1;
                
                const response = await fetch(`b_orders.php?action=load_more_orders&type=${currentTabType}&page=${nextPage}&loaded_count=${config.loadedCount}`);
                const data = await response.json();
                
                if (data.success) {
                    if (data.orders && data.orders.length > 0) {
                        data.orders.forEach(order => {
                            if (!container.querySelector(`.order-item[data-id="${order.id}"]`)) {
                                const orderHtml = currentTabType === 'buyer' ? createBuyerOrderHtml(order) : createSellerOrderHtml(order);
                                container.insertAdjacentHTML('beforeend', orderHtml);
                            }
                        });
                        
                        config.page = data.current_page;
                        config.loadedCount = data.loaded_count;
                        config.hasMore = data.has_more;
                        
                        document.getElementById(`${currentTabType}Tab`).setAttribute('data-page', config.page);
                        document.getElementById(`${currentTabType}Tab`).setAttribute('data-has-more', config.hasMore ? 'true' : 'false');
                        
                        updateTabCount(currentTabType);
                    } else {
                        config.hasMore = false;
                    }
                } else {
                    config.hasMore = false;
                }
                
                const noMoreOrders = document.getElementById('noMoreOrders');
                if (!config.hasMore) {
                    noMoreOrders.classList.remove('hidden');
                } else {
                    noMoreOrders.classList.add('hidden');
                }
                
            } catch (error) {
                console.error('Error loading more orders:', error);
                showMessage('Error', 'Failed to load more orders', 'error');
            } finally {
                config.isLoading = false;
                isLoading = false;
                loader.classList.add('hidden');
            }
        }
        
        // ============================================
        // INFINITE SCROLL EVENT HANDLER
        // ============================================
        function handleInfiniteScroll() {
            const mainContent = document.getElementById('mainContent');
            
            if (!mainContent) return;
            
            const scrollTop = mainContent.scrollTop;
            const scrollHeight = mainContent.scrollHeight;
            const clientHeight = mainContent.clientHeight;
            
            // Load more when 80% from the bottom
            if (scrollTop + clientHeight >= scrollHeight - 200) {
                loadMoreOrders();
            }
        }
        
        // ============================================
        // POLLING FOR NEW ORDERS
        // ============================================
        async function checkForNewOrders() {
            if (!isPolling) return;
            
            try {
                const response = await fetch(`b_orders.php?action=check_new_orders&last_buyer_id=${lastBuyerId}&last_seller_id=${lastSellerId}`);
                const data = await response.json();
                
                if (data.success) {
                    const freshNewBuyerOrders = data.new_buyer_orders || [];
                    const freshNewSellerOrders = data.new_seller_orders || [];
                    
                    freshNewBuyerOrders.forEach(order => {
                        if (!newBuyerOrders.some(existing => existing.id === order.id)) {
                            newBuyerOrders.push(order);
                        }
                    });
                    
                    freshNewSellerOrders.forEach(order => {
                        if (!newSellerOrders.some(existing => existing.id === order.id)) {
                            newSellerOrders.push(order);
                        }
                    });
                    
                    newBuyerCount = newBuyerOrders.length;
                    newSellerCount = newSellerOrders.length;
                    
                    saveBadgeCounts();
                    
                    if (freshNewBuyerOrders.length > 0) {
                        lastBuyerId = Math.max(lastBuyerId, freshNewBuyerOrders[0].id);
                    }
                    if (freshNewSellerOrders.length > 0) {
                        lastSellerId = Math.max(lastSellerId, freshNewSellerOrders[0].id);
                    }
                    
                    updateNewOrdersBadges();
                    
                    if (currentTabType === 'buyer' && newBuyerOrders.length > 0) {
                        addNewBuyerOrders();
                    } else if (currentTabType === 'seller' && newSellerOrders.length > 0) {
                        addNewSellerOrders();
                    }
                }
            } catch (error) {
                console.error('Error checking for new orders:', error);
            }
        }
        
        function startPolling() {
            setInterval(checkForNewOrders, pollingInterval);
        }
        
        // ============================================
        // INITIALIZE BADGES
        // ============================================
        function initializeBadges() {
            updateNewOrdersBadges();
        }
        
        // ============================================
        // SEARCH ORDERS
        // ============================================
        async function searchOrders() {
            const searchId = document.getElementById('searchInput').value.trim();
            const searchType = document.getElementById('searchType').value;
            const searchResults = document.getElementById('searchResults');
            
            if (!searchId) {
                showMessage('Error', 'Please enter an order ID to search', 'error');
                return;
            }
            
            searchResults.innerHTML = '<div style="text-align: center; padding: 20px;"><span class="loading-spinner"></span> Searching...</div>';
            searchResults.classList.add('active');
            
            try {
                const response = await fetch(`b_orders.php?action=search_orders&search_id=${searchId}&search_type=${searchType}`);
                const data = await response.json();
                
                if (data.success) {
                    const order = data.order;
                    const type = data.type;
                    
                    const orderHtml = `
                        <div class="search-result-item" onclick="${type === 'buyer' ? 'viewBuyerOrder' : 'viewSellerOrder'}(${order.id})">
                            <div class="order-header">
                                <div class="order-id">${type === 'buyer' ? 'Buy' : 'Sell'} Order #${order.id}</div>
                                <span class="order-status status-${order.status}">
                                    ${order.status ? order.status.replace('_', ' ') : ''}
                                </span>
                            </div>
                            <div class="order-user">
                                <div class="user-avatar">
                                    ${order.profile_url ? 
                                        `<img src="${order.profile_url}" alt="${order.first_name} ${order.last_name}">` : 
                                        (order.first_name ? order.first_name[0].toUpperCase() : '') + (order.last_name ? order.last_name[0].toUpperCase() : '')
                                    }
                                </div>
                                <div class="user-info">
                                    <h4>${order.first_name || ''} ${order.last_name || ''}</h4>
                                    <p>${order.email || ''}</p>
                                </div>
                            </div>
                            <div class="order-details">
                                <div class="order-info-item">
                                    <label>Amount</label>
                                    <span>₦${parseFloat(order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>Remaining</label>
                                    <span>₦${parseFloat(order.amount_remaining || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>${type === 'buyer' ? 'Interest' : 'Match Status'}</label>
                                    <span>${type === 'buyer' ? '₦' + parseFloat(order.interest || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2}) : order.match_status || ''}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>Status</label>
                                    <span>${order.status ? order.status.replace('_', ' ') : ''}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>Created</label>
                                    <span>${order.created_at ? new Date(order.created_at).toLocaleDateString('en-US', {month: 'short', day: 'numeric', year: 'numeric'}) : ''}</span>
                                </div>
                            </div>
                            ${type === 'buyer' && order.status === 'pending' ? `
                                <div class="order-actions">
                                    <button class="action-btn refund" onclick="event.stopPropagation(); showRefundConfirmation(${order.id}, '${order.first_name || ''}')">
                                        <i class="fas fa-undo"></i> Refund
                                    </button>
                                </div>
                            ` : ''}
                            ${type === 'seller' && order.match_status === 'pending' ? `
                                <div class="order-actions">
                                    ${order.status === 'pause' ? `
                                        <button class="action-btn resume" onclick="event.stopPropagation(); showToggleConfirmation(${order.id}, 'resume')">
                                            <i class="fas fa-play"></i> Resume
                                        </button>
                                    ` : `
                                        <button class="action-btn pause" onclick="event.stopPropagation(); showToggleConfirmation(${order.id}, 'pause')">
                                            <i class="fas fa-pause"></i> Pause
                                        </button>
                                    `}
                                </div>
                            ` : ''}
                        </div>
                    `;
                    
                    searchResults.innerHTML = orderHtml;
                } else {
                    searchResults.innerHTML = `
                        <div class="no-results">
                            <i class="fas fa-search"></i>
                            <h4>No Orders Found</h4>
                            <p>${data.message || 'Order not found'}</p>
                        </div>
                    `;
                }
            } catch (error) {
                console.error('Error searching orders:', error);
                searchResults.innerHTML = `
                    <div class="no-results">
                        <i class="fas fa-exclamation-triangle"></i>
                        <h4>Search Error</h4>
                        <p>An error occurred while searching. Please try again.</p>
                    </div>
                `;
            }
        }
        
        // ============================================
        // TAB SWITCHING
        // ============================================
        function switchTab(tab) {
            document.querySelectorAll('.order-tab').forEach(t => t.classList.remove('active'));
            tab.classList.add('active');
            
            currentTabType = tab.getAttribute('data-type');
            
            document.getElementById('buyerOrders').style.display = currentTabType === 'buyer' ? 'block' : 'none';
            document.getElementById('sellerOrders').style.display = currentTabType === 'seller' ? 'block' : 'none';
            
            const config = INFINITE_SCROLL_CONFIG[currentTabType];
            const noMoreOrders = document.getElementById('noMoreOrders');
            
            if (config.hasMore) {
                noMoreOrders.classList.add('hidden');
            } else {
                noMoreOrders.classList.remove('hidden');
            }
            
            if (currentTabType === 'buyer' && newBuyerOrders.length > 0) {
                addNewBuyerOrders();
            } else if (currentTabType === 'seller' && newSellerOrders.length > 0) {
                addNewSellerOrders();
            }
        }
        
        // ============================================
        // VIEW ORDER DETAILS
        // ============================================
        async function viewBuyerOrder(orderId) {
            try {
                const response = await fetch(`b_orders.php?action=get_buyer_order_details&order_id=${orderId}`);
                const data = await response.json();
                
                if (data.success) {
                    const modalContent = document.getElementById('buyerOrderDetails');
                    
                    let html = `
                        <div class="profile-header">
                            <div class="profile-avatar-container">
                                <div class="profile-avatar">
                                    ${data.user.profile_url ? 
                                        `<img src="${data.user.profile_url}" alt="${data.user.first_name} ${data.user.last_name}">` : 
                                        data.user.initials
                                    }
                                </div>
                                ${data.user.verification === 'verified' ? `
                                    <div class="profile-verified-badge">
                                        <i class="fas fa-check"></i>
                                    </div>
                                ` : ''}
                            </div>
                            
                            <div class="profile-info">
                                <div class="profile-name">
                                    <h3>${data.user.first_name || ''} ${data.user.last_name || ''}</h3>
                                    <span class="profile-status status-${data.user.account_status || 'active'}">
                                        ${data.user.account_status || 'active'}
                                    </span>
                                </div>
                                
                                <div class="profile-details">
                                    <div class="profile-detail-item">
                                        <label>Username</label>
                                        <span>${data.user.username || ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Email</label>
                                        <span>
                                            ${data.user.email || ''}
                                            ${data.user.email_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}
                                        </span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Phone Number</label>
                                        <span>
                                            ${data.user.phone_number || ''}
                                            ${data.user.phone_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}
                                        </span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Date of Birth</label>
                                        <span>${data.user.dob ? new Date(data.user.dob).toLocaleDateString('en-US', {month: 'long', day: 'numeric', year: 'numeric'}) : ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Country</label>
                                        <span>${data.user.country || ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Joined</label>
                                        <span>${data.user.created_at ? new Date(data.user.created_at).toLocaleDateString('en-US', {month: 'long', day: 'numeric', year: 'numeric'}) : ''}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="order-detail-group">
                            <h4>Buyer Order Information</h4>
                            <div class="order-detail-row">
                                <div class="order-detail-item">
                                    <label>Buy Order ID</label>
                                    <span>#${data.order.id || ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Amount</label>
                                    <span>₦${parseFloat(data.order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Remaining Amount</label>
                                    <span>₦${parseFloat(data.order.amount_remaining || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Interest</label>
                                    <span>₦${parseFloat(data.order.interest || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                            </div>
                            <div class="order-detail-row">
                                <div class="order-detail-item">
                                    <label>Status</label>
                                    <span>${data.order.status ? data.order.status.replace('_', ' ') : ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Match Status</label>
                                    <span>${data.order.match_status || ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Created At</label>
                                    <span>${data.order.created_at ? new Date(data.order.created_at).toLocaleString() : ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Updated At</label>
                                    <span>${data.order.updated_at ? new Date(data.order.updated_at).toLocaleString() : ''}</span>
                                </div>
                            </div>
                        </div>
                    `;
                    
                    if (data.orders && data.orders.length > 0) {
                        html += `<div class="order-detail-group">
                            <h4>Matched Orders</h4>`;
                        
                        data.orders.forEach(order => {
                            html += `
                                <div class="bank-details" style="margin-bottom: 20px;">
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <label>Order ID</label>
                                            <span>#${order.order_id || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Status</label>
                                            <span>${order.status ? order.status.replace('_', ' ') : ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Payer Status</label>
                                            <span>${order.payer_status || ''}</span>
                                        </div>
                                    </div>
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <label>Seller Name</label>
                                            <span>${order.seller_name || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Seller Email</label>
                                            <span>${order.seller_email || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Seller Phone</label>
                                            <span>${order.seller_phone || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Created At</label>
                                            <span>${order.created_at ? new Date(order.created_at).toLocaleString() : ''}</span>
                                        </div>
                                    </div>`;
                            
                            if (order.seller_bank) {
                                html += `
                                    <div class="order-detail-group" style="margin-top: 15px;">
                                        <h5>Seller Bank Details</h5>
                                        <div class="order-detail-row">
                                            <div class="order-detail-item">
                                                <label>Bank Name</label>
                                                <span>${order.seller_bank.bank_name || ''}</span>
                                            </div>
                                            <div class="order-detail-item">
                                                <label>Account Number</label>
                                                <span>${order.seller_bank.account_number || ''}</span>
                                            </div>
                                            <div class="order-detail-item">
                                                <label>Account Name</label>
                                                <span>${order.seller_bank.account_name || ''}</span>
                                            </div>
                                        </div>
                                    </div>`;
                            } else {
                                html += `
                                    <div class="order-detail-group" style="margin-top: 15px;">
                                        <h5>Seller Bank Details</h5>
                                        <div class="order-detail-row">
                                            <div class="order-detail-item">
                                                <span style="color: var(--gray); font-style: italic;">No bank details available</span>
                                            </div>
                                        </div>
                                    </div>`;
                            }
                            
                            if (order.payment_proof) {
                                html += `
                                    <div class="order-detail-row" style="margin-top: 10px;">
                                        <div class="order-detail-item">
                                            <button class="modal-btn primary" onclick="viewReceipt('${order.payment_proof}')" style="padding: 8px 16px; font-size: 12px; margin-top: 10px;">
                                                <i class="fas fa-eye"></i> View Receipt
                                            </button>
                                        </div>
                                    </div>`;
                            }
                            
                            if (order.status === 'not approved' || order.status === 'pending') {
                                html += `
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <button class="modal-btn success" onclick="approveBuyerPayment(${order.id || ''}, ${order.buyer_id || ''}, ${order.seller_id || ''})" style="padding: 8px 16px; font-size: 12px; margin-top: 10px;" id="approve-btn-${order.id || ''}">
                                                <i class="fas fa-check"></i> Approve Payment
                                            </button>
                                        </div>
                                    </div>`;
                            }
                            
                            html += `</div>`;
                        });
                        
                        html += `</div>`;
                    } else {
                        html += `
                            <div class="order-detail-group">
                                <h4>Matched Orders</h4>
                                <div class="empty-state" style="padding: 20px;">
                                    <i class="fas fa-shopping-cart"></i>
                                    <h3>No Matched Orders</h3>
                                    <p>There are no matched orders for this buyer order</p>
                                </div>
                            </div>`;
                    }
                    
                    modalContent.innerHTML = html;
                    document.getElementById('buyerOrderModal').classList.add('active');
                    
                    const approveBtn = document.getElementById('approvePaymentBtn');
                    if (data.orders && data.orders.some(order => order.status === 'not approved' || order.status === 'pending')) {
                        approveBtn.style.display = 'flex';
                    } else {
                        approveBtn.style.display = 'none';
                    }
                } else {
                    showMessage('Error', data.message || 'Failed to load order details', 'error');
                }
            } catch (error) {
                console.error('Error loading buyer order:', error);
                showMessage('Error', 'Error loading order details', 'error');
            }
        }
        
        async function viewSellerOrder(orderId) {
            try {
                const response = await fetch(`b_orders.php?action=get_seller_order_details&order_id=${orderId}`);
                const data = await response.json();
                
                if (data.success) {
                    const modalContent = document.getElementById('sellerOrderDetails');
                    
                    let html = `
                        <div class="profile-header">
                            <div class="profile-avatar-container">
                                <div class="profile-avatar">
                                    ${data.user.profile_url ? 
                                        `<img src="${data.user.profile_url}" alt="${data.user.first_name} ${data.user.last_name}">` : 
                                        data.user.initials
                                    }
                                </div>
                                ${data.user.verification === 'verified' ? `
                                    <div class="profile-verified-badge">
                                        <i class="fas fa-check"></i>
                                    </div>
                                ` : ''}
                            </div>
                            
                            <div class="profile-info">
                                <div class="profile-name">
                                    <h3>${data.user.first_name || ''} ${data.user.last_name || ''}</h3>
                                    <span class="profile-status status-${data.user.account_status || 'active'}">
                                        ${data.user.account_status || 'active'}
                                    </span>
                                </div>
                                
                                <div class="profile-details">
                                    <div class="profile-detail-item">
                                        <label>Username</label>
                                        <span>${data.user.username || ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Email</label>
                                        <span>
                                            ${data.user.email || ''}
                                            ${data.user.email_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}
                                        </span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Phone Number</label>
                                        <span>
                                            ${data.user.phone_number || ''}
                                            ${data.user.phone_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}
                                        </span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Date of Birth</label>
                                        <span>${data.user.dob ? new Date(data.user.dob).toLocaleDateString('en-US', {month: 'long', day: 'numeric', year: 'numeric'}) : ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Country</label>
                                        <span>${data.user.country || ''}</span>
                                    </div>
                                    <div class="profile-detail-item">
                                        <label>Joined</label>
                                        <span>${data.user.created_at ? new Date(data.user.created_at).toLocaleDateString('en-US', {month: 'long', day: 'numeric', year: 'numeric'}) : ''}</span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="order-detail-group">
                            <h4>Order Information</h4>
                            <div class="order-detail-row">
                                <div class="order-detail-item">
                                    <label>Order ID</label>
                                    <span>#${data.order.id || ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Amount</label>
                                    <span>₦${parseFloat(data.order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Remaining Amount</label>
                                    <span>₦${parseFloat(data.order.amount_remaining || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Status</label>
                                    <span>${data.order.status ? data.order.status.replace('_', ' ') : ''}</span>
                                </div>
                            </div>
                            <div class="order-detail-row">
                                <div class="order-detail-item">
                                    <label>Match Status</label>
                                    <span>${data.order.match_status || ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Created At</label>
                                    <span>${data.order.created_at ? new Date(data.order.created_at).toLocaleString() : ''}</span>
                                </div>
                                <div class="order-detail-item">
                                    <label>Updated At</label>
                                    <span>${data.order.updated_at ? new Date(data.order.updated_at).toLocaleString() : ''}</span>
                                </div>
                            </div>
                        </div>
                    `;
                    
                    if (data.orders && data.orders.length > 0) {
                        html += `<div class="order-detail-group">
                            <h4>Matched Orders</h4>`;
                        
                        data.orders.forEach(order => {
                            html += `
                                <div class="bank-details" style="margin-bottom: 15px;">
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <label>Order ID</label>
                                            <span>#${order.id || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount || 0).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Status</label>
                                            <span>${order.status ? order.status.replace('_', ' ') : ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Payer Status</label>
                                            <span>${order.payer_status || ''}</span>
                                        </div>
                                    </div>
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <label>Buyer Name</label>
                                            <span>${order.buyer_name || ''}</span>
                                        </div>
                                        <div class="order-detail-item">
                                            <label>Created At</label>
                                            <span>${order.created_at ? new Date(order.created_at).toLocaleString() : ''}</span>
                                        </div>
                                    </div>`;
                            
                            if (order.payment_proof) {
                                html += `
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <button class="modal-btn primary" onclick="viewReceipt('${order.payment_proof}')" style="padding: 8px 16px; font-size: 12px; margin-top: 10px;">
                                                <i class="fas fa-eye"></i> View Receipt
                                            </button>
                                        </div>
                                    </div>`;
                            }
                            
                            if (order.status === 'not approved' || order.status === 'pending') {
                                html += `
                                    <div class="order-detail-row">
                                        <div class="order-detail-item">
                                            <button class="modal-btn success" onclick="approveSellerPayment(${order.id || ''}, ${order.buyer_id || ''}, ${order.seller_id || ''})" style="padding: 8px 16px; font-size: 12px; margin-top: 10px;" id="approve-btn-${order.id || ''}">
                                                <i class="fas fa-check"></i> Approve Payment
                                            </button>
                                        </div>
                                    </div>`;
                            }
                            
                            html += `</div>`;
                        });
                        
                        html += `</div>`;
                    }
                    
                    modalContent.innerHTML = html;
                    document.getElementById('sellerOrderModal').classList.add('active');
                } else {
                    showMessage('Error', data.message || 'Failed to load order details', 'error');
                }
            } catch (error) {
                console.error('Error loading seller order:', error);
                showMessage('Error', 'Error loading order details', 'error');
            }
        }
        
        function viewReceipt(imageUrl) {
            document.getElementById('receiptImage').src = imageUrl;
            document.getElementById('receiptModal').classList.add('active');
        }
        
        // ============================================
        // ORDER ACTIONS
        // ============================================
        async function refundBuyerOrder(orderId, userName) {
            const refundBtn = event?.target?.closest?.('.action-btn');
            let originalText = '';
            
            if (refundBtn) {
                originalText = refundBtn.innerHTML;
                refundBtn.innerHTML = '<span class="loading-spinner"></span> Processing...';
                refundBtn.disabled = true;
            }
            
            try {
                const response = await fetch('refund.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'refund_buyer_order',
                        order_id: orderId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    
                    const orderItem = document.querySelector(`.order-item[data-id="${orderId}"]`);
                    if (orderItem) {
                        orderItem.style.opacity = '0.5';
                        setTimeout(() => {
                            orderItem.remove();
                            
                            const ordersContainer = document.getElementById('buyerOrders');
                            if (ordersContainer.querySelectorAll('.order-item').length === 0) {
                                ordersContainer.innerHTML = `
                                    <div class="empty-state">
                                        <i class="fas fa-shopping-cart"></i>
                                        <h3>No Buyer Orders</h3>
                                        <p>There are no pending buyer orders</p>
                                    </div>
                                `;
                            }
                            
                            updateTabCount('buyer');
                            
                            INFINITE_SCROLL_CONFIG.buyer.loadedCount = ordersContainer.querySelectorAll('.order-item').length;
                            INFINITE_SCROLL_CONFIG.buyer.hasMore = INFINITE_SCROLL_CONFIG.buyer.loadedCount < INFINITE_SCROLL_CONFIG.buyer.totalItems;
                        }, 1000);
                    }
                } else {
                    showMessage('Error', data.message, 'error');
                    if (refundBtn) {
                        refundBtn.innerHTML = originalText;
                        refundBtn.disabled = false;
                    }
                }
            } catch (error) {
                console.error('Error refunding order:', error);
                showMessage('Error', 'Error processing refund', 'error');
                if (refundBtn) {
                    refundBtn.innerHTML = originalText;
                    refundBtn.disabled = false;
                }
            }
        }
        
        async function toggleSellerOrderStatus(orderId, action) {
            const actionBtn = event?.target?.closest?.('.action-btn');
            let originalText = '';
            
            if (actionBtn) {
                originalText = actionBtn.innerHTML;
                actionBtn.innerHTML = '<span class="loading-spinner"></span> Processing...';
                actionBtn.disabled = true;
            }
            
            try {
                const response = await fetch('b_orders.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'toggle_seller_order_status',
                        order_id: orderId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    
                    const orderItem = document.querySelector(`.order-item[data-id="${orderId}"]`);
                    if (orderItem) {
                        const statusBadge = orderItem.querySelector('.order-status');
                        const actionContainer = orderItem.querySelector('.order-actions');
                        
                        if (data.new_status === 'pause') {
                            if (actionContainer) {
                                actionContainer.innerHTML = `
                                    <button class="action-btn resume" onclick="event.stopPropagation(); showToggleConfirmation(${orderId}, 'resume')">
                                        <i class="fas fa-play"></i> Resume
                                    </button>
                                `;
                            }
                            statusBadge.className = 'order-status status-pause';
                            statusBadge.textContent = 'Pause';
                        } else {
                            if (actionContainer) {
                                actionContainer.innerHTML = `
                                    <button class="action-btn pause" onclick="event.stopPropagation(); showToggleConfirmation(${orderId}, 'pause')">
                                        <i class="fas fa-pause"></i> Pause
                                    </button>
                                `;
                            }
                            statusBadge.className = 'order-status status-in_progress';
                            statusBadge.textContent = 'In Progress';
                        }
                    }
                } else {
                    showMessage('Error', data.message, 'error');
                    if (actionBtn) {
                        actionBtn.innerHTML = originalText;
                        actionBtn.disabled = false;
                    }
                }
            } catch (error) {
                console.error('Error toggling seller order status:', error);
                showMessage('Error', 'Error processing request', 'error');
                if (actionBtn) {
                    actionBtn.innerHTML = originalText;
                    actionBtn.disabled = false;
                }
            }
        }
        
        async function approveBuyerPayment(orderId, buyerId, sellerId) {
            const approveBtn = document.getElementById(`approve-btn-${orderId}`);
            let originalText = '';
            
            if (approveBtn) {
                originalText = approveBtn.innerHTML;
                approveBtn.innerHTML = '<span class="loading-spinner"></span> Approving...';
                approveBtn.disabled = true;
            }
            
            try {
                const response = await fetch('b_orders.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'approve_payment',
                        order_id: orderId,
                        buyer_id: buyerId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    document.getElementById('buyerOrderModal').classList.remove('active');
                    setTimeout(() => {
                        viewBuyerOrder(orderId);
                    }, 2000);
                } else {
                    showMessage('Error', data.message, 'error');
                    if (approveBtn) {
                        approveBtn.innerHTML = originalText;
                        approveBtn.disabled = false;
                    }
                }
            } catch (error) {
                console.error('Error approving payment:', error);
                showMessage('Error', 'Error approving payment', 'error');
                if (approveBtn) {
                    approveBtn.innerHTML = originalText;
                    approveBtn.disabled = false;
                }
            }
        }
        
        async function approveSellerPayment(orderId, buyerId, sellerId) {
            const approveBtn = document.getElementById(`approve-btn-${orderId}`);
            let originalText = '';
            
            if (approveBtn) {
                originalText = approveBtn.innerHTML;
                approveBtn.innerHTML = '<span class="loading-spinner"></span> Approving...';
                approveBtn.disabled = true;
            }
            
            try {
                const response = await fetch('b_orders.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'approve_payment',
                        order_id: orderId,
                        buyer_id: buyerId
                    })
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showMessage('Success', data.message, 'success');
                    document.getElementById('sellerOrderModal').classList.remove('active');
                    setTimeout(() => {
                        viewSellerOrder(orderId);
                    }, 2000);
                } else {
                    showMessage('Error', data.message, 'error');
                    if (approveBtn) {
                        approveBtn.innerHTML = originalText;
                        approveBtn.disabled = false;
                    }
                }
            } catch (error) {
                console.error('Error approving payment:', error);
                showMessage('Error', 'Error approving payment', 'error');
                if (approveBtn) {
                    approveBtn.innerHTML = originalText;
                    approveBtn.disabled = false;
                }
            }
        }
        
        // ============================================
        // EVENT LISTENERS
        // ============================================
        document.addEventListener('DOMContentLoaded', () => {
            startPolling();
            initializeBadges();
            
            const mainContent = document.getElementById('mainContent');
            mainContent.addEventListener('scroll', handleHeaderScroll);
            mainContent.addEventListener('scroll', handleInfiniteScroll);
            
            document.querySelectorAll('.order-tab').forEach(tab => {
                tab.addEventListener('click', function() {
                    switchTab(this);
                });
            });
            
            document.getElementById('searchForm').addEventListener('submit', function(e) {
                e.preventDefault();
                searchOrders();
            });
            
            document.getElementById('closeBuyerModalBtn').addEventListener('click', () => {
                document.getElementById('buyerOrderModal').classList.remove('active');
            });
            
            document.getElementById('closeSellerModalBtn').addEventListener('click', () => {
                document.getElementById('sellerOrderModal').classList.remove('active');
            });
            
            document.getElementById('closeReceiptBtn').addEventListener('click', () => {
                document.getElementById('receiptModal').classList.remove('active');
            });
            
            document.getElementById('messageOkBtn').addEventListener('click', () => {
                document.getElementById('messageModal').classList.remove('active');
            });
            
            document.getElementById('confirmationCancelBtn').addEventListener('click', () => {
                document.getElementById('confirmationModal').classList.remove('active');
            });
            
            document.querySelectorAll('.modal-overlay').forEach(modal => {
                modal.addEventListener('click', (e) => {
                    if (e.target === modal) {
                        modal.classList.remove('active');
                    }
                });
            });
        });
    </script>
</body>
</html>