<?php
// profile.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Check if user_id is provided
if (!isset($_GET['id']) || !is_numeric($_GET['id'])) {
    header('Location: users.php');
    exit;
}

$user_id = (int)$_GET['id'];

// Include database connection
require_once 'php/db.php';

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}

// Fetch user data
try {
    // Get user basic info with KYC
    $stmt = $pdo->prepare("
        SELECT 
            u.*,
            COALESCE(k.email_status, 'not verified') as email_verified,
            COALESCE(k.phone_status, 'not verified') as phone_verified,
            COALESCE(sd.status, 'not paid') as deposit_status,
            COALESCE(sd.amount, 0) as deposit_amount
        FROM users u
        LEFT JOIN basic_kyc k ON u.user_id = k.user_id
        LEFT JOIN security_deposit sd ON u.user_id = sd.user_id
        WHERE u.user_id = :user_id
    ");
    $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $stmt->execute();
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        header('Location: users.php');
        exit;
    }
    
    // Get bank accounts
    $bankStmt = $pdo->prepare("SELECT * FROM bank_accounts WHERE user_id = :user_id ORDER BY created_at DESC");
    $bankStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $bankStmt->execute();
    $bank_accounts = $bankStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get buyer orders (limited to 5)
    $buyerStmt = $pdo->prepare("SELECT * FROM buyer_orders WHERE user_id = :user_id ORDER BY created_at DESC LIMIT 5");
    $buyerStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $buyerStmt->execute();
    $buyer_orders = $buyerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get seller orders (limited to 5)
    $sellerStmt = $pdo->prepare("SELECT * FROM seller_orders WHERE user_id = :user_id ORDER BY created_at DESC LIMIT 5");
    $sellerStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $sellerStmt->execute();
    $seller_orders = $sellerStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get total counts for infinite scroll
    $buyerCountStmt = $pdo->prepare("SELECT COUNT(*) as total FROM buyer_orders WHERE user_id = :user_id");
    $buyerCountStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $buyerCountStmt->execute();
    $buyer_total = $buyerCountStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    $sellerCountStmt = $pdo->prepare("SELECT COUNT(*) as total FROM seller_orders WHERE user_id = :user_id");
    $sellerCountStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $sellerCountStmt->execute();
    $seller_total = $sellerCountStmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Get referral link and balance
    $referralStmt = $pdo->prepare("SELECT * FROM referral_link WHERE user_id = :user_id");
    $referralStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $referralStmt->execute();
    $referral_link = $referralStmt->fetch(PDO::FETCH_ASSOC);
    
    // Get referrals (users referred by this user)
    $referralsStmt = $pdo->prepare("
        SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, u.created_at 
        FROM referrals r 
        JOIN users u ON r.referred_user_id = u.user_id 
        WHERE r.referrer_user_id = :user_id 
        ORDER BY r.created_at DESC
    ");
    $referralsStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $referralsStmt->execute();
    $referrals = $referralsStmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Get testimonies
    $testimonyStmt = $pdo->prepare("
        SELECT t.*, u.username, u.first_name, u.last_name, o.buyer_id, o.seller_id
        FROM testimony t
        JOIN orders o ON t.order_id = o.id
        JOIN users u ON u.user_id = :user_id
        WHERE o.seller_id IN (SELECT id FROM seller_orders WHERE user_id = :user_id)
           OR o.buyer_id IN (SELECT id FROM buyer_orders WHERE user_id = :user_id)
        ORDER BY t.created_at DESC
    ");
    $testimonyStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
    $testimonyStmt->execute();
    $testimonies = $testimonyStmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>User Profile - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .back-btn {
            background: var(--gray-light);
            color: var(--gray);
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            text-decoration: none;
        }
        
        .back-btn:hover {
            background: #d1d5db;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Profile Header */
        .profile-header {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 30px;
        }
        
        .profile-avatar-container {
            position: relative;
        }
        
        .profile-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 36px;
            flex-shrink: 0;
        }
        
        .profile-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .profile-verified-badge {
            position: absolute;
            bottom: 5px;
            right: 5px;
            background: var(--primary);
            color: white;
            width: 30px;
            height: 30px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            border: 3px solid white;
        }
        
        .profile-info {
            flex: 1;
        }
        
        .profile-name {
            display: flex;
            align-items: center;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .profile-name h3 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .profile-status {
            padding: 6px 16px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-active {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-suspended {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-closed {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .profile-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .profile-detail-item {
            display: flex;
            flex-direction: column;
        }
        
        .profile-detail-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .profile-detail-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .verified-icon {
            color: var(--primary);
            font-size: 12px;
        }
        
        .profile-actions {
            display: flex;
            flex-direction: column;
            gap: 10px;
        }
        
        .edit-profile-btn, .suspend-btn, .activate-btn {
            padding: 10px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            width: 100%;
        }
        
        .edit-profile-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-profile-btn:hover {
            background: var(--primary-dark);
        }
        
        .suspend-btn {
            background: var(--warning);
            color: white;
        }
        
        .suspend-btn:hover {
            background: #d97706;
        }
        
        .activate-btn {
            background: var(--success);
            color: white;
        }
        
        .activate-btn:hover {
            background: var(--secondary-dark);
        }
        
        /* Security Deposit Badge */
        .deposit-badge {
            display: inline-flex;
            align-items: center;
            gap: 6px;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            margin-top: 10px;
        }
        
        .deposit-paid {
            background: #d1fae5;
            color: #065f46;
        }
        
        .deposit-not-paid {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .deposit-refunded {
            background: #dbeafe;
            color: #1e40af;
        }
        
        /* Profile Sections */
        .profile-sections {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(400px, 1fr));
            gap: 30px;
            margin-bottom: 30px;
        }
        
        .profile-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        /* Bank Accounts Compact */
        .bank-accounts-compact {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 15px;
        }
        
        .bank-account-compact {
            background: var(--light);
            border-radius: 10px;
            padding: 15px;
            transition: all 0.3s ease;
        }
        
        .bank-account-compact:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .bank-info-compact {
            display: flex;
            flex-direction: column;
            gap: 8px;
        }
        
        .bank-info-row {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .bank-info-label {
            font-size: 12px;
            color: var(--gray);
            font-weight: 500;
        }
        
        .bank-info-value {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .bank-actions {
            display: flex;
            gap: 10px;
            margin-top: 10px;
            padding-top: 10px;
            border-top: 1px solid var(--gray-light);
        }
        
        .edit-bank-btn, .delete-bank-btn {
            flex: 1;
            padding: 8px 12px;
            border-radius: 8px;
            font-size: 12px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 6px;
        }
        
        .edit-bank-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-bank-btn:hover {
            background: var(--primary-dark);
        }
        
        .delete-bank-btn {
            background: var(--danger);
            color: white;
        }
        
        .delete-bank-btn:hover {
            background: #dc2626;
        }
        
        /* Orders Section */
        .orders-section {
            margin-top: 30px;
        }
        
        .orders-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
            padding-bottom: 10px;
        }
        
        .order-tab {
            padding: 10px 20px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .order-tab.active {
            background: var(--primary);
            color: white;
        }
        
        .order-tab:hover:not(.active) {
            background: #d1d5db;
        }
        
        .orders-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .order-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
        }
        
        .order-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .order-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .order-id {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .order-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-pending {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-approved {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-not_approved {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .status-pause {
            background: #d1d5db;
            color: #4b5563;
        }
        
        .status-sold {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .status-in_progress {
            background: #f3e8ff;
            color: #7c3aed;
        }
        
        .order-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .order-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .order-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .order-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .order-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        
        .view-orders-btn, .approve-order-btn, .pause-play-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .view-orders-btn:hover, .approve-order-btn:hover, .pause-play-btn:hover {
            background: var(--primary-dark);
        }
        
        .approve-order-btn {
            background: var(--success);
        }
        
        .approve-order-btn:hover {
            background: var(--secondary-dark);
        }
        
        .pause-play-btn {
            background: var(--warning);
        }
        
        .pause-play-btn:hover {
            background: #d97706;
        }
        
        .load-more-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            margin: 20px auto;
            width: 200px;
        }
        
        .load-more-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }
        
        .load-more-btn.hidden {
            display: none;
        }
        
        /* Testimony Section */
        .testimonies-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .testimony-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
        }
        
        .testimony-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .testimony-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .testimony-id {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .testimony-status {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .testimony-details {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .testimony-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .testimony-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .testimony-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .testimony-actions {
            display: flex;
            gap: 10px;
            justify-content: flex-end;
        }
        
        .view-testimony-btn, .approve-testimony-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .view-testimony-btn {
            background: var(--primary);
            color: white;
        }
        
        .view-testimony-btn:hover {
            background: var(--primary-dark);
        }
        
        .approve-testimony-btn {
            background: var(--success);
            color: white;
        }
        
        .approve-testimony-btn:hover {
            background: var(--secondary-dark);
        }
        
        /* Referral Section */
        .referral-stats {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 20px;
            margin-bottom: 25px;
        }
        
        .referral-stat {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            text-align: center;
        }
        
        .referral-stat h4 {
            font-size: 24px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .referral-stat p {
            font-size: 14px;
            color: var(--gray);
            font-weight: 500;
        }
        
        .referral-link {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 25px;
        }
        
        .referral-link p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 10px;
        }
        
        .referral-code {
            background: white;
            padding: 15px;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            border: 1px solid var(--gray-light);
            word-break: break-all;
        }
        
        .referral-item {
            cursor: pointer;
        }
        
        .referral-user {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .referral-avatar {
            width: 40px;
            height: 40px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 14px;
            flex-shrink: 0;
        }
        
        .referral-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .referral-user-info h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .referral-user-info p {
            font-size: 14px;
            color: var(--gray);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 14px;
        }
        
        /* Modals */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .edit-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .edit-modal {
            transform: translateY(0);
        }
        
        .edit-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid var(--gray-light);
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .form-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .form-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .form-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .form-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .form-btn.secondary:hover {
            background: #d1d5db;
        }
        
        /* Orders Modal */
        .orders-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 800px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .orders-list-modal {
            display: flex;
            flex-direction: column;
            gap: 15px;
            margin-top: 20px;
        }
        
        .order-detail-modal {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            transition: all 0.3s ease;
        }
        
        .order-detail-modal:hover {
            background: #f1f5f9;
        }
        
        .user-link {
            color: var(--primary);
            text-decoration: none;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .user-link:hover {
            text-decoration: underline;
            color: var(--primary-dark);
        }
        
        /* Receipt Modal */
        .receipt-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 600px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .receipt-image {
            width: 100%;
            max-height: 500px;
            object-fit: contain;
            border-radius: 10px;
            margin: 20px 0;
        }
        
        /* Alert Modal */
        .alert-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .alert-modal {
            transform: translateY(0);
        }
        
        .alert-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .alert-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-icon.warning {
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .alert-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .alert-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .alert-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .alert-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .alert-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .alert-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .alert-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .alert-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        /* Confirmation Modal */
        .confirm-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .profile-header {
                flex-direction: column;
                text-align: center;
            }
            
            .profile-actions {
                flex-direction: row;
                width: 100%;
            }
            
            .profile-sections {
                grid-template-columns: 1fr;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .profile-details {
                grid-template-columns: 1fr;
            }
            
            .orders-tabs {
                flex-wrap: wrap;
            }
            
            .edit-modal, .orders-modal, .receipt-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .alert-modal, .confirm-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
        
        @media (max-width: 480px) {
            .profile-actions {
                flex-direction: column;
            }
            
            .bank-actions {
                flex-direction: column;
            }
            
            .form-actions, .alert-actions {
                flex-direction: column;
            }
            
            .form-btn, .alert-btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>User Profile</h2>
                    <p>User ID: #<?php echo $user['user_id']; ?></p>
                </div>
                <div class="header-actions">
                    <a href="users.php" class="back-btn">
                        <i class="fas fa-arrow-left"></i> Back to Users
                    </a>
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Profile Header -->
            <div class="profile-header">
                <div class="profile-avatar-container">
                    <div class="profile-avatar">
                        <?php if ($user['profile_url']): ?>
                            <img src="<?php echo htmlspecialchars($user['profile_url']); ?>" alt="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>">
                        <?php else: ?>
                            <?php echo getUserInitials($user['first_name'] . ' ' . $user['last_name']); ?>
                        <?php endif; ?>
                    </div>
                    <?php if ($user['verification'] === 'verified'): ?>
                        <div class="profile-verified-badge">
                            <i class="fas fa-check"></i>
                        </div>
                    <?php endif; ?>
                </div>
                
                <div class="profile-info">
                    <div class="profile-name">
                        <h3><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h3>
                        <span class="profile-status status-<?php echo $user['account_status']; ?>">
                            <?php echo ucfirst($user['account_status']); ?>
                        </span>
                    </div>
                    
                    <div class="profile-details">
                        <div class="profile-detail-item">
                            <label>Username</label>
                            <span><?php echo htmlspecialchars($user['username']); ?></span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Email</label>
                            <span>
                                <?php echo htmlspecialchars($user['email']); ?>
                                <?php if ($user['email_verified'] === 'verified'): ?>
                                    <i class="fas fa-check-circle verified-icon"></i>
                                <?php endif; ?>
                            </span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Phone Number</label>
                            <span>
                                <?php echo htmlspecialchars($user['phone_number']); ?>
                                <?php if ($user['phone_verified'] === 'verified'): ?>
                                    <i class="fas fa-check-circle verified-icon"></i>
                                <?php endif; ?>
                            </span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Date of Birth</label>
                            <span><?php echo date('F j, Y', strtotime($user['dob'])); ?></span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Country</label>
                            <span><?php echo htmlspecialchars($user['country']); ?></span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Joined</label>
                            <span><?php echo date('F j, Y', strtotime($user['created_at'])); ?></span>
                        </div>
                        <div class="profile-detail-item">
                            <label>Security Deposit</label>
                            <span>₦<?php echo number_format($user['deposit_amount'], 2); ?></span>
                            <?php if ($user['deposit_status'] !== 'not paid'): ?>
                                <span class="deposit-badge deposit-<?php echo str_replace(' ', '-', $user['deposit_status']); ?>">
                                    <?php echo ucfirst($user['deposit_status']); ?>
                                </span>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
                
                <div class="profile-actions">
                    <button class="edit-profile-btn" id="editProfileBtn">
                        <i class="fas fa-edit"></i> Edit Profile
                    </button>
                    <?php if ($user['account_status'] === 'active'): ?>
                        <button class="suspend-btn" onclick="suspendUser(<?php echo $user['user_id']; ?>, '<?php echo htmlspecialchars($user['first_name']); ?>')">
                            <i class="fas fa-user-slash"></i> Suspend
                        </button>
                    <?php elseif ($user['account_status'] === 'suspended'): ?>
                        <button class="activate-btn" onclick="activateUser(<?php echo $user['user_id']; ?>, '<?php echo htmlspecialchars($user['first_name']); ?>')">
                            <i class="fas fa-user-check"></i> Activate
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Bank Accounts Section -->
            <div class="profile-section">
                <div class="section-header">
                    <h3 class="section-title">
                        <i class="fas fa-credit-card"></i>
                        Bank Accounts
                    </h3>
                </div>
                
                <div class="bank-accounts-compact" id="bankAccountsList">
                    <?php if (empty($bank_accounts)): ?>
                        <div class="empty-state">
                            <i class="fas fa-credit-card"></i>
                            <h3>No Bank Accounts</h3>
                            <p>This user hasn't added any bank accounts yet</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($bank_accounts as $account): ?>
                            <div class="bank-account-compact" data-id="<?php echo $account['id']; ?>">
                                <div class="bank-info-compact">
                                    <div class="bank-info-row">
                                        <span class="bank-info-label">Bank:</span>
                                        <span class="bank-info-value"><?php echo htmlspecialchars($account['bank_name']); ?></span>
                                    </div>
                                    <div class="bank-info-row">
                                        <span class="bank-info-label">Account No:</span>
                                        <span class="bank-info-value"><?php echo htmlspecialchars($account['account_number']); ?></span>
                                    </div>
                                    <div class="bank-info-row">
                                        <span class="bank-info-label">Account Name:</span>
                                        <span class="bank-info-value"><?php echo htmlspecialchars($account['account_name']); ?></span>
                                    </div>
                                </div>
                                <div class="bank-actions">
                                    <button class="edit-bank-btn" onclick="editBankAccount(<?php echo $account['id']; ?>)">
                                        <i class="fas fa-edit"></i> Edit
                                    </button>
                                    <button class="delete-bank-btn" onclick="deleteBankAccount(<?php echo $account['id']; ?>)">
                                        <i class="fas fa-trash"></i> Delete
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
           <!-- Testimonies Section -->
<div class="profile-section">
    <div class="section-header">
        <h3 class="section-title">
            <i class="fas fa-comment"></i>
            Testimonies
        </h3>
    </div>
    
    <?php
    // Calculate testimony statistics
    $approved_count = 0;
    $not_approved_count = 0;
    $pending_count = 0;
    $waiting_count = 0;
    $testified_count = 0;
    
    if (!empty($testimonies)) {
        foreach ($testimonies as $testimony) {
            switch ($testimony['status']) {
                case 'approved':
                    $approved_count++;
                    break;
                case 'not approved':
                    $not_approved_count++;
                    break;
                case 'pending':
                    $pending_count++;
                    break;
                case 'waiting':
                    $waiting_count++;
                    break;
            }
            
            if ($testimony['user_status'] === 'I have testified') {
                $testified_count++;
            }
        }
    }
    ?>
    
    <!-- Testimonies Section -->
<div class="profile-section">
    <div class="section-header">
        <h3 class="section-title">
            <i class="fas fa-comment"></i>
            Testimonies
        </h3>
    </div>
    
    <?php
    // Calculate testimony statistics
    $approved_count = 0;
    $not_approved_count = 0;
    $pending_count = 0;
    $waiting_count = 0;
    $testified_count = 0;
    
    if (!empty($testimonies)) {
        foreach ($testimonies as $testimony) {
            switch ($testimony['status']) {
                case 'approved':
                    $approved_count++;
                    break;
                case 'not approved':
                    $not_approved_count++;
                    break;
                case 'pending':
                    $pending_count++;
                    break;
                case 'waiting':
                    $waiting_count++;
                    break;
            }
            
            if ($testimony['user_status'] === 'I have testified') {
                $testified_count++;
            }
        }
    }
    ?>
    
    <!-- Testimony Statistics -->
    <div class="referral-stats" style="margin-bottom: 25px;">
        <div class="referral-stat">
            <h4><?php echo $testified_count; ?></h4>
            <p>Testified</p>
        </div>
        <div class="referral-stat">
            <h4><?php echo $approved_count; ?></h4>
            <p>Approved</p>
        </div>
        <div class="referral-stat">
            <h4><?php echo $not_approved_count; ?></h4>
            <p>Not Approved</p>
        </div>
        <div class="referral-stat">
            <h4><?php echo $pending_count + $waiting_count; ?></h4>
            <p>Pending/Waiting</p>
        </div>
    </div>
    
    <div class="testimonies-list" id="testimoniesList">
        <?php if (empty($testimonies)): ?>
            <div class="empty-state">
                <i class="fas fa-comment"></i>
                <h3>No Testimonies</h3>
                <p>This user hasn't submitted any testimonies yet</p>
            </div>
        <?php else: ?>
            <?php 
            $displayed_testimonies = 0;
            foreach ($testimonies as $testimony): 
                // Only show testimonies where user_status is "I have testified" AND status is NOT "approved"
                if ($testimony['user_status'] === 'I have testified' && $testimony['status'] !== 'approved'):
                    $displayed_testimonies++;
            ?>
                    <div class="testimony-item" data-id="<?php echo $testimony['id']; ?>">
                        <div class="testimony-header">
                            <div class="testimony-id">Testimony #<?php echo $testimony['id']; ?></div>
                            <span class="testimony-status status-<?php echo $testimony['status']; ?>">
                                <?php echo ucfirst($testimony['status']); ?>
                            </span>
                        </div>
                        <div class="testimony-details">
                            <div class="testimony-info-item">
                                <label>User Status</label>
                                <span><?php echo str_replace('_', ' ', ucfirst($testimony['user_status'])); ?></span>
                            </div>
                            <div class="testimony-info-item">
                                <label>Channel</label>
                                <span><?php echo ucfirst($testimony['channel']); ?></span>
                            </div>
                            <div class="testimony-info-item">
                                <label>Status</label>
                                <span><?php echo ucfirst($testimony['status']); ?></span>
                            </div>
                            <div class="testimony-info-item">
                                <label>Created At</label>
                                <span><?php echo date('F j, Y H:i:s', strtotime($testimony['created_at'])); ?></span>
                            </div>
                        </div>
                        <div class="testimony-actions">
                            <?php if ($testimony['testimony_url']): ?>
                                <button class="view-testimony-btn" onclick="viewTestimony('<?php echo htmlspecialchars($testimony['testimony_url']); ?>')">
                                    <i class="fas fa-eye"></i> View
                                </button>
                            <?php endif; ?>
                            <?php if ($testimony['status'] === 'pending' || $testimony['status'] === 'waiting'): ?>
                                <button class="approve-testimony-btn" onclick="approveTestimony(<?php echo $testimony['id']; ?>)">
                                    <i class="fas fa-check"></i> Approve
                                </button>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endif; ?>
            <?php endforeach; ?>
            
            <?php if ($displayed_testimonies === 0 && !empty($testimonies)): ?>
                <div class="empty-state">
                    <i class="fas fa-check-circle" style="color: var(--success);"></i>
                    <h3>All Testimonies Approved</h3>
                    <p>All testimonies for this user have been approved</p>
                </div>
            <?php endif; ?>
        <?php endif; ?>
    </div>
</div> 
            <!-- Referrals Section -->
            <div class="profile-section">
                <div class="section-header">
                    <h3 class="section-title">
                        <i class="fas fa-users"></i>
                        Referrals
                    </h3>
                </div>
                
                <div class="referral-stats">
                    <div class="referral-stat">
                        <h4><?php echo $referral_link ? $referral_link['clicks'] : 0; ?></h4>
                        <p>Referral Clicks</p>
                    </div>
                    <div class="referral-stat">
                        <h4><?php echo $referral_link ? $referral_link['total_register'] : 0; ?></h4>
                        <p>Total Registrations</p>
                    </div>
                    <div class="referral-stat">
                        <h4>₦<?php echo $referral_link ? number_format($referral_link['balance'], 2) : '0.00'; ?></h4>
                        <p>Referral Balance</p>
                    </div>
                </div>
                
                <?php if ($referral_link): ?>
                    <div class="referral-link">
                        <p>Referral Link:</p>
                        <div class="referral-code">
                            <?php echo htmlspecialchars($referral_link['referral_code']); ?>
                        </div>
                    </div>
                <?php endif; ?>
                
                <h4 style="margin-bottom: 15px; color: var(--dark);">Referred Users (<?php echo count($referrals); ?>)</h4>
                <div class="referrals-list">
                    <?php if (empty($referrals)): ?>
                        <div class="empty-state">
                            <i class="fas fa-users"></i>
                            <h3>No Referrals</h3>
                            <p>This user hasn't referred anyone yet</p>
                        </div>
                    <?php else: ?>
                        <?php foreach ($referrals as $referral): ?>
                            <div class="referral-item" onclick="viewUserProfile(<?php echo $referral['user_id']; ?>)">
                                <div class="referral-user">
                                    <div class="referral-avatar">
                                        <?php echo getUserInitials($referral['first_name'] . ' ' . $referral['last_name']); ?>
                                    </div>
                                    <div class="referral-user-info">
                                        <h4><?php echo htmlspecialchars($referral['first_name'] . ' ' . $referral['last_name']); ?></h4>
                                        <p><?php echo htmlspecialchars($referral['email']); ?></p>
                                        <p>Joined: <?php echo date('F j, Y', strtotime($referral['created_at'])); ?></p>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Orders Section -->
            <div class="profile-section orders-section">
                <div class="section-header">
                    <h3 class="section-title">
                        <i class="fas fa-shopping-cart"></i>
                        Orders
                    </h3>
                </div>
                
                <div class="orders-tabs">
                    <button class="order-tab active" data-type="buyer" data-offset="5" data-total="<?php echo $buyer_total; ?>">
                        Buyer Orders (<?php echo min(5, $buyer_total); ?>/<?php echo $buyer_total; ?>)
                    </button>
                    <button class="order-tab" data-type="seller" data-offset="5" data-total="<?php echo $seller_total; ?>">
                        Seller Orders (<?php echo min(5, $seller_total); ?>/<?php echo $seller_total; ?>)
                    </button>
                </div>
                
                <div class="orders-content">
                    <!-- Buyer Orders -->
                    <div class="orders-list" id="buyerOrders">
                        <?php if (empty($buyer_orders)): ?>
                            <div class="empty-state">
                                <i class="fas fa-shopping-cart"></i>
                                <h3>No Buyer Orders</h3>
                                <p>This user hasn't placed any buyer orders yet</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($buyer_orders as $order): ?>
                                <div class="order-item">
                                    <div class="order-header">
                                        <div class="order-id">Buy Order #<?php echo $order['id']; ?></div>
                                        <span class="order-status status-<?php echo $order['status']; ?>">
                                            <?php echo str_replace('_', ' ', ucfirst($order['status'])); ?>
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦<?php echo number_format($order['amount'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining</label>
                                            <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Interest</label>
                                            <span>₦<?php echo number_format($order['interest'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Status</label>
                                            <span><?php echo ucfirst($order['match_status']); ?></span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        <button class="view-orders-btn" onclick="viewBuyerOrders(<?php echo $order['id']; ?>)">
                                            <i class="fas fa-eye"></i> View Orders
                                        </button>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                    
                    <!-- Seller Orders -->
                    <div class="orders-list" id="sellerOrders" style="display: none;">
                        <?php if (empty($seller_orders)): ?>
                            <div class="empty-state">
                                <i class="fas fa-shopping-cart"></i>
                                <h3>No Seller Orders</h3>
                                <p>This user hasn't placed any seller orders yet</p>
                            </div>
                        <?php else: ?>
                            <?php foreach ($seller_orders as $order): ?>
                                <div class="order-item" id="seller-order-<?php echo $order['id']; ?>">
                                    <div class="order-header">
                                        <div class="order-id">Sell Order #<?php echo $order['id']; ?></div>
                                        <span class="order-status status-<?php echo $order['status']; ?>">
                                            <?php echo str_replace('_', ' ', ucfirst($order['status'])); ?>
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦<?php echo number_format($order['amount'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining</label>
                                            <span>₦<?php echo number_format($order['amount_remaining'], 2); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Match Status</label>
                                            <span><?php echo ucfirst($order['match_status']); ?></span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Status</label>
                                            <span><?php echo str_replace('_', ' ', ucfirst($order['status'])); ?></span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        <?php if ($order['match_status'] === 'pending'): ?>
                                            <button class="pause-play-btn" 
                                                    onclick="toggleSellerOrderStatus(<?php echo $order['id']; ?>, '<?php echo $order['status']; ?>')"
                                                    id="pause-play-btn-<?php echo $order['id']; ?>">
                                                <?php if ($order['status'] === 'pause'): ?>
                                                    <i class="fas fa-play"></i> Play
                                                <?php else: ?>
                                                    <i class="fas fa-pause"></i> Pause
                                                <?php endif; ?>
                                            </button>
                                        <?php endif; ?>
                                        <button class="view-orders-btn" onclick="viewSellerOrders(<?php echo $order['id']; ?>)">
                                            <i class="fas fa-eye"></i> View Orders
                                        </button>
                                    </div>
                                </div>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </div>
                </div>
                
                <!-- Load More Button -->
                <button class="load-more-btn hidden" id="loadMoreBtn" data-type="buyer" data-offset="5">
                    <i class="fas fa-spinner fa-spin hidden"></i>
                    <span>Load More Orders</span>
                </button>
            </div>
        </div>
    </div>
    
    <!-- Edit Profile Modal -->
    <div class="modal-overlay" id="editProfileModal">
        <div class="edit-modal">
            <h3>Edit User Profile</h3>
            <form id="editProfileForm">
                <input type="hidden" id="editUserId" name="user_id" value="<?php echo $user['user_id']; ?>">
                
                <div class="form-group">
                    <label for="editFirstName">First Name</label>
                    <input type="text" class="form-control" id="editFirstName" name="first_name" value="<?php echo htmlspecialchars($user['first_name']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editLastName">Last Name</label>
                    <input type="text" class="form-control" id="editLastName" name="last_name" value="<?php echo htmlspecialchars($user['last_name']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editUsername">Username</label>
                    <input type="text" class="form-control" id="editUsername" name="username" value="<?php echo htmlspecialchars($user['username']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editEmail">Email</label>
                    <input type="email" class="form-control" id="editEmail" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editPhone">Phone Number</label>
                    <input type="tel" class="form-control" id="editPhone" name="phone_number" value="<?php echo htmlspecialchars($user['phone_number']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editDob">Date of Birth</label>
                    <input type="date" class="form-control" id="editDob" name="dob" value="<?php echo htmlspecialchars($user['dob']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editCountry">Country</label>
                    <input type="text" class="form-control" id="editCountry" name="country" value="<?php echo htmlspecialchars($user['country']); ?>" required>
                </div>
                
                <div class="form-group">
                    <label for="editAccountStatus">Account Status</label>
                    <select class="form-control" id="editAccountStatus" name="account_status" required>
                        <option value="active" <?php echo $user['account_status'] === 'active' ? 'selected' : ''; ?>>Active</option>
                        <option value="suspended" <?php echo $user['account_status'] === 'suspended' ? 'selected' : ''; ?>>Suspended</option>
                        <option value="closed" <?php echo $user['account_status'] === 'closed' ? 'selected' : ''; ?>>Closed</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="editVerification">Verification Status</label>
                    <select class="form-control" id="editVerification" name="verification" required>
                        <option value="verified" <?php echo $user['verification'] === 'verified' ? 'selected' : ''; ?>>Verified</option>
                        <option value="not verify" <?php echo $user['verification'] === 'not verify' ? 'selected' : ''; ?>>Not Verified</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="editProfileUrl">Profile Picture URL</label>
                    <input type="url" class="form-control" id="editProfileUrl" name="profile_url" value="<?php echo htmlspecialchars($user['profile_url'] ?? ''); ?>">
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelEditBtn">Cancel</button>
                    <button type="submit" class="form-btn primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Edit Bank Account Modal -->
    <div class="modal-overlay" id="editBankModal">
        <div class="edit-modal">
            <h3>Edit Bank Account</h3>
            <form id="editBankForm">
                <input type="hidden" id="editBankId" name="bank_id">
                <input type="hidden" id="editBankUserId" name="user_id" value="<?php echo $user['user_id']; ?>">
                
                <div class="form-group">
                    <label for="editBankName">Bank Name</label>
                    <input type="text" class="form-control" id="editBankName" name="bank_name" required>
                </div>
                
                <div class="form-group">
                    <label for="editAccountNumber">Account Number</label>
                    <input type="text" class="form-control" id="editAccountNumber" name="account_number" required>
                </div>
                
                <div class="form-group">
                    <label for="editAccountName">Account Name</label>
                    <input type="text" class="form-control" id="editAccountName" name="account_name" required>
                </div>
                
                <div class="form-group">
                    <label for="editBankCode">Bank Code</label>
                    <input type="text" class="form-control" id="editBankCode" name="bank_code" required>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelEditBankBtn">Cancel</button>
                    <button type="submit" class="form-btn primary">Save Changes</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Orders Modal -->
    <div class="modal-overlay" id="ordersModal">
        <div class="orders-modal">
            <h3 id="ordersModalTitle">Orders</h3>
            <div class="orders-list-modal" id="ordersModalList">
                <!-- Orders will be loaded here -->
            </div>
            <div class="form-actions">
                <button type="button" class="form-btn secondary" id="closeOrdersBtn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Receipt Modal -->
    <div class="modal-overlay" id="receiptModal">
        <div class="receipt-modal">
            <h3>Payment Receipt</h3>
            <img id="receiptImage" src="" alt="Payment Receipt" class="receipt-image">
            <div class="form-actions">
                <button type="button" class="form-btn secondary" id="closeReceiptBtn">Close</button>
            </div>
        </div>
    </div>
    
    <!-- Success/Error Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="alert-modal">
            <div class="alert-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="alert-actions">
                <button class="alert-btn primary" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <!-- Confirmation Modal -->
    <div class="modal-overlay" id="confirmModal">
        <div class="confirm-modal">
            <div class="alert-icon warning" id="confirmIcon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle">Confirm Action</h3>
            <p id="confirmMessage">Are you sure you want to perform this action?</p>
            <div class="alert-actions">
                <button class="alert-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button class="alert-btn danger" id="confirmActionBtn">Confirm</button>
            </div>
        </div>
    </div>
    
    <script>
        // Global variables
        let currentUserId = <?php echo $user['user_id']; ?>;
        let confirmCallback = null;
        let isLoading = false;
        let currentTabType = 'buyer';
        let currentTabOffset = 5;
        let currentTabTotal = <?php echo $buyer_total; ?>;
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon and color based on type
            icon.className = 'alert-icon ' + type;
            switch(type) {
                case 'success':
                    icon.innerHTML = '<i class="fas fa-check-circle"></i>';
                    break;
                case 'error':
                    icon.innerHTML = '<i class="fas fa-times-circle"></i>';
                    break;
                case 'warning':
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    break;
                case 'info':
                    icon.innerHTML = '<i class="fas fa-info-circle"></i>';
                    break;
            }
            
            modal.classList.add('active');
            
            // Auto hide after 2 seconds
            setTimeout(() => {
                modal.classList.remove('active');
            }, 2000);
        }
        
        // Show confirmation modal
        function showConfirm(title, message, callback) {
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmMessage').textContent = message;
            confirmCallback = callback;
            document.getElementById('confirmModal').classList.add('active');
        }
        
        // Orders tabs functionality
        document.querySelectorAll('.order-tab').forEach(tab => {
            tab.addEventListener('click', () => {
                // Update active tab
                document.querySelectorAll('.order-tab').forEach(t => t.classList.remove('active'));
                tab.classList.add('active');
                
                // Update global variables
                currentTabType = tab.getAttribute('data-type');
                currentTabOffset = parseInt(tab.getAttribute('data-offset'));
                currentTabTotal = parseInt(tab.getAttribute('data-total'));
                
                // Show/hide content
                document.getElementById('buyerOrders').style.display = currentTabType === 'buyer' ? 'block' : 'none';
                document.getElementById('sellerOrders').style.display = currentTabType === 'seller' ? 'block' : 'none';
                
                // Show/hide load more button
                const loadMoreBtn = document.getElementById('loadMoreBtn');
                loadMoreBtn.setAttribute('data-type', currentTabType);
                loadMoreBtn.setAttribute('data-offset', currentTabOffset);
                
                if (currentTabOffset < currentTabTotal) {
                    loadMoreBtn.classList.remove('hidden');
                } else {
                    loadMoreBtn.classList.add('hidden');
                }
            });
        });
        
        // View user profile
        function viewUserProfile(userId) {
            window.location.href = `profile.php?id=${userId}`;
        }
        
        // Suspend user - Updated to use b_update_users.php
        async function suspendUser(userId, userName) {
            showConfirm(
                'Suspend User',
                `Are you sure you want to suspend ${userName}?`,
                async () => {
                    try {
                        const response = await fetch('b_update_users.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'suspend',
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'User suspended successfully!', 'success');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showMessage('Error', data.message || 'Failed to suspend user', 'error');
                        }
                    } catch (error) {
                        console.error('Error suspending user:', error);
                        showMessage('Error', 'Error suspending user', 'error');
                    }
                }
            );
        }
        
        // Activate user - Updated to use b_update_users.php
        async function activateUser(userId, userName) {
            showConfirm(
                'Activate User',
                `Are you sure you want to activate ${userName}?`,
                async () => {
                    try {
                        const response = await fetch('b_update_users.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'activate',
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'User activated successfully!', 'success');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showMessage('Error', data.message || 'Failed to activate user', 'error');
                        }
                    } catch (error) {
                        console.error('Error activating user:', error);
                        showMessage('Error', 'Error activating user', 'error');
                    }
                }
            );
        }
        
        // Edit profile
        document.getElementById('editProfileBtn').addEventListener('click', () => {
            document.getElementById('editProfileModal').classList.add('active');
        });
        
        document.getElementById('cancelEditBtn').addEventListener('click', () => {
            document.getElementById('editProfileModal').classList.remove('active');
        });
        
        // Edit profile form submission - Updated to use b_update_users.php
        document.getElementById('editProfileForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            data.action = 'update_profile';
            
            try {
                const response = await fetch('b_update_users.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Profile updated successfully!', 'success');
                    document.getElementById('editProfileModal').classList.remove('active');
                    setTimeout(() => location.reload(), 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to update profile', 'error');
                }
            } catch (error) {
                console.error('Error updating profile:', error);
                showMessage('Error', 'Error updating profile', 'error');
            }
        });
        
        // Edit bank account - Still uses b_profile.php
        async function editBankAccount(bankId) {
            try {
                const response = await fetch(`b_profile.php?action=get_bank_account&bank_id=${bankId}`);
                const data = await response.json();
                
                if (data.success) {
                    document.getElementById('editBankId').value = data.bank.id;
                    document.getElementById('editBankName').value = data.bank.bank_name;
                    document.getElementById('editAccountNumber').value = data.bank.account_number;
                    document.getElementById('editAccountName').value = data.bank.account_name;
                    document.getElementById('editBankCode').value = data.bank.bank_code;
                    
                    document.getElementById('editBankModal').classList.add('active');
                } else {
                    showMessage('Error', data.message || 'Failed to load bank account', 'error');
                }
            } catch (error) {
                console.error('Error loading bank account:', error);
                showMessage('Error', 'Error loading bank account', 'error');
            }
        }
        
        document.getElementById('cancelEditBankBtn').addEventListener('click', () => {
            document.getElementById('editBankModal').classList.remove('active');
        });
        
        // Edit bank form submission - Still uses b_profile.php
        document.getElementById('editBankForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            data.action = 'update_bank_account';
            
            try {
                const response = await fetch('b_profile.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify(data)
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Bank account updated successfully!', 'success');
                    document.getElementById('editBankModal').classList.remove('active');
                    setTimeout(() => location.reload(), 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to update bank account', 'error');
                }
            } catch (error) {
                console.error('Error updating bank account:', error);
                showMessage('Error', 'Error updating bank account', 'error');
            }
        });
        
        // Delete bank account - Still uses b_profile.php
        async function deleteBankAccount(bankId) {
            showConfirm(
                'Delete Bank Account',
                'Are you sure you want to delete this bank account?',
                async () => {
                    try {
                        const response = await fetch('b_profile.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'delete_bank_account',
                                bank_id: bankId,
                                user_id: currentUserId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'Bank account deleted successfully!', 'success');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showMessage('Error', data.message || 'Failed to delete bank account', 'error');
                        }
                    } catch (error) {
                        console.error('Error deleting bank account:', error);
                        showMessage('Error', 'Error deleting bank account', 'error');
                    }
                }
            );
        }
        
        // View buyer orders - Still uses b_profile.php
        async function viewBuyerOrders(buyOrderId) {
            try {
                const response = await fetch(`b_profile.php?action=get_buyer_orders&buy_order_id=${buyOrderId}`);
                const data = await response.json();
                
                if (data.success) {
                    document.getElementById('ordersModalTitle').textContent = 'Buyer Orders Details';
                    const ordersList = document.getElementById('ordersModalList');
                    
                    if (data.orders.length === 0) {
                        ordersList.innerHTML = `
                            <div class="empty-state">
                                <i class="fas fa-shopping-cart"></i>
                                <h3>No Orders Found</h3>
                                <p>No orders found for this buyer order</p>
                            </div>
                        `;
                    } else {
                        let html = '';
                        data.orders.forEach(order => {
                            html += `
                                <div class="order-detail-modal">
                                    <div class="order-header">
                                        <div class="order-id">Order #${order.id}</div>
                                        <span class="order-status status-${order.status}">
                                            ${order.status.replace('_', ' ')}
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Seller</label>
                                            <span class="user-link" onclick="viewUserProfile(${order.seller_user_id})">
                                                ${order.seller_name}
                                            </span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Payer Status</label>
                                            <span>${order.payer_status}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Created At</label>
                                            <span>${new Date(order.created_at).toLocaleString()}</span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        ${order.payment_proof ? `
                                            <button class="view-orders-btn" onclick="viewReceipt('${order.payment_proof}')">
                                                <i class="fas fa-eye"></i> View Receipt
                                            </button>
                                        ` : ''}
                                        ${order.status === 'not approved' || order.status === 'pending' ? `
                                            <button class="approve-order-btn" onclick="approvePayment(${order.id}, ${order.buyer_id}, ${order.seller_user_id}, '${order.seller_name}')">
                                                <i class="fas fa-check"></i> Approve
                                            </button>
                                        ` : ''}
                                    </div>
                                </div>
                            `;
                        });
                        ordersList.innerHTML = html;
                    }
                    
                    document.getElementById('ordersModal').classList.add('active');
                } else {
                    showMessage('Error', data.message || 'Failed to load orders', 'error');
                }
            } catch (error) {
                console.error('Error loading orders:', error);
                showMessage('Error', 'Error loading orders', 'error');
            }
        }
        
        // View seller orders - Still uses b_profile.php
        async function viewSellerOrders(sellOrderId) {
            try {
                const response = await fetch(`b_profile.php?action=get_seller_orders&sell_order_id=${sellOrderId}`);
                const data = await response.json();
                
                if (data.success) {
                    document.getElementById('ordersModalTitle').textContent = 'Seller Orders Details';
                    const ordersList = document.getElementById('ordersModalList');
                    
                    if (data.orders.length === 0) {
                        ordersList.innerHTML = `
                            <div class="empty-state">
                                <i class="fas fa-shopping-cart"></i>
                                <h3>No Orders Found</h3>
                                <p>No orders found for this seller order</p>
                            </div>
                        `;
                    } else {
                        let html = '';
                        data.orders.forEach(order => {
                            html += `
                                <div class="order-detail-modal">
                                    <div class="order-header">
                                        <div class="order-id">Order #${order.id}</div>
                                        <span class="order-status status-${order.status}">
                                            ${order.status.replace('_', ' ')}
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Buyer</label>
                                            <span class="user-link" onclick="viewUserProfile(${order.buyer_user_id})">
                                                ${order.buyer_name}
                                            </span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Payer Status</label>
                                            <span>${order.payer_status}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Created At</label>
                                            <span>${new Date(order.created_at).toLocaleString()}</span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        ${order.payment_proof ? `
                                            <button class="view-orders-btn" onclick="viewReceipt('${order.payment_proof}')">
                                                <i class="fas fa-eye"></i> View Receipt
                                            </button>
                                        ` : ''}
                                        ${order.status === 'not approved' || order.status === 'pending' ? `
                                            <button class="approve-order-btn" onclick="approvePayment(${order.id}, ${order.buyer_id}, ${order.buyer_user_id}, '${order.buyer_name}')">
                                                <i class="fas fa-check"></i> Approve
                                            </button>
                                        ` : ''}
                                    </div>
                                </div>
                            `;
                        });
                        ordersList.innerHTML = html;
                    }
                    
                    document.getElementById('ordersModal').classList.add('active');
                } else {
                    showMessage('Error', data.message || 'Failed to load orders', 'error');
                }
            } catch (error) {
                console.error('Error loading orders:', error);
                showMessage('Error', 'Error loading orders', 'error');
            }
        }
        
        // View receipt
        function viewReceipt(imageUrl) {
            document.getElementById('receiptImage').src = imageUrl;
            document.getElementById('receiptModal').classList.add('active');
        }
        
        // Approve payment - Still uses b_profile.php
        async function approvePayment(orderMatchId, buyerId, otherUserId, otherUserName) {
            showConfirm(
                'Approve Payment',
                `Are you sure you want to approve this payment?`,
                async () => {
                    try {
                        const response = await fetch('b_profile.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'approve_payment',
                                order_id: orderMatchId,
                                buyer_id: buyerId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'Payment approved successfully!', 'success');
                            // Close orders modal and reload page after 2 seconds
                            document.getElementById('ordersModal').classList.remove('active');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showMessage('Error', data.message || 'Failed to approve payment', 'error');
                        }
                    } catch (error) {
                        console.error('Error approving payment:', error);
                        showMessage('Error', 'Error approving payment', 'error');
                    }
                }
            );
        }
        
        // Toggle seller order status (pause/play) - Still uses b_profile.php
        async function toggleSellerOrderStatus(orderId, currentStatus) {
            const action = currentStatus === 'pause' ? 'play' : 'pause';
            const actionText = currentStatus === 'pause' ? 'activate' : 'pause';
            
            showConfirm(
                `${actionText === 'pause' ? 'Pause' : 'Activate'} Seller Order`,
                `Are you sure you want to ${actionText} this seller order?`,
                async () => {
                    try {
                        const response = await fetch('b_profile.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'toggle_seller_order_status',
                                order_id: orderId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            // Update the button and status on the page without reloading
                            const btn = document.getElementById(`pause-play-btn-${orderId}`);
                            const orderItem = document.getElementById(`seller-order-${orderId}`);
                            const statusSpan = orderItem.querySelector('.order-status');
                            
                            // Update button
                            if (data.new_status === 'pause') {
                                btn.innerHTML = '<i class="fas fa-play"></i> Play';
                                btn.title = 'Click to activate order';
                                statusSpan.textContent = 'Pause';
                                statusSpan.className = 'order-status status-pause';
                            } else {
                                btn.innerHTML = '<i class="fas fa-pause"></i> Pause';
                                btn.title = 'Click to pause order';
                                statusSpan.textContent = 'In Progress';
                                statusSpan.className = 'order-status status-in_progress';
                            }
                            
                            showMessage('Success', `Order ${data.new_status === 'pause' ? 'paused' : 'activated'} successfully!`, 'success');
                        } else {
                            showMessage('Error', data.message || 'Failed to update order status', 'error');
                        }
                    } catch (error) {
                        console.error('Error toggling order status:', error);
                        showMessage('Error', 'Error updating order status', 'error');
                    }
                }
            );
        }
        
        // Close modals
        document.getElementById('closeOrdersBtn').addEventListener('click', () => {
            document.getElementById('ordersModal').classList.remove('active');
        });
        
        document.getElementById('closeReceiptBtn').addEventListener('click', () => {
            document.getElementById('receiptModal').classList.remove('active');
        });
        
        // View testimony
        function viewTestimony(url) {
            window.open(url, '_blank');
        }
        
        // Approve testimony - Still uses b_profile.php
        async function approveTestimony(testimonyId) {
            showConfirm(
                'Approve Testimony',
                'Are you sure you want to approve this testimony?',
                async () => {
                    try {
                        const response = await fetch('b_profile.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'approve_testimony',
                                testimony_id: testimonyId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'Testimony approved successfully!', 'success');
                            setTimeout(() => location.reload(), 2000);
                        } else {
                            showMessage('Error', data.message || 'Failed to approve testimony', 'error');
                        }
                    } catch (error) {
                        console.error('Error approving testimony:', error);
                        showMessage('Error', 'Error approving testimony', 'error');
                    }
                }
            );
        }
        
        // Load more orders - Still uses b_profile.php
        document.getElementById('loadMoreBtn').addEventListener('click', async () => {
            if (isLoading) return;
            
            const loadMoreBtn = document.getElementById('loadMoreBtn');
            const offset = parseInt(loadMoreBtn.getAttribute('data-offset'));
            const type = loadMoreBtn.getAttribute('data-type');
            
            try {
                isLoading = true;
                loadMoreBtn.querySelector('i').classList.remove('hidden');
                loadMoreBtn.querySelector('span').textContent = 'Loading...';
                
                const response = await fetch(`b_profile.php?action=load_more_orders&user_id=${currentUserId}&type=${type}&offset=${offset}`);
                const data = await response.json();
                
                if (data.success) {
                    const containerId = type === 'buyer' ? 'buyerOrders' : 'sellerOrders';
                    const container = document.getElementById(containerId);
                    
                    data.orders.forEach(order => {
                        if (type === 'buyer') {
                            const orderHtml = `
                                <div class="order-item">
                                    <div class="order-header">
                                        <div class="order-id">Buy Order #${order.id}</div>
                                        <span class="order-status status-${order.status}">
                                            ${order.status.replace('_', ' ')}
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining</label>
                                            <span>₦${parseFloat(order.amount_remaining).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Interest</label>
                                            <span>₦${parseFloat(order.interest).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Status</label>
                                            <span>${order.match_status}</span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        <button class="view-orders-btn" onclick="viewBuyerOrders(${order.id})">
                                            <i class="fas fa-eye"></i> View Orders
                                        </button>
                                    </div>
                                </div>
                            `;
                            container.insertAdjacentHTML('beforeend', orderHtml);
                        } else {
                            // For seller orders, include pause/play button
                            const orderHtml = `
                                <div class="order-item" id="seller-order-${order.id}">
                                    <div class="order-header">
                                        <div class="order-id">Sell Order #${order.id}</div>
                                        <span class="order-status status-${order.status}">
                                            ${order.status.replace('_', ' ')}
                                        </span>
                                    </div>
                                    <div class="order-details">
                                        <div class="order-info-item">
                                            <label>Amount</label>
                                            <span>₦${parseFloat(order.amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Remaining</label>
                                            <span>₦${parseFloat(order.amount_remaining).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Match Status</label>
                                            <span>${order.match_status}</span>
                                        </div>
                                        <div class="order-info-item">
                                            <label>Status</label>
                                            <span>${order.status.replace('_', ' ')}</span>
                                        </div>
                                    </div>
                                    <div class="order-actions">
                                        ${order.match_status === 'pending' ? `
                                            <button class="pause-play-btn" 
                                                    onclick="toggleSellerOrderStatus(${order.id}, '${order.status}')"
                                                    id="pause-play-btn-${order.id}">
                                                ${order.status === 'pause' ? '<i class="fas fa-play"></i> Play' : '<i class="fas fa-pause"></i> Pause'}
                                            </button>
                                        ` : ''}
                                        <button class="view-orders-btn" onclick="viewSellerOrders(${order.id})">
                                            <i class="fas fa-eye"></i> View Orders
                                        </button>
                                    </div>
                                </div>
                            `;
                            container.insertAdjacentHTML('beforeend', orderHtml);
                        }
                    });
                    
                    // Update offset
                    const newOffset = offset + 10;
                    loadMoreBtn.setAttribute('data-offset', newOffset);
                    
                    // Update tab
                    const tab = document.querySelector(`.order-tab[data-type="${type}"]`);
                    tab.setAttribute('data-offset', newOffset);
                    tab.textContent = `${type === 'buyer' ? 'Buyer' : 'Seller'} Orders (${Math.min(newOffset, currentTabTotal)}/${currentTabTotal})`;
                    
                    // Hide button if no more orders
                    if (newOffset >= currentTabTotal) {
                        loadMoreBtn.classList.add('hidden');
                    }
                } else {
                    showMessage('Error', data.message || 'Failed to load more orders', 'error');
                }
            } catch (error) {
                console.error('Error loading more orders:', error);
                showMessage('Error', 'Error loading more orders', 'error');
            } finally {
                isLoading = false;
                loadMoreBtn.querySelector('i').classList.add('hidden');
                loadMoreBtn.querySelector('span').textContent = 'Load More Orders';
            }
        });
        
        // Infinite scroll
        window.addEventListener('scroll', () => {
            const loadMoreBtn = document.getElementById('loadMoreBtn');
            if (loadMoreBtn.classList.contains('hidden') || isLoading) return;
            
            const rect = loadMoreBtn.getBoundingClientRect();
            if (rect.top <= window.innerHeight + 100) {
                loadMoreBtn.click();
            }
        });
        
        // Modal event listeners
        document.getElementById('messageOkBtn').addEventListener('click', () => {
            document.getElementById('messageModal').classList.remove('active');
        });
        
        document.getElementById('confirmCancelBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            confirmCallback = null;
        });
        
        document.getElementById('confirmActionBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            if (confirmCallback) {
                confirmCallback();
                confirmCallback = null;
            }
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                    if (modal.id === 'confirmModal') {
                        confirmCallback = null;
                    }
                }
            });
        });
        
        // Initialize load more button
        const loadMoreBtn = document.getElementById('loadMoreBtn');
        if (currentTabOffset < currentTabTotal) {
            loadMoreBtn.classList.remove('hidden');
        }
    </script>
</body>
</html>