<?php
// refund.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Get action from request
$input = json_decode(file_get_contents('php://input'), true);
if (!$input) {
    $input = $_POST;
}

$action = $input['action'] ?? '';

header('Content-Type: application/json');

try {
    switch ($action) {
        case 'refund_buyer_order':
            refundBuyerOrder($input);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
    }
} catch (PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

function refundBuyerOrder($input) {
    global $pdo;
    
    $order_id = $input['order_id'] ?? 0;
    
    if (!$order_id) {
        echo json_encode(['success' => false, 'message' => 'Order ID is required']);
        return;
    }
    
    // Start transaction
    $pdo->beginTransaction();
    
    try {
        // 1. Get buyer order details
        $stmt = $pdo->prepare("SELECT * FROM buyer_orders WHERE id = :order_id");
        $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        $buyer_order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$buyer_order) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Buyer order not found']);
            return;
        }
        
        $buyer_user_id = $buyer_order['user_id'];
        $buyer_amount = $buyer_order['amount'];
        
        // 2. Check if this buyer order is in recommit table as primary_buy_id
        $stmt = $pdo->prepare("SELECT * FROM recommit WHERE primary_buy_id = :order_id");
        $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        $recommit_records = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // Delete any recommit records where this is the primary buy id
        if (!empty($recommit_records)) {
            $stmt = $pdo->prepare("DELETE FROM recommit WHERE primary_buy_id = :order_id");
            $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
            $stmt->execute();
        }
        
        // 3. Get all matched orders for this buyer order from orders table
        $stmt = $pdo->prepare("
            SELECT o.*, so.user_id as seller_user_id, so.amount_remaining as seller_amount_remaining
            FROM orders o
            JOIN seller_orders so ON o.seller_id = so.id
            WHERE o.buyer_id = :order_id
        ");
        $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        $matched_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        // 4. Delete the buyer order
        $stmt = $pdo->prepare("DELETE FROM buyer_orders WHERE id = :order_id");
        $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Failed to delete buyer order']);
            return;
        }
        
        // 5. Check if user has any approved buyer orders
        $stmt = $pdo->prepare("
            SELECT id, amount, status 
            FROM buyer_orders 
            WHERE user_id = :user_id 
            AND status = 'approved'
            ORDER BY created_at ASC
            LIMIT 1
        ");
        $stmt->bindParam(':user_id', $buyer_user_id, PDO::PARAM_INT);
        $stmt->execute();
        $approved_buyer_order = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // 6. Get refund fee from fund_settings
        $stmt = $pdo->prepare("SELECT `refund fee` FROM fund_settings ORDER BY `created at` DESC LIMIT 1");
        $stmt->execute();
        $fund_settings = $stmt->fetch(PDO::FETCH_ASSOC);
        $refund_fee_percentage = $fund_settings['refund fee'] ?? 0;
        
        if ($approved_buyer_order) {
            // User has an approved buyer order - deduct refund fee from it
            $refund_fee_amount = ($buyer_amount * $refund_fee_percentage) / 100;
            $new_amount = floatval($approved_buyer_order['amount']) - $refund_fee_amount;
            
            $stmt = $pdo->prepare("
                UPDATE buyer_orders 
                SET amount = :new_amount,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = :approved_order_id
            ");
            $stmt->bindParam(':new_amount', $new_amount);
            $stmt->bindParam(':approved_order_id', $approved_buyer_order['id'], PDO::PARAM_INT);
            $stmt->execute();
        } else {
            // User has no approved buyer orders - check security deposit
            $stmt = $pdo->prepare("
                SELECT id, status 
                FROM security_deposit 
                WHERE user_id = :user_id 
                ORDER BY created_at DESC 
                LIMIT 1
            ");
            $stmt->bindParam(':user_id', $buyer_user_id, PDO::PARAM_INT);
            $stmt->execute();
            $security_deposit = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($security_deposit && ($security_deposit['status'] == 'paid' || $security_deposit['status'] == 'refunded')) {
                // Mark security deposit as not paid
                $stmt = $pdo->prepare("
                    UPDATE security_deposit 
                    SET status = 'not paid',
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = :deposit_id
                ");
                $stmt->bindParam(':deposit_id', $security_deposit['id'], PDO::PARAM_INT);
                $stmt->execute();
            }
        }
        
        // 7. Process refunds for each matched seller
        foreach ($matched_orders as $matched_order) {
            $seller_order_id = $matched_order['seller_id'];
            $seller_user_id = $matched_order['seller_user_id'];
            $refund_amount = $matched_order['amount'];
            
            // Delete the matched order record
            $stmt = $pdo->prepare("DELETE FROM orders WHERE id = :order_match_id");
            $stmt->bindParam(':order_match_id', $matched_order['id'], PDO::PARAM_INT);
            $stmt->execute();
            
            // Get current seller order details
            $stmt = $pdo->prepare("SELECT * FROM seller_orders WHERE id = :seller_order_id");
            $stmt->bindParam(':seller_order_id', $seller_order_id, PDO::PARAM_INT);
            $stmt->execute();
            $seller_order = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($seller_order) {
                // Update seller order: refund amount and reset status if needed
                $new_amount_remaining = floatval($seller_order['amount_remaining']) + floatval($refund_amount);
                
                // If status was 'sold', change it back to 'in progress' or 'pause' based on current status
                $new_status = $seller_order['status'];
                if ($seller_order['status'] === 'sold') {
                    $new_status = 'in progress';
                }
                
                // If match_status was 'completed', change it back to 'pending'
                $new_match_status = ($seller_order['match_status'] === 'completed') ? 'pending' : $seller_order['match_status'];
                
                $stmt = $pdo->prepare("
                    UPDATE seller_orders 
                    SET amount_remaining = :amount_remaining,
                        status = :status,
                        match_status = :match_status,
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = :seller_order_id
                ");
                $stmt->bindParam(':amount_remaining', $new_amount_remaining);
                $stmt->bindParam(':status', $new_status);
                $stmt->bindParam(':match_status', $new_match_status);
                $stmt->bindParam(':seller_order_id', $seller_order_id, PDO::PARAM_INT);
                $stmt->execute();
            }
        }
        
        // 8. Commit transaction
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Refund processed successfully. Buyer order deleted and sellers refunded.'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
    }
}