<?php
// users.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Users Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        .refresh-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .refresh-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }
        
        .refresh-btn.loading i {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Search and Filter Container */
        .filter-container {
            background: white;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 30px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            display: flex;
            gap: 15px;
            align-items: center;
            flex-wrap: wrap;
        }
        
        .search-box {
            flex: 1;
            min-width: 300px;
            display: flex;
            gap: 10px;
            align-items: center;
        }
        
        .search-input {
            flex: 1;
            padding: 12px 20px;
            border: 1px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .search-input:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .search-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 24px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .search-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }
        
        .search-btn.loading i {
            animation: spin 1s linear infinite;
        }
        
        .clear-search {
            background: var(--gray-light);
            color: var(--gray);
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .clear-search:hover {
            background: #d1d5db;
        }
        
        /* Status Filter Dropdown */
        .status-filter {
            position: relative;
            min-width: 200px;
        }
        
        .filter-dropdown {
            width: 100%;
            padding: 12px 20px;
            border: 1px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            background: white;
            color: var(--dark);
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .filter-dropdown:hover {
            border-color: var(--primary);
        }
        
        .filter-dropdown.active {
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .filter-options {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid var(--gray-light);
            border-radius: 10px;
            margin-top: 5px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
            z-index: 100;
            opacity: 0;
            visibility: hidden;
            transform: translateY(-10px);
            transition: all 0.3s ease;
        }
        
        .filter-options.active {
            opacity: 1;
            visibility: visible;
            transform: translateY(0);
        }
        
        .filter-option {
            padding: 12px 20px;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .filter-option:hover {
            background: var(--light);
        }
        
        .filter-option.selected {
            background: var(--light);
            color: var(--primary);
        }
        
        .filter-option.selected .option-check {
            display: block;
        }
        
        .option-check {
            display: none;
            color: var(--primary);
            margin-left: auto;
        }
        
        /* Stats Overview */
        .stats-overview {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            color: white;
        }
        
        .icon-total { background: linear-gradient(135deg, #3b82f6, #1d4ed8); }
        .icon-active { background: linear-gradient(135deg, #10b981, #059669); }
        .icon-verified { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
        .icon-suspended { background: linear-gradient(135deg, #f59e0b, #d97706); }
        
        .stat-info h3 {
            font-size: 24px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .stat-info p {
            font-size: 14px;
            color: var(--gray);
            font-weight: 500;
        }
        
        /* Users List */
        .users-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .users-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .user-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary);
        }
        
        .user-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
        }
        
        .user-item.active {
            border-left-color: var(--success);
        }
        
        .user-item.suspended {
            border-left-color: var(--warning);
        }
        
        .user-item.closed {
            border-left-color: var(--danger);
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
            flex: 1;
        }
        
        .user-avatar-container {
            position: relative;
        }
        
        .user-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .verified-badge {
            position: absolute;
            bottom: 0;
            right: 0;
            background: var(--primary);
            color: white;
            width: 20px;
            height: 20px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 12px;
            border: 2px solid white;
        }
        
        .user-details {
            flex: 2;
        }
        
        .user-details a {
            text-decoration: none;
            color: inherit;
            transition: color 0.3s ease;
        }
        
        .user-details a:hover {
            color: var(--primary);
        }
        
        .user-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .user-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 3px;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .verified-icon {
            color: var(--primary);
            font-size: 12px;
        }
        
        .user-meta {
            flex: 1;
            padding: 0 20px;
        }
        
        .user-meta h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .meta-info {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .meta-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .meta-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .meta-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .user-actions {
            display: flex;
            gap: 10px;
            flex-shrink: 0;
        }
        
        .view-btn, .suspend-btn, .activate-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .view-btn {
            background: var(--primary);
            color: white;
        }
        
        .view-btn:hover {
            background: var(--primary-dark);
        }
        
        .activate-btn {
            background: var(--success);
            color: white;
        }
        
        .activate-btn:hover {
            background: var(--secondary-dark);
        }
        
        .suspend-btn {
            background: var(--warning);
            color: white;
        }
        
        .suspend-btn:hover {
            background: #d97706;
        }
        
        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-active {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-suspended {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-closed {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .verification-badge {
            background: #dbeafe;
            color: #1e40af;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        /* Custom Modal Overlay */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        /* Success/Error Modal */
        .alert-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .alert-modal {
            transform: translateY(0);
        }
        
        .alert-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .alert-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-icon.warning {
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .alert-icon.info {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .alert-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .alert-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .alert-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .alert-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .alert-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .alert-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .alert-btn.success {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-btn.success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .alert-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-btn.danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        /* Toast Notification */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 16px 24px;
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 5000;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            max-width: 400px;
        }
        
        .toast.active {
            opacity: 1;
            transform: translateX(0);
        }
        
        .toast.success {
            border-left: 4px solid var(--success);
        }
        
        .toast.error {
            border-left: 4px solid var(--danger);
        }
        
        .toast.warning {
            border-left: 4px solid var(--warning);
        }
        
        .toast.info {
            border-left: 4px solid var(--primary);
        }
        
        .toast-icon {
            font-size: 20px;
        }
        
        .toast.success .toast-icon {
            color: var(--success);
        }
        
        .toast.error .toast-icon {
            color: var(--danger);
        }
        
        .toast.warning .toast-icon {
            color: var(--warning);
        }
        
        .toast.info .toast-icon {
            color: var(--primary);
        }
        
        .toast-content {
            flex: 1;
        }
        
        .toast-content h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .toast-content p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .close-toast {
            background: none;
            border: none;
            color: var(--gray);
            cursor: pointer;
            font-size: 16px;
            padding: 4px;
            transition: color 0.3s ease;
        }
        
        .close-toast:hover {
            color: var(--danger);
        }
        
        /* Loading Spinner */
        .loading-spinner {
            text-align: center;
            padding: 20px;
        }
        
        .spinner {
            width: 40px;
            height: 40px;
            border: 4px solid var(--gray-light);
            border-top: 4px solid var(--primary);
            border-radius: 50%;
            animation: spin 1s linear infinite;
            margin: 0 auto;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 20px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .empty-state p {
            font-size: 16px;
            opacity: 0.7;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .user-item {
                flex-direction: column;
                gap: 20px;
                align-items: stretch;
            }
            
            .user-info, .user-meta {
                padding: 0;
            }
            
            .user-actions {
                justify-content: flex-end;
                margin-top: 15px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .filter-container {
                flex-direction: column;
                align-items: stretch;
            }
            
            .search-box {
                min-width: auto;
            }
            
            .status-filter {
                width: 100%;
            }
            
            .stats-overview {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .users-section {
                padding: 20px;
            }
            
            .meta-info {
                flex-direction: column;
                gap: 10px;
            }
            
            .alert-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .alert-actions {
                flex-direction: column;
            }
            
            .alert-btn {
                width: 100%;
            }
        }
        
        @media (max-width: 480px) {
            .stats-overview {
                grid-template-columns: 1fr;
            }
            
            .user-info {
                flex-direction: column;
                text-align: center;
            }
            
            .user-actions {
                flex-direction: column;
            }
            
            .view-btn, .suspend-btn, .activate-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h2>Users Management</h2>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                    <button class="refresh-btn" id="refreshBtn">
                        <i class="fas fa-sync-alt"></i> Refresh
                    </button>
                </div>
            </div>
            
            <!-- Search and Filter Container -->
            <div class="filter-container">
                <div class="search-box">
                    <input type="text" class="search-input" id="searchInput" placeholder="Search by first name or email..." value="">
                    <button class="search-btn" id="searchBtn">
                        <i class="fas fa-search"></i> Search
                    </button>
                    <button class="clear-search" id="clearSearch" style="display: none;">
                        <i class="fas fa-times"></i> Clear
                    </button>
                </div>
                
                <!-- Status Filter Dropdown -->
                <div class="status-filter">
                    <div class="filter-dropdown" id="statusFilter">
                        <span>Filter by Status</span>
                        <i class="fas fa-chevron-down"></i>
                    </div>
                    <div class="filter-options" id="statusOptions">
                        <div class="filter-option" data-value="all">
                            All Users
                            <i class="fas fa-check option-check"></i>
                        </div>
                        <div class="filter-option" data-value="active">
                            Active Users
                            <i class="fas fa-check option-check"></i>
                        </div>
                        <div class="filter-option" data-value="suspended">
                            Suspended Users
                            <i class="fas fa-check option-check"></i>
                        </div>
                        <div class="filter-option" data-value="closed">
                            Closed Accounts
                            <i class="fas fa-check option-check"></i>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Stats Overview -->
            <div class="stats-overview">
                <div class="stat-card">
                    <div class="stat-icon icon-total">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="totalCount">0</h3>
                        <p>Total Users</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon icon-active">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="activeCount">0</h3>
                        <p>Active Users</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon icon-verified">
                        <i class="fas fa-shield-check"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="verifiedCount">0</h3>
                        <p>Verified Users</p>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon icon-suspended">
                        <i class="fas fa-user-slash"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="suspendedCount">0</h3>
                        <p>Suspended Users</p>
                    </div>
                </div>
            </div>
            
            <!-- Users List -->
            <div class="users-section">
                <h3 class="section-title">
                    <i class="fas fa-users"></i>
                    All Users
                </h3>
                
                <div class="users-list" id="usersList">
                    <!-- Users will be loaded here -->
                    <div class="empty-state">
                        <i class="fas fa-user-slash"></i>
                        <h3>No Users Found</h3>
                        <p>There are no users in the system yet</p>
                    </div>
                </div>
                <div class="loading-spinner" id="loadingSpinner" style="display: none;">
                    <div class="spinner"></div>
                    <p>Loading more users...</p>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Custom Modals -->
    <div class="modal-overlay" id="confirmModal">
        <div class="alert-modal">
            <div class="alert-icon warning">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle">Confirm Action</h3>
            <p id="confirmMessage">Are you sure you want to perform this action?</p>
            <div class="alert-actions">
                <button class="alert-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button class="alert-btn danger" id="confirmActionBtn">Confirm</button>
            </div>
        </div>
    </div>
    
    <!-- Success/Error Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="alert-modal">
            <div class="alert-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="alert-actions">
                <button class="alert-btn primary" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <!-- Toast Notification -->
    <div class="toast" id="toastNotification">
        <div class="toast-icon">
            <i class="fas fa-info-circle"></i>
        </div>
        <div class="toast-content">
            <h4 id="toastTitle">Notification</h4>
            <p id="toastMessage">This is a notification message</p>
        </div>
        <button class="close-toast" id="closeToast">
            <i class="fas fa-times"></i>
        </button>
    </div>
    
    <script>
        // Global variables
        let currentUserId = null;
        let confirmCallback = null;
        let currentPage = 1;
        let isLoading = false;
        let hasMore = true;
        let currentSearch = '';
        let currentStatus = 'all';
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Show confirmation modal
        function showConfirm(title, message, callback) {
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmMessage').textContent = message;
            confirmCallback = callback;
            document.getElementById('confirmModal').classList.add('active');
        }
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon and color based on type
            icon.className = 'alert-icon ' + type;
            switch(type) {
                case 'success':
                    icon.innerHTML = '<i class="fas fa-check-circle"></i>';
                    break;
                case 'error':
                    icon.innerHTML = '<i class="fas fa-times-circle"></i>';
                    break;
                case 'warning':
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    break;
                case 'info':
                    icon.innerHTML = '<i class="fas fa-info-circle"></i>';
                    break;
            }
            
            modal.classList.add('active');
        }
        
        // Show toast notification
        function showToast(title, message, type = 'info', duration = 3000) {
            const toast = document.getElementById('toastNotification');
            const toastTitle = document.getElementById('toastTitle');
            const toastMessage = document.getElementById('toastMessage');
            const toastIcon = toast.querySelector('.toast-icon i');
            
            toastTitle.textContent = title;
            toastMessage.textContent = message;
            
            // Set type
            toast.className = 'toast ' + type;
            switch(type) {
                case 'success':
                    toastIcon.className = 'fas fa-check-circle';
                    break;
                case 'error':
                    toastIcon.className = 'fas fa-times-circle';
                    break;
                case 'warning':
                    toastIcon.className = 'fas fa-exclamation-triangle';
                    break;
                case 'info':
                    toastIcon.className = 'fas fa-info-circle';
                    break;
            }
            
            toast.classList.add('active');
            
            // Auto hide
            setTimeout(() => {
                toast.classList.remove('active');
            }, duration);
        }
        
        // Initialize status filter
        function initStatusFilter() {
            const filterDropdown = document.getElementById('statusFilter');
            const filterOptions = document.getElementById('statusOptions');
            const options = filterOptions.querySelectorAll('.filter-option');
            
            // Set initial selection
            options.forEach(option => {
                if (option.getAttribute('data-value') === currentStatus) {
                    option.classList.add('selected');
                }
            });
            
            // Toggle dropdown
            filterDropdown.addEventListener('click', () => {
                filterDropdown.classList.toggle('active');
                filterOptions.classList.toggle('active');
            });
            
            // Handle option selection
            options.forEach(option => {
                option.addEventListener('click', () => {
                    const value = option.getAttribute('data-value');
                    
                    // Update selected state
                    options.forEach(opt => opt.classList.remove('selected'));
                    option.classList.add('selected');
                    
                    // Update current status
                    currentStatus = value;
                    
                    // Update dropdown text
                    const text = option.textContent.trim();
                    filterDropdown.querySelector('span').textContent = text.split('\n')[0];
                    
                    // Close dropdown
                    filterDropdown.classList.remove('active');
                    filterOptions.classList.remove('active');
                    
                    // Reload users with new filter
                    loadUsers(true);
                });
            });
            
            // Close dropdown when clicking outside
            document.addEventListener('click', (e) => {
                if (!filterDropdown.contains(e.target) && !filterOptions.contains(e.target)) {
                    filterDropdown.classList.remove('active');
                    filterOptions.classList.remove('active');
                }
            });
        }
        
        // Load users
        async function loadUsers(reset = false) {
            if (isLoading) return;
            
            if (reset) {
                currentPage = 1;
                hasMore = true;
                document.getElementById('usersList').innerHTML = '';
            }
            
            if (!hasMore) return;
            
            isLoading = true;
            document.getElementById('loadingSpinner').style.display = 'block';
            
            try {
                let url = `b_users.php?page=${currentPage}&limit=10`;
                if (currentSearch) {
                    url += `&search=${encodeURIComponent(currentSearch)}`;
                }
                if (currentStatus !== 'all') {
                    url += `&status=${encodeURIComponent(currentStatus)}`;
                }
                
                const response = await fetch(url);
                const data = await response.json();
                
                if (data.success) {
                    if (reset) {
                        updateStats(data.stats);
                    }
                    updateUsersList(data.users, reset);
                    hasMore = data.hasMore;
                    if (hasMore) {
                        currentPage++;
                    }
                } else {
                    showToast('Error', data.message || 'Failed to load users', 'error');
                }
            } catch (error) {
                console.error('Error loading users:', error);
                showToast('Error', 'Error loading users', 'error');
            } finally {
                isLoading = false;
                document.getElementById('loadingSpinner').style.display = 'none';
            }
        }
        
        // Search users
        async function searchUsers() {
            currentSearch = document.getElementById('searchInput').value.trim();
            const clearBtn = document.getElementById('clearSearch');
            
            if (currentSearch) {
                clearBtn.style.display = 'flex';
            } else {
                clearBtn.style.display = 'none';
                currentSearch = '';
            }
            
            const btn = document.getElementById('searchBtn');
            btn.classList.add('loading');
            
            await loadUsers(true);
            
            setTimeout(() => {
                btn.classList.remove('loading');
            }, 500);
        }
        
        // Clear search
        function clearSearch() {
            document.getElementById('searchInput').value = '';
            document.getElementById('clearSearch').style.display = 'none';
            currentSearch = '';
            loadUsers(true);
        }
        
        // Update users list
        function updateUsersList(users, reset = false) {
            const listContainer = document.getElementById('usersList');
            
            if (!users || users.length === 0) {
                if (reset) {
                    listContainer.innerHTML = `
                        <div class="empty-state">
                            <i class="fas fa-user-slash"></i>
                            <h3>No Users Found</h3>
                            <p>${currentSearch || currentStatus !== 'all' ? 'No users match your search criteria' : 'There are no users in the system yet'}</p>
                        </div>
                    `;
                }
                return;
            }
            
            if (reset) {
                listContainer.innerHTML = '';
            }
            
            let html = '';
            
            users.forEach(user => {
                const firstName = user.first_name || 'Unknown';
                const lastName = user.last_name || '';
                const fullName = `${firstName} ${lastName}`;
                const initials = getUserInitials(firstName);
                const createdAt = new Date(user.created_at).toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
                
                html += `
                    <div class="user-item ${user.account_status}" data-id="${user.user_id}">
                        <div class="user-info">
                            <div class="user-avatar-container">
                                <div class="user-avatar">
                                    ${user.profile_url ? 
                                        `<img src="${user.profile_url}" alt="${firstName}" onerror="this.style.display='none'; this.parentElement.innerHTML='${initials}';">` : 
                                        initials}
                                </div>
                                ${user.verification === 'verified' ? '<div class="verified-badge"><i class="fas fa-check"></i></div>' : ''}
                            </div>
                            <div class="user-details">
                                <a href="profile.php?id=${user.user_id}">
                                    <h4>${fullName}</h4>
                                </a>
                                <a href="profile.php?id=${user.user_id}">
                                    <p>${user.email} ${user.email_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}</p>
                                </a>
                                <p>${user.phone_number} ${user.phone_verified === 'verified' ? '<i class="fas fa-check-circle verified-icon"></i>' : ''}</p>
                                <p>Username: ${user.username}</p>
                            </div>
                        </div>
                        
                        <div class="user-meta">
                            <h4>User Information</h4>
                            <div class="meta-info">
                                <div class="meta-info-item">
                                    <label>Account Status</label>
                                    <span class="status-badge status-${user.account_status}">${user.account_status}</span>
                                </div>
                                <div class="meta-info-item">
                                    <label>Verification</label>
                                    <span class="verification-badge">${user.verification}</span>
                                </div>
                                <div class="meta-info-item">
                                    <label>Joined</label>
                                    <span>${createdAt}</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="user-actions">
                            <a href="profile.php?id=${user.user_id}" class="view-btn">
                                <i class="fas fa-eye"></i> View
                            </a>
                            ${user.account_status === 'active' ? `
                                <button class="suspend-btn" onclick="suspendUser(${user.user_id}, '${firstName}')">
                                    <i class="fas fa-user-slash"></i> Suspend
                                </button>
                            ` : ''}
                            ${user.account_status === 'suspended' ? `
                                <button class="activate-btn" onclick="activateUser(${user.user_id}, '${firstName}')">
                                    <i class="fas fa-user-check"></i> Activate
                                </button>
                            ` : ''}
                        </div>
                    </div>
                `;
            });
            
            if (reset) {
                listContainer.innerHTML = html;
            } else {
                listContainer.insertAdjacentHTML('beforeend', html);
            }
        }
        
        // Update statistics
        function updateStats(stats) {
            document.getElementById('totalCount').textContent = stats.total || 0;
            document.getElementById('activeCount').textContent = stats.active || 0;
            document.getElementById('verifiedCount').textContent = stats.verified || 0;
            document.getElementById('suspendedCount').textContent = stats.suspended || 0;
        }
        
        // Get user initials
        function getUserInitials(name) {
            const names = name.split(' ');
            let initials = '';
            names.forEach(name => {
                if (name.trim()) {
                    initials += name.trim().charAt(0).toUpperCase();
                }
            });
            return initials.substring(0, 2);
        }
        
        // Suspend user
        async function suspendUser(userId, userName) {
            showConfirm(
                'Suspend User',
                `Are you sure you want to suspend ${userName}?`,
                async () => {
                    try {
                        const response = await fetch('b_users.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'suspend',
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'User suspended successfully!', 'success');
                            // Reload users
                            loadUsers(true);
                        } else {
                            showMessage('Error', data.message || 'Failed to suspend user', 'error');
                        }
                    } catch (error) {
                        console.error('Error suspending user:', error);
                        showMessage('Error', 'Error suspending user', 'error');
                    }
                }
            );
        }
        
        // Activate user
        async function activateUser(userId, userName) {
            showConfirm(
                'Activate User',
                `Are you sure you want to activate ${userName}?`,
                async () => {
                    try {
                        const response = await fetch('b_users.php', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                            },
                            body: JSON.stringify({
                                action: 'activate',
                                user_id: userId
                            })
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'User activated successfully!', 'success');
                            // Reload users
                            loadUsers(true);
                        } else {
                            showMessage('Error', data.message || 'Failed to activate user', 'error');
                        }
                    } catch (error) {
                        console.error('Error activating user:', error);
                        showMessage('Error', 'Error activating user', 'error');
                    }
                }
            );
        }
        
        // Modal event listeners
        document.getElementById('confirmCancelBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            confirmCallback = null;
        });
        
        document.getElementById('confirmActionBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            if (confirmCallback) {
                confirmCallback();
                confirmCallback = null;
            }
        });
        
        document.getElementById('messageOkBtn').addEventListener('click', () => {
            document.getElementById('messageModal').classList.remove('active');
        });
        
        document.getElementById('closeToast').addEventListener('click', () => {
            document.getElementById('toastNotification').classList.remove('active');
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                    if (modal.id === 'confirmModal') {
                        confirmCallback = null;
                    }
                }
            });
        });
        
        // Refresh button
        document.getElementById('refreshBtn').addEventListener('click', () => {
            const btn = document.getElementById('refreshBtn');
            btn.classList.add('loading');
            
            loadUsers(true).finally(() => {
                setTimeout(() => {
                    btn.classList.remove('loading');
                }, 500);
            });
        });
        
        // Search button
        document.getElementById('searchBtn').addEventListener('click', searchUsers);
        
        // Clear search button
        document.getElementById('clearSearch').addEventListener('click', clearSearch);
        
        // Search on Enter key
        document.getElementById('searchInput').addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                searchUsers();
            }
        });
        
        // Infinite scroll
        window.addEventListener('scroll', () => {
            if (isLoading || !hasMore) return;
            
            const scrollPosition = window.innerHeight + window.scrollY;
            const pageHeight = document.documentElement.scrollHeight;
            
            // Load more when 100px from bottom
            if (pageHeight - scrollPosition < 100) {
                loadUsers(false);
            }
        });
        
        // Initialize status filter
        initStatusFilter();
        
        // Initial load
        loadUsers(true);
    </script>
</body>
</html>