<?php
// admin.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        .add-admin-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .add-admin-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }
        
        /* Admin List Section */
        .admin-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .admin-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .admin-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
            border-left: 4px solid var(--primary);
        }
        
        .admin-item:hover {
            background: #f1f5f9;
            transform: translateX(5px);
        }
        
        .admin-item.root {
            border-left-color: var(--warning);
        }
        
        .admin-item.moderator {
            border-left-color: var(--primary);
        }
        
        .admin-info {
            display: flex;
            align-items: center;
            gap: 15px;
            flex: 1;
        }
        
        .admin-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
        }
        
        .admin-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .admin-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 3px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .admin-details p i {
            font-size: 12px;
        }
        
        .role-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .role-root {
            background: #fef3c7;
            color: #92400e;
        }
        
        .role-moderator {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .admin-actions {
            display: flex;
            gap: 10px;
            flex-shrink: 0;
        }
        
        .edit-btn, .delete-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .edit-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-btn:hover {
            background: var(--primary-dark);
        }
        
        .delete-btn {
            background: var(--danger);
            color: white;
        }
        
        .delete-btn:hover {
            background: #dc2626;
        }
        
        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .admin-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .admin-modal {
            transform: translateY(0);
        }
        
        .modal-header {
            margin-bottom: 25px;
        }
        
        .modal-header h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .modal-header p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            color: var(--dark);
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .form-control.error {
            border-color: var(--danger);
        }
        
        .form-text {
            font-size: 12px;
            color: var(--gray);
            margin-top: 5px;
        }
        
        .form-select {
            width: 100%;
            padding: 12px 16px;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            color: var(--dark);
            background: white;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .form-select:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        
        .modal-btn {
            flex: 1;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
        }
        
        .modal-btn.primary {
            background: var(--primary);
            color: white;
        }
        
        .modal-btn.primary:hover {
            background: var(--primary-dark);
        }
        
        .modal-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .modal-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .modal-btn.danger {
            background: var(--danger);
            color: white;
        }
        
        .modal-btn.danger:hover {
            background: #dc2626;
        }
        
        /* Toast Notification */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 16px 24px;
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 5000;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            max-width: 400px;
        }
        
        .toast.active {
            opacity: 1;
            transform: translateX(0);
        }
        
        .toast.success {
            border-left: 4px solid var(--success);
        }
        
        .toast.error {
            border-left: 4px solid var(--danger);
        }
        
        .toast.warning {
            border-left: 4px solid var(--warning);
        }
        
        .toast.info {
            border-left: 4px solid var(--primary);
        }
        
        .toast-icon {
            font-size: 20px;
        }
        
        .toast.success .toast-icon {
            color: var(--success);
        }
        
        .toast.error .toast-icon {
            color: var(--danger);
        }
        
        .toast.warning .toast-icon {
            color: var(--warning);
        }
        
        .toast.info .toast-icon {
            color: var(--primary);
        }
        
        .toast-content {
            flex: 1;
        }
        
        .toast-content h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .toast-content p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .close-toast {
            background: none;
            border: none;
            color: var(--gray);
            cursor: pointer;
            font-size: 16px;
            padding: 4px;
            transition: color 0.3s ease;
        }
        
        .close-toast:hover {
            color: var(--danger);
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 20px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .empty-state p {
            font-size: 16px;
            opacity: 0.7;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .admin-section {
                padding: 20px;
            }
            
            .admin-item {
                flex-direction: column;
                gap: 20px;
                align-items: stretch;
            }
            
            .admin-actions {
                justify-content: flex-end;
                margin-top: 15px;
            }
            
            .admin-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .modal-actions {
                flex-direction: column;
            }
        }
        
        @media (max-width: 480px) {
            .admin-info {
                flex-direction: column;
                text-align: center;
            }
            
            .admin-actions {
                flex-direction: column;
            }
            
            .edit-btn, .delete-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h2>Admin Management</h2>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                    <button class="add-admin-btn" id="addAdminBtn">
                        <i class="fas fa-user-plus"></i> Add Admin
                    </button>
                </div>
            </div>
            
            <!-- Admin List -->
            <div class="admin-section">
                <h3 class="section-title">
                    <i class="fas fa-users-cog"></i>
                    System Administrators
                </h3>
                
                <div class="admin-list" id="adminList">
                    <!-- Admins will be loaded here -->
                    <div class="empty-state">
                        <i class="fas fa-user-slash"></i>
                        <h3>No Administrators Found</h3>
                        <p>Add your first administrator to get started</p>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Admin Modal -->
    <div class="modal-overlay" id="adminModal">
        <div class="admin-modal">
            <div class="modal-header">
                <h3 id="modalTitle">Add New Admin</h3>
                <p id="modalDescription">Fill in the admin details below</p>
            </div>
            
            <form id="adminForm">
                <input type="hidden" id="adminId" name="admin_id">
                
                <div class="form-group">
                    <label for="adminName">Full Name *</label>
                    <input type="text" id="adminName" name="name" class="form-control" placeholder="Enter full name" required>
                    <div class="form-text">Enter the admin's full name</div>
                </div>
                
                <div class="form-group">
                    <label for="adminEmail">Email Address *</label>
                    <input type="email" id="adminEmail" name="email" class="form-control" placeholder="Enter email address" required>
                    <div class="form-text">This will be used for login</div>
                </div>
                
                <div class="form-group" id="passwordGroup">
                    <label for="adminPassword">Password *</label>
                    <input type="password" id="adminPassword" name="password" class="form-control" placeholder="Enter password">
                    <div class="form-text" id="passwordHelp">Minimum 6 characters</div>
                </div>
                
                <div class="form-group" id="confirmPasswordGroup">
                    <label for="confirmPassword">Confirm Password *</label>
                    <input type="password" id="confirmPassword" name="confirm_password" class="form-control" placeholder="Confirm password">
                </div>
                
                <div class="form-group">
                    <label for="adminRole">Role *</label>
                    <select id="adminRole" name="role" class="form-select" required>
                        <option value="moderator">Moderator</option>
                        <option value="root">Root Administrator</option>
                    </select>
                    <div class="form-text">Root has full access, moderator has limited access</div>
                </div>
                
                <div class="modal-actions">
                    <button type="button" class="modal-btn secondary" id="cancelBtn">Cancel</button>
                    <button type="submit" class="modal-btn primary" id="submitBtn">Save Admin</button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Confirm Delete Modal -->
    <div class="modal-overlay" id="confirmModal">
        <div class="admin-modal">
            <div class="modal-header">
                <h3>Confirm Deletion</h3>
                <p>This action cannot be undone</p>
            </div>
            
            <div class="form-group">
                <p>Are you sure you want to delete this administrator?</p>
                <p class="form-text">The admin will be permanently removed from the system</p>
            </div>
            
            <div class="modal-actions">
                <button type="button" class="modal-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button type="button" class="modal-btn danger" id="confirmDeleteBtn">Delete Admin</button>
            </div>
        </div>
    </div>
    
    <!-- Toast Notification -->
    <div class="toast" id="toastNotification">
        <div class="toast-icon">
            <i class="fas fa-info-circle"></i>
        </div>
        <div class="toast-content">
            <h4 id="toastTitle">Notification</h4>
            <p id="toastMessage">This is a notification message</p>
        </div>
        <button class="close-toast" id="closeToast">
            <i class="fas fa-times"></i>
        </button>
    </div>
    
    <script>
        // Global variables
        let currentAdminId = null;
        let isEditMode = false;
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Load admins
        async function loadAdmins() {
            try {
                const response = await fetch('get_admins.php');
                const data = await response.json();
                
                if (data.success) {
                    updateAdminsList(data.admins);
                } else {
                    showToast('Error', data.message || 'Failed to load admins', 'error');
                }
            } catch (error) {
                console.error('Error loading admins:', error);
                showToast('Error', 'Error loading admins', 'error');
            }
        }
        
        // Update admins list
        function updateAdminsList(admins) {
            const listContainer = document.getElementById('adminList');
            
            if (!admins || admins.length === 0) {
                listContainer.innerHTML = `
                    <div class="empty-state">
                        <i class="fas fa-user-slash"></i>
                        <h3>No Administrators Found</h3>
                        <p>Add your first administrator to get started</p>
                    </div>
                `;
                return;
            }
            
            let html = '';
            
            admins.forEach(admin => {
                const initials = getUserInitials(admin.name);
                const lastLogin = admin.last_login ? 
                    new Date(admin.last_login).toLocaleDateString('en-US', {
                        month: 'short',
                        day: 'numeric',
                        year: 'numeric',
                        hour: '2-digit',
                        minute: '2-digit'
                    }) : 'Never logged in';
                
                html += `
                    <div class="admin-item ${admin.role}" data-id="${admin.id}">
                        <div class="admin-info">
                            <div class="admin-avatar">${initials}</div>
                            <div class="admin-details">
                                <h4>${admin.name}</h4>
                                <p><i class="fas fa-envelope"></i> ${admin.email}</p>
                                <p><i class="fas fa-calendar-alt"></i> Joined: ${new Date(admin.created_at).toLocaleDateString()}</p>
                                <p><i class="fas fa-sign-in-alt"></i> Last login: ${lastLogin}</p>
                                <p>
                                    <i class="fas fa-shield-alt"></i> 
                                    <span class="role-badge role-${admin.role}">${admin.role}</span>
                                </p>
                            </div>
                        </div>
                        
                        <div class="admin-actions">
                            ${admin.id !== <?php echo $_SESSION['admin_id']; ?> ? `
                                <button class="edit-btn" onclick="editAdmin(${admin.id}, '${admin.name}', '${admin.email}', '${admin.role}')">
                                    <i class="fas fa-edit"></i> Edit
                                </button>
                                <button class="delete-btn" onclick="deleteAdmin(${admin.id}, '${admin.name}')">
                                    <i class="fas fa-trash"></i> Delete
                                </button>
                            ` : `
                                <span style="color: var(--gray); font-size: 14px; padding: 8px 16px;">
                                    <i class="fas fa-info-circle"></i> Current User
                                </span>
                            `}
                        </div>
                    </div>
                `;
            });
            
            listContainer.innerHTML = html;
        }
        
        // Get user initials
        function getUserInitials(name) {
            const names = name.split(' ');
            let initials = '';
            names.forEach(name => {
                if (name.trim()) {
                    initials += name.trim().charAt(0).toUpperCase();
                }
            });
            return initials.substring(0, 2);
        }
        
        // Show add admin modal
        function showAddModal() {
            isEditMode = false;
            document.getElementById('modalTitle').textContent = 'Add New Admin';
            document.getElementById('modalDescription').textContent = 'Fill in the admin details below';
            document.getElementById('submitBtn').textContent = 'Save Admin';
            document.getElementById('adminId').value = '';
            document.getElementById('adminForm').reset();
            
            // Show password fields for new admin
            document.getElementById('passwordGroup').style.display = 'block';
            document.getElementById('confirmPasswordGroup').style.display = 'block';
            document.getElementById('adminPassword').required = true;
            document.getElementById('confirmPassword').required = true;
            
            document.getElementById('adminModal').classList.add('active');
        }
        
        // Show edit admin modal
        function editAdmin(id, name, email, role) {
            isEditMode = true;
            currentAdminId = id;
            
            document.getElementById('modalTitle').textContent = 'Edit Admin';
            document.getElementById('modalDescription').textContent = 'Update admin details';
            document.getElementById('submitBtn').textContent = 'Update Admin';
            document.getElementById('adminId').value = id;
            document.getElementById('adminName').value = name;
            document.getElementById('adminEmail').value = email;
            document.getElementById('adminRole').value = role;
            
            // Hide password fields for edit (optional)
            document.getElementById('passwordGroup').style.display = 'block';
            document.getElementById('confirmPasswordGroup').style.display = 'block';
            document.getElementById('adminPassword').required = false;
            document.getElementById('confirmPassword').required = false;
            document.getElementById('adminPassword').placeholder = 'Leave blank to keep current password';
            document.getElementById('confirmPassword').placeholder = 'Leave blank to keep current password';
            document.getElementById('passwordHelp').textContent = 'Leave blank to keep current password';
            
            document.getElementById('adminModal').classList.add('active');
        }
        
        // Show delete confirmation
        function deleteAdmin(id, name) {
            currentAdminId = id;
            
            document.getElementById('confirmModal').classList.add('active');
        }
        
        // Show toast notification
        function showToast(title, message, type = 'info', duration = 3000) {
            const toast = document.getElementById('toastNotification');
            const toastTitle = document.getElementById('toastTitle');
            const toastMessage = document.getElementById('toastMessage');
            const toastIcon = toast.querySelector('.toast-icon i');
            
            toastTitle.textContent = title;
            toastMessage.textContent = message;
            
            // Set type
            toast.className = 'toast ' + type;
            switch(type) {
                case 'success':
                    toastIcon.className = 'fas fa-check-circle';
                    break;
                case 'error':
                    toastIcon.className = 'fas fa-times-circle';
                    break;
                case 'warning':
                    toastIcon.className = 'fas fa-exclamation-triangle';
                    break;
                case 'info':
                    toastIcon.className = 'fas fa-info-circle';
                    break;
            }
            
            toast.classList.add('active');
            
            // Auto hide
            setTimeout(() => {
                toast.classList.remove('active');
            }, duration);
        }
        
        // Form submission
        document.getElementById('adminForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            
            // Validate passwords if in add mode or if password is provided
            if (!isEditMode || formData.get('password')) {
                const password = formData.get('password');
                const confirmPassword = formData.get('confirm_password');
                
                if (password.length < 6) {
                    showToast('Error', 'Password must be at least 6 characters', 'error');
                    return;
                }
                
                if (password !== confirmPassword) {
                    showToast('Error', 'Passwords do not match', 'error');
                    return;
                }
            }
            
            try {
                // Add action type
                formData.append('action', isEditMode ? 'update' : 'add');
                
                const response = await fetch('b_admin.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showToast('Success', data.message, 'success');
                    document.getElementById('adminModal').classList.remove('active');
                    loadAdmins();
                } else {
                    showToast('Error', data.message || 'Operation failed', 'error');
                }
            } catch (error) {
                console.error('Error saving admin:', error);
                showToast('Error', 'Error saving admin', 'error');
            }
        });
        
        // Delete admin
        document.getElementById('confirmDeleteBtn').addEventListener('click', async () => {
            try {
                const formData = new FormData();
                formData.append('action', 'delete');
                formData.append('admin_id', currentAdminId);
                
                const response = await fetch('b_admin.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    showToast('Success', data.message, 'success');
                    document.getElementById('confirmModal').classList.remove('active');
                    loadAdmins();
                } else {
                    showToast('Error', data.message || 'Delete failed', 'error');
                }
            } catch (error) {
                console.error('Error deleting admin:', error);
                showToast('Error', 'Error deleting admin', 'error');
            }
        });
        
        // Event listeners
        document.getElementById('addAdminBtn').addEventListener('click', showAddModal);
        
        document.getElementById('cancelBtn').addEventListener('click', () => {
            document.getElementById('adminModal').classList.remove('active');
        });
        
        document.getElementById('confirmCancelBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
        });
        
        document.getElementById('closeToast').addEventListener('click', () => {
            document.getElementById('toastNotification').classList.remove('active');
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                }
            });
        });
        
        // Initial load
        loadAdmins();
    </script>
</body>
</html>