<?php
// b_admin.php
session_start();

// Check if user is logged in and is root admin
if (!isset($_SESSION['admin_id']) || $_SESSION['admin_role'] !== 'root') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

header('Content-Type: application/json');

try {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'add':
            addAdmin();
            break;
            
        case 'update':
            updateAdmin();
            break;
            
        case 'delete':
            deleteAdmin();
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action']);
            exit;
    }
} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
    exit;
}

function addAdmin() {
    global $pdo;
    
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'moderator';
    
    // Validation
    if (empty($name) || empty($email) || empty($password)) {
        echo json_encode(['success' => false, 'message' => 'All fields are required']);
        exit;
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Invalid email format']);
        exit;
    }
    
    if (strlen($password) < 6) {
        echo json_encode(['success' => false, 'message' => 'Password must be at least 6 characters']);
        exit;
    }
    
    // Check if email already exists
    $stmt = $pdo->prepare("SELECT id FROM admin WHERE email = ?");
    $stmt->execute([$email]);
    
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Email already exists']);
        exit;
    }
    
    // Hash password
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Insert new admin
    $stmt = $pdo->prepare("INSERT INTO admin (name, email, passkey, role, created_at) VALUES (?, ?, ?, ?, NOW())");
    $stmt->execute([$name, $email, $hashedPassword, $role]);
    
    echo json_encode(['success' => true, 'message' => 'Admin added successfully']);
    exit;
}

function updateAdmin() {
    global $pdo;
    
    $admin_id = intval($_POST['admin_id'] ?? 0);
    $name = trim($_POST['name'] ?? '');
    $email = trim($_POST['email'] ?? '');
    $password = $_POST['password'] ?? '';
    $role = $_POST['role'] ?? 'moderator';
    
    if ($admin_id == $_SESSION['admin_id']) {
        echo json_encode(['success' => false, 'message' => 'You cannot edit your own account from here']);
        exit;
    }
    
    // Validation
    if (empty($name) || empty($email)) {
        echo json_encode(['success' => false, 'message' => 'Name and email are required']);
        exit;
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        echo json_encode(['success' => false, 'message' => 'Invalid email format']);
        exit;
    }
    
    if (!empty($password) && strlen($password) < 6) {
        echo json_encode(['success' => false, 'message' => 'Password must be at least 6 characters']);
        exit;
    }
    
    // Check if email already exists for another admin
    $stmt = $pdo->prepare("SELECT id FROM admin WHERE email = ? AND id != ?");
    $stmt->execute([$email, $admin_id]);
    
    if ($stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Email already exists']);
        exit;
    }
    
    // Update admin with or without password
    if (!empty($password)) {
        $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE admin SET name = ?, email = ?, passkey = ?, role = ? WHERE id = ?");
        $stmt->execute([$name, $email, $hashedPassword, $role, $admin_id]);
    } else {
        $stmt = $pdo->prepare("UPDATE admin SET name = ?, email = ?, role = ? WHERE id = ?");
        $stmt->execute([$name, $email, $role, $admin_id]);
    }
    
    echo json_encode(['success' => true, 'message' => 'Admin updated successfully']);
    exit;
}

function deleteAdmin() {
    global $pdo;
    
    $admin_id = intval($_POST['admin_id'] ?? 0);
    
    if ($admin_id == $_SESSION['admin_id']) {
        echo json_encode(['success' => false, 'message' => 'You cannot delete your own account']);
        exit;
    }
    
    // Check if admin exists
    $stmt = $pdo->prepare("SELECT id FROM admin WHERE id = ?");
    $stmt->execute([$admin_id]);
    
    if (!$stmt->fetch()) {
        echo json_encode(['success' => false, 'message' => 'Admin not found']);
        exit;
    }
    
    // Delete admin
    $stmt = $pdo->prepare("DELETE FROM admin WHERE id = ?");
    $stmt->execute([$admin_id]);
    
    echo json_encode(['success' => true, 'message' => 'Admin deleted successfully']);
    exit;
}
?>