<?php
// b_bonus.php
session_start();

// Set headers first
header('Content-Type: application/json; charset=utf-8');

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle GET requests
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
    
    try {
        switch ($action) {
            case 'search_users':
                $query = $_GET['query'] ?? '';
                
                if (strlen($query) < 3) {
                    echo json_encode(['success' => false, 'message' => 'Please enter at least 3 characters']);
                    exit;
                }
                
                $searchQuery = '%' . $query . '%';
                
                $stmt = $pdo->prepare("
                    SELECT user_id, username, first_name, last_name, email, phone_number, 
                           account_status, verification, profile_url
                    FROM users 
                    WHERE first_name LIKE :query 
                       OR last_name LIKE :query 
                       OR email LIKE :query 
                       OR phone_number LIKE :query
                    ORDER BY first_name, last_name
                    LIMIT 50
                ");
                $stmt->bindParam(':query', $searchQuery, PDO::PARAM_STR);
                $stmt->execute();
                $users = $stmt->fetchAll(PDO::FETCH_ASSOC);
                
                echo json_encode(['success' => true, 'users' => $users]);
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
                break;
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        echo json_encode(['success' => false, 'message' => 'Invalid JSON input']);
        exit;
    }
    
    $action = $input['action'] ?? '';
    
    try {
        switch ($action) {
            case 'credit_user':
                $user_id = $input['user_id'] ?? 0;
                $amount = $input['amount'] ?? 0;
                
                // Validate inputs
                if (!$user_id || $user_id <= 0) {
                    echo json_encode(['success' => false, 'message' => 'Invalid user ID']);
                    exit;
                }
                
                if (!$amount || $amount <= 0) {
                    echo json_encode(['success' => false, 'message' => 'Invalid amount']);
                    exit;
                }
                
                if ($amount > 1000000) {
                    echo json_encode(['success' => false, 'message' => 'Maximum amount is ₦1,000,000']);
                    exit;
                }
                
                // Start transaction
                $pdo->beginTransaction();
                
                try {
                    // Get the first approved buy order for the user
                    $stmt = $pdo->prepare("
                        SELECT id, interest 
                        FROM buyer_orders 
                        WHERE user_id = :user_id 
                        AND status = 'approved'
                        ORDER BY id ASC 
                        LIMIT 1
                    ");
                    $stmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $stmt->execute();
                    $buyOrder = $stmt->fetch(PDO::FETCH_ASSOC);
                    
                    if (!$buyOrder) {
                        throw new Exception('User does not have any approved buy order');
                    }
                    
                    // Update the interest column in buy order
                    $new_interest = $buyOrder['interest'] + $amount;
                    
                    $updateStmt = $pdo->prepare("
                        UPDATE buyer_orders 
                        SET interest = :new_interest, 
                            updated_at = CURRENT_TIMESTAMP 
                        WHERE id = :order_id
                    ");
                    $updateStmt->bindParam(':new_interest', $new_interest);
                    $updateStmt->bindParam(':order_id', $buyOrder['id'], PDO::PARAM_INT);
                    
                    if (!$updateStmt->execute()) {
                        throw new Exception('Failed to update buy order');
                    }
                    
                    // Commit transaction
                    $pdo->commit();
                    
                    // Get user details for response
                    $userStmt = $pdo->prepare("
                        SELECT first_name, last_name 
                        FROM users 
                        WHERE user_id = :user_id
                    ");
                    $userStmt->bindParam(':user_id', $user_id, PDO::PARAM_INT);
                    $userStmt->execute();
                    $user = $userStmt->fetch(PDO::FETCH_ASSOC);
                    
                    $formattedAmount = number_format($amount, 2);
                    $message = "Successfully credited ₦{$formattedAmount} to {$user['first_name']} {$user['last_name']}'s interest";
                    
                    echo json_encode([
                        'success' => true, 
                        'message' => $message,
                        'data' => [
                            'user_id' => $user_id,
                            'amount_credited' => $amount,
                            'order_id' => $buyOrder['id'],
                            'new_interest' => $new_interest
                        ]
                    ]);
                    
                } catch (Exception $e) {
                    // Rollback transaction on error
                    $pdo->rollBack();
                    throw $e;
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action']);
                break;
        }
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

// If no valid method or action
echo json_encode(['success' => false, 'message' => 'Invalid request']);
?>