<?php
// Start session
session_start();

// Include database connection
require_once 'php/db.php';

// Set content type to JSON
header('Content-Type: application/json');

// Check if request is POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    echo json_encode([
        'success' => false,
        'message' => 'Invalid request method.'
    ]);
    exit;
}

// Get and sanitize input
$email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
$passkey = $_POST['passkey'] ?? '';

// Validate input
if (empty($email) || empty($passkey)) {
    echo json_encode([
        'success' => false,
        'message' => 'Please fill in all fields.'
    ]);
    exit;
}

if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
    echo json_encode([
        'success' => false,
        'message' => 'Please enter a valid email address.'
    ]);
    exit;
}

try {
    // Prepare SQL statement to find admin by email
    $stmt = $pdo->prepare("SELECT id, name, email, passkey, role FROM admin WHERE email = ?");
    $stmt->execute([$email]);
    $admin = $stmt->fetch();
    
    if (!$admin) {
        // Admin not found
        echo json_encode([
            'success' => false,
            'message' => 'Invalid credentials.'
        ]);
        exit;
    }
    
    // Verify password
    if (password_verify($passkey, $admin['passkey'])) {
        // Get user's IP address
        $ip_address = $_SERVER['HTTP_CLIENT_IP'] ?? 
                     $_SERVER['HTTP_X_FORWARDED_FOR'] ?? 
                     $_SERVER['REMOTE_ADDR'] ?? 
                     '0.0.0.0';
        
        // Update last login and IP address
        $updateStmt = $pdo->prepare("UPDATE admin SET last_login = NOW(), ip_address = ? WHERE id = ?");
        $updateStmt->execute([$ip_address, $admin['id']]);
        
        // Start session with admin data
        $_SESSION['admin_id'] = $admin['id'];
        $_SESSION['admin_role'] = $admin['role'];
        $_SESSION['admin_name'] = $admin['name'];
        $_SESSION['admin_email'] = $admin['email'];
        
        // Set session timeout (8 hours)
        $_SESSION['last_activity'] = time();
        
        echo json_encode([
            'success' => true,
            'message' => 'Login successful!',
            'user' => [
                'id' => $admin['id'],
                'name' => $admin['name'],
                'role' => $admin['role']
            ]
        ]);
    } else {
        // Invalid password
        echo json_encode([
            'success' => false,
            'message' => 'Invalid credentials.'
        ]);
    }
    
} catch (PDOException $e) {
    // Log error for debugging
    error_log("Login error: " . $e->getMessage());
    
    echo json_encode([
        'success' => false,
        'message' => 'An error occurred. Please try again later.'
    ]);
    exit;
}
?>