<?php
// b_orders.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit;
}

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}

// Function to send API request silently
function sendRefundApiRequest($buyer_user_id) {
    $url = 'https://biddersconnectsafrica.com/auto_refund.php';
    $data = [
        'user_id' => $buyer_user_id,
        'message' => 'users need refund'
    ];
    
    $options = [
        'http' => [
            'header'  => "Content-Type: application/json\r\n",
            'method'  => 'POST',
            'content' => json_encode($data),
            'timeout' => 5,
            'ignore_errors' => true
        ]
    ];
    
    $context = stream_context_create($options);
    @file_get_contents($url, false, $context);
}

require_once 'php/db.php';

// Get action from request
$action = '';
if ($_SERVER['REQUEST_METHOD'] === 'GET') {
    $action = $_GET['action'] ?? '';
} else if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (!empty($_POST['action'])) {
        $action = $_POST['action'];
    } else {
        $json_input = file_get_contents('php://input');
        if (!empty($json_input)) {
            $input_data = json_decode($json_input, true);
            $action = $input_data['action'] ?? '';
        }
    }
}

header('Content-Type: application/json');

try {
    switch ($action) {
        case 'get_buyer_order_details':
            getBuyerOrderDetails();
            break;
            
        case 'get_seller_order_details':
            getSellerOrderDetails();
            break;
            
        case 'load_more_orders':
            loadMoreOrders();
            break;
            
        case 'approve_payment':
            approvePayment();
            break;
            
        case 'toggle_seller_order_status':
            toggleSellerOrderStatus();
            break;
            
        case 'search_orders':
            searchOrders();
            break;
            
        case 'check_new_orders':
            checkNewOrders();
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Invalid action: ' . $action]);
    }
} catch (PDOException $e) {
    echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
}

function getBuyerOrderDetails() {
    global $pdo;
    
    $orderId = $_GET['order_id'] ?? 0;
    
    if (!$orderId) {
        echo json_encode(['success' => false, 'message' => 'Order ID is required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT bo.*, 
               u.username, u.first_name, u.last_name, u.email, u.phone_number, 
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at,
               COALESCE(k.email_status, 'not verified') as email_verified,
               COALESCE(k.phone_status, 'not verified') as phone_verified
        FROM buyer_orders bo
        JOIN users u ON bo.user_id = u.user_id
        LEFT JOIN basic_kyc k ON u.user_id = k.user_id
        WHERE bo.id = :order_id
    ");
    $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
    $stmt->execute();
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT o.*, 
               u.first_name as seller_first_name, u.last_name as seller_last_name,
               u.email as seller_email, u.phone_number as seller_phone,
               so.user_id as seller_user_id
        FROM orders o
        JOIN seller_orders so ON o.seller_id = so.id
        JOIN users u ON so.user_id = u.user_id
        WHERE o.buyer_id = :order_id
        ORDER BY o.created_at DESC
    ");
    $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
    $stmt->execute();
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $formattedOrders = [];
    foreach ($orders as $orderItem) {
        $bankStmt = $pdo->prepare("
            SELECT ba.bank_name, ba.account_number, ba.account_name
            FROM bank_accounts ba
            WHERE ba.user_id = :seller_user_id
            ORDER BY ba.created_at DESC
            LIMIT 1
        ");
        $bankStmt->bindParam(':seller_user_id', $orderItem['seller_user_id'], PDO::PARAM_INT);
        $bankStmt->execute();
        $seller_bank = $bankStmt->fetch(PDO::FETCH_ASSOC);
        
        $formattedOrders[] = [
            'id' => $orderItem['id'],
            'order_id' => $orderItem['id'],
            'amount' => $orderItem['amount'],
            'status' => $orderItem['status'],
            'payer_status' => $orderItem['payer_status'],
            'payment_proof' => $orderItem['payment_proof'],
            'seller_name' => $orderItem['seller_first_name'] . ' ' . $orderItem['seller_last_name'],
            'seller_email' => $orderItem['seller_email'],
            'seller_phone' => $orderItem['seller_phone'],
            'seller_id' => $orderItem['seller_user_id'],
            'buyer_id' => $orderItem['buyer_id'],
            'created_at' => $orderItem['created_at'],
            'seller_bank' => $seller_bank ?: null
        ];
    }
    
    $userData = [
        'username' => $order['username'],
        'first_name' => $order['first_name'],
        'last_name' => $order['last_name'],
        'email' => $order['email'],
        'phone_number' => $order['phone_number'],
        'profile_url' => $order['profile_url'],
        'account_status' => $order['account_status'],
        'verification' => $order['verification'],
        'country' => $order['country'],
        'dob' => $order['dob'],
        'created_at' => $order['created_at'],
        'email_verified' => $order['email_verified'],
        'phone_verified' => $order['phone_verified'],
        'initials' => getUserInitials($order['first_name'] . ' ' . $order['last_name'])
    ];
    
    echo json_encode([
        'success' => true,
        'order' => [
            'id' => $order['id'],
            'amount' => $order['amount'],
            'amount_remaining' => $order['amount_remaining'],
            'interest' => $order['interest'],
            'status' => $order['status'],
            'match_status' => $order['match_status'],
            'created_at' => $order['created_at'],
            'updated_at' => $order['updated_at'],
            'user_id' => $order['user_id']
        ],
        'user' => $userData,
        'orders' => $formattedOrders
    ]);
}

function getSellerOrderDetails() {
    global $pdo;
    
    $orderId = $_GET['order_id'] ?? 0;
    
    if (!$orderId) {
        echo json_encode(['success' => false, 'message' => 'Order ID is required']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT so.*, 
               u.username, u.first_name, u.last_name, u.email, u.phone_number,
               u.profile_url, u.account_status, u.verification, u.country, u.dob, u.created_at,
               COALESCE(k.email_status, 'not verified') as email_verified,
               COALESCE(k.phone_status, 'not verified') as phone_verified
        FROM seller_orders so
        JOIN users u ON so.user_id = u.user_id
        LEFT JOIN basic_kyc k ON u.user_id = k.user_id
        WHERE so.id = :order_id
    ");
    $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
    $stmt->execute();
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        return;
    }
    
    $stmt = $pdo->prepare("
        SELECT o.*, 
               u.first_name as buyer_first_name, u.last_name as buyer_last_name,
               u.email as buyer_email, u.phone_number as buyer_phone,
               bo.user_id as buyer_user_id
        FROM orders o
        JOIN buyer_orders bo ON o.buyer_id = bo.id
        JOIN users u ON bo.user_id = u.user_id
        WHERE o.seller_id = :order_id
        ORDER BY o.created_at DESC
    ");
    $stmt->bindParam(':order_id', $orderId, PDO::PARAM_INT);
    $stmt->execute();
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $formattedOrders = [];
    foreach ($orders as $orderItem) {
        $formattedOrders[] = [
            'id' => $orderItem['id'],
            'order_id' => $orderItem['id'],
            'amount' => $orderItem['amount'],
            'status' => $orderItem['status'],
            'payer_status' => $orderItem['payer_status'],
            'payment_proof' => $orderItem['payment_proof'],
            'buyer_name' => $orderItem['buyer_first_name'] . ' ' . $orderItem['buyer_last_name'],
            'buyer_email' => $orderItem['buyer_email'],
            'buyer_phone' => $orderItem['buyer_phone'],
            'buyer_id' => $orderItem['buyer_user_id'],
            'seller_id' => $orderItem['seller_id'],
            'created_at' => $orderItem['created_at']
        ];
    }
    
    $userData = [
        'username' => $order['username'],
        'first_name' => $order['first_name'],
        'last_name' => $order['last_name'],
        'email' => $order['email'],
        'phone_number' => $order['phone_number'],
        'profile_url' => $order['profile_url'],
        'account_status' => $order['account_status'],
        'verification' => $order['verification'],
        'country' => $order['country'],
        'dob' => $order['dob'],
        'created_at' => $order['created_at'],
        'email_verified' => $order['email_verified'],
        'phone_verified' => $order['phone_verified'],
        'initials' => getUserInitials($order['first_name'] . ' ' . $order['last_name'])
    ];
    
    echo json_encode([
        'success' => true,
        'order' => [
            'id' => $order['id'],
            'amount' => $order['amount'],
            'amount_remaining' => $order['amount_remaining'],
            'status' => $order['status'],
            'match_status' => $order['match_status'],
            'created_at' => $order['created_at'],
            'updated_at' => $order['updated_at'],
            'user_id' => $order['user_id']
        ],
        'user' => $userData,
        'orders' => $formattedOrders
    ]);
}

function loadMoreOrders() {
    global $pdo;
    
    $type = $_GET['type'] ?? 'buyer';
    $offset = intval($_GET['offset'] ?? 0);
    $limit = 10; // Load 10 orders at a time
    
    if ($type === 'buyer') {
        $stmt = $pdo->prepare("
            SELECT bo.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM buyer_orders bo 
            JOIN users u ON bo.user_id = u.user_id 
            WHERE bo.status != 'completed'
            ORDER BY bo.created_at DESC 
            LIMIT :offset, :limit
        ");
    } else {
        $stmt = $pdo->prepare("
            SELECT so.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM seller_orders so 
            JOIN users u ON so.user_id = u.user_id 
            WHERE so.status != 'completed'
            ORDER BY so.created_at DESC 
            LIMIT :offset, :limit
        ");
    }
    
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->execute();
    $orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'orders' => $orders,
        'offset' => $offset,
        'limit' => $limit
    ]);
}

function approvePayment() {
    global $pdo;
    
    $input_data = [];
    if (!empty($_POST)) {
        $input_data = $_POST;
    } else {
        $json_input = file_get_contents('php://input');
        if (!empty($json_input)) {
            $input_data = json_decode($json_input, true);
        }
    }
    
    $order_match_id = $input_data['order_id'] ?? 0;
    $buyer_id = $input_data['buyer_id'] ?? 0;
    
    if (!$order_match_id || !$buyer_id) {
        echo json_encode(['success' => false, 'message' => 'Order ID and Buyer ID required']);
        return;
    }
    
    $pdo->beginTransaction();
    
    try {
        $stmt = $pdo->prepare("
            SELECT o.id, o.seller_id, o.status 
            FROM orders o
            WHERE o.id = :order_match_id 
            AND o.buyer_id = :buyer_id
        ");
        $stmt->bindParam(':order_match_id', $order_match_id);
        $stmt->bindParam(':buyer_id', $buyer_id);
        $stmt->execute();
        $order_info = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$order_info) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Order not found']);
            return;
        }
        
        if ($order_info['status'] === 'approved') {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Payment already approved']);
            return;
        }
        
        $seller_id = $order_info['seller_id'];
        
        $stmt = $pdo->prepare("
            UPDATE orders 
            SET status = 'approved', 
                updated_at = CURRENT_TIMESTAMP 
            WHERE id = :order_match_id 
            AND buyer_id = :buyer_id
        ");
        $stmt->bindParam(':order_match_id', $order_match_id);
        $stmt->bindParam(':buyer_id', $buyer_id);
        $stmt->execute();
        
        if ($stmt->rowCount() === 0) {
            $pdo->rollBack();
            echo json_encode(['success' => false, 'message' => 'Failed to approve payment']);
            return;
        }
        
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as pending_count 
            FROM orders 
            WHERE buyer_id = :buyer_id 
            AND status = 'not approved'
        ");
        $stmt->bindParam(':buyer_id', $buyer_id);
        $stmt->execute();
        $pending_count = $stmt->fetch(PDO::FETCH_ASSOC)['pending_count'];
        
        if ($pending_count == 0) {
            $mature_at = date('Y-m-d H:i:s', strtotime('+14 days'));
            
            $stmt = $pdo->prepare("
                SELECT bo.user_id, bo.amount 
                FROM buyer_orders bo 
                WHERE bo.id = :buyer_id
            ");
            $stmt->bindParam(':buyer_id', $buyer_id);
            $stmt->execute();
            $buyer_details = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $buyer_user_id = $buyer_details['user_id'];
            $buyer_amount = floatval($buyer_details['amount']);
            
            $stmt = $pdo->prepare("
                UPDATE buyer_orders 
                SET status = 'approved',
                    mature_at = :mature_at,
                    updated_at = CURRENT_TIMESTAMP
                WHERE id = :buyer_id
            ");
            $stmt->bindParam(':buyer_id', $buyer_id);
            $stmt->bindParam(':mature_at', $mature_at);
            $stmt->execute();
            
            $stmt = $pdo->prepare("
                SELECT id, status, buy_id 
                FROM recommit 
                WHERE primary_buy_id = :buyer_id
                ORDER BY created_at DESC
                LIMIT 1
            ");
            $stmt->bindParam(':buyer_id', $buyer_id);
            $stmt->execute();
            $recommit_record = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($recommit_record) {
                $stmt = $pdo->prepare("
                    UPDATE recommit 
                    SET status = 'successful',
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = :recommit_id
                ");
                $stmt->bindParam(':recommit_id', $recommit_record['id']);
                $stmt->execute();
            }
            
            $stmt = $pdo->prepare("
                SELECT COUNT(*) as total_buy_orders 
                FROM buyer_orders 
                WHERE user_id = :buyer_user_id 
                AND status = 'approved'
            ");
            $stmt->bindParam(':buyer_user_id', $buyer_user_id);
            $stmt->execute();
            $total_approved_orders = $stmt->fetch(PDO::FETCH_ASSOC)['total_buy_orders'];
            
            if ($total_approved_orders == 1) {
                $stmt = $pdo->prepare("
                    SELECT id, referrer_user_id, commission 
                    FROM referrals 
                    WHERE referred_user_id = :buyer_user_id
                ");
                $stmt->bindParam(':buyer_user_id', $buyer_user_id);
                $stmt->execute();
                $referral_record = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($referral_record && $referral_record['commission'] == 'pending') {
                    $referral_id = $referral_record['id'];
                    $referrer_user_id = $referral_record['referrer_user_id'];
                    
                    $stmt = $pdo->prepare("
                        UPDATE referrals 
                        SET commission = 'paid' 
                        WHERE id = :referral_id
                    ");
                    $stmt->bindParam(':referral_id', $referral_id);
                    $stmt->execute();
                    
                    $commission_amount = $buyer_amount * 0.05;
                    
                    $stmt = $pdo->prepare("
                        UPDATE referral_link 
                        SET balance = balance + :commission_amount 
                        WHERE user_id = :referrer_user_id
                    ");
                    $stmt->bindParam(':commission_amount', $commission_amount);
                    $stmt->bindParam(':referrer_user_id', $referrer_user_id);
                    $stmt->execute();
                }
                
                $stmt = $pdo->prepare("
                    SELECT id, status 
                    FROM security_deposit 
                    WHERE user_id = :buyer_user_id 
                    ORDER BY created_at DESC 
                    LIMIT 1
                ");
                $stmt->bindParam(':buyer_user_id', $buyer_user_id);
                $stmt->execute();
                $security_deposit = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($security_deposit && $security_deposit['status'] == 'paid') {
                    sendRefundApiRequest($buyer_user_id);
                }
            }
        }
        
        $stmt = $pdo->prepare("
            SELECT so.amount_remaining, 
                   (SELECT COUNT(*) FROM orders o2 WHERE o2.seller_id = :seller_id AND o2.status = 'not approved') as pending_seller_orders
            FROM seller_orders so
            WHERE so.id = :seller_id
        ");
        $stmt->bindParam(':seller_id', $seller_id);
        $stmt->execute();
        $seller_info = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($seller_info) {
            $amount_remaining = floatval($seller_info['amount_remaining']);
            $pending_seller_orders = intval($seller_info['pending_seller_orders']);
            
            if ($amount_remaining == 0 && $pending_seller_orders == 0) {
                $stmt = $pdo->prepare("
                    UPDATE seller_orders 
                    SET status = 'sold',
                        updated_at = CURRENT_TIMESTAMP
                    WHERE id = :seller_id
                ");
                $stmt->bindParam(':seller_id', $seller_id);
                $stmt->execute();
            }
        }
        
        $pdo->commit();
        
        echo json_encode([
            'success' => true,
            'message' => 'Payment approved successfully'
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
    }
}

function toggleSellerOrderStatus() {
    global $pdo;
    
    $input_data = [];
    if (!empty($_POST)) {
        $input_data = $_POST;
    } else {
        $json_input = file_get_contents('php://input');
        if (!empty($json_input)) {
            $input_data = json_decode($json_input, true);
        }
    }
    
    $order_id = $input_data['order_id'] ?? 0;
    
    if (!$order_id) {
        echo json_encode(['success' => false, 'message' => 'Order ID is required']);
        return;
    }
    
    $stmt = $pdo->prepare("SELECT status FROM seller_orders WHERE id = :order_id");
    $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
    $stmt->execute();
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        return;
    }
    
    $new_status = ($order['status'] === 'pause') ? 'in progress' : 'pause';
    
    $stmt = $pdo->prepare("UPDATE seller_orders SET status = :status WHERE id = :order_id");
    $stmt->bindParam(':status', $new_status);
    $stmt->bindParam(':order_id', $order_id, PDO::PARAM_INT);
    $stmt->execute();
    
    echo json_encode(['success' => true, 'message' => 'Order status updated', 'new_status' => $new_status]);
}

function searchOrders() {
    global $pdo;
    
    $search_id = $_GET['search_id'] ?? 0;
    $search_type = $_GET['search_type'] ?? 'buyer';
    
    if (!$search_id) {
        echo json_encode(['success' => false, 'message' => 'Please enter an order ID to search']);
        return;
    }
    
    if ($search_type === 'buyer') {
        $stmt = $pdo->prepare("
            SELECT bo.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM buyer_orders bo 
            JOIN users u ON bo.user_id = u.user_id 
            WHERE bo.id = :search_id
        ");
    } else {
        $stmt = $pdo->prepare("
            SELECT so.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM seller_orders so 
            JOIN users u ON so.user_id = u.user_id 
            WHERE so.id = :search_id
        ");
    }
    
    $stmt->bindParam(':search_id', $search_id, PDO::PARAM_INT);
    $stmt->execute();
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        return;
    }
    
    echo json_encode([
        'success' => true,
        'order' => $order,
        'type' => $search_type
    ]);
}

function checkNewOrders() {
    global $pdo;
    
    $last_buyer_id = intval($_GET['last_buyer_id'] ?? 0);
    $last_seller_id = intval($_GET['last_seller_id'] ?? 0);
    
    $new_buyer_orders = [];
    $new_seller_orders = [];
    
    // Check for new buyer orders
    if ($last_buyer_id > 0) {
        $stmt = $pdo->prepare("
            SELECT bo.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM buyer_orders bo 
            JOIN users u ON bo.user_id = u.user_id 
            WHERE bo.status != 'completed'
            AND bo.id > :last_buyer_id
            ORDER BY bo.created_at DESC
        ");
        $stmt->bindParam(':last_buyer_id', $last_buyer_id, PDO::PARAM_INT);
        $stmt->execute();
        $new_buyer_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    // Check for new seller orders
    if ($last_seller_id > 0) {
        $stmt = $pdo->prepare("
            SELECT so.*, u.username, u.first_name, u.last_name, u.email, u.phone_number, u.profile_url
            FROM seller_orders so 
            JOIN users u ON so.user_id = u.user_id 
            WHERE so.status != 'completed'
            AND so.id > :last_seller_id
            ORDER BY so.created_at DESC
        ");
        $stmt->bindParam(':last_seller_id', $last_seller_id, PDO::PARAM_INT);
        $stmt->execute();
        $new_seller_orders = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
    
    echo json_encode([
        'success' => true,
        'new_buyer_orders' => $new_buyer_orders,
        'new_seller_orders' => $new_seller_orders,
        'new_buyer_count' => count($new_buyer_orders),
        'new_seller_count' => count($new_seller_orders)
    ]);
}
?>