<?php
// b_set.php
session_start();

// Check if user is logged in as admin
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Set content type to JSON
header('Content-Type: application/json');

// Get POST data
$order_type = $_POST['order_type'] ?? '';
$order_id = $_POST['order_id'] ?? '';
$amount = $_POST['amount'] ?? '';
$transaction_type = $_POST['transaction_type'] ?? 'credit';
$status = $_POST['status'] ?? '';

// Validate required fields
if (empty($order_type) || empty($order_id) || empty($status)) {
    echo json_encode(['success' => false, 'message' => 'Order type, order ID, and status are required']);
    exit;
}

// Validate order type
if (!in_array($order_type, ['buyer', 'seller'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid order type']);
    exit;
}

// Validate transaction type
if (!in_array($transaction_type, ['credit', 'debit'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid transaction type']);
    exit;
}

try {
    // Start transaction
    $pdo->beginTransaction();
    
    // Check if order exists
    if ($order_type === 'buyer') {
        $table = 'buyer_orders';
        // Validate buyer status
        $valid_statuses = ['approved', 'not_approved', 'pending', 'pause', 'sold'];
        if (!in_array($status, $valid_statuses)) {
            throw new Exception('Invalid status for buyer order');
        }
    } else {
        $table = 'seller_orders';
        // Validate seller status
        $valid_statuses = ['in progress', 'sold', 'pause'];
        if (!in_array($status, $valid_statuses)) {
            throw new Exception('Invalid status for seller order');
        }
    }
    
    // Get current order data
    $stmt = $pdo->prepare("SELECT * FROM $table WHERE id = ?");
    $stmt->execute([$order_id]);
    $order = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        throw new Exception("Order #$order_id not found");
    }
    
    // Initialize update query
    $update_query = "UPDATE $table SET status = ?";
    $params = [$status];
    
    // Handle amount update if provided
    if (!empty($amount) && is_numeric($amount) && $amount > 0) {
        $amount = (float) $amount;
        
        if ($order_type === 'buyer') {
            // For buyer orders, update amount column
            if ($transaction_type === 'credit') {
                // Credit: add to amount column
                $update_query .= ", amount = amount + ?";
                $params[] = $amount;
            } else {
                // Debit: subtract from amount column
                // Check if debit amount is not greater than current amount
                if ($amount > $order['amount']) {
                    throw new Exception("Debit amount cannot be greater than current amount");
                }
                $update_query .= ", amount = amount - ?";
                $params[] = $amount;
            }
        } else {
            // For seller orders, update amount_remaining column
            if ($transaction_type === 'credit') {
                // Credit: add to amount_remaining column
                $update_query .= ", amount_remaining = amount_remaining + ?";
                $params[] = $amount;
            } else {
                // Debit: subtract from amount_remaining column
                // Check if debit amount is not greater than current amount_remaining
                if ($amount > $order['amount_remaining']) {
                    throw new Exception("Debit amount cannot be greater than current amount remaining");
                }
                $update_query .= ", amount_remaining = amount_remaining - ?";
                $params[] = $amount;
            }
        }
    }
    
    // If updating seller order, set match_status to 'pending'
    if ($order_type === 'seller') {
        $update_query .= ", match_status = 'pending'";
    }
    
    // Add updated_at timestamp
    $update_query .= ", updated_at = NOW() WHERE id = ?";
    $params[] = $order_id;
    
    // Execute update
    $update_stmt = $pdo->prepare($update_query);
    $update_stmt->execute($params);
    
    // Commit transaction
    $pdo->commit();
    
    // Prepare response message
    $message = "Order #$order_id updated successfully";
    if (!empty($amount)) {
        $action = $transaction_type === 'credit' ? 'credited' : 'debited';
        $amount_column = $order_type === 'seller' ? 'amount_remaining' : 'amount';
        $message .= ". ₦" . number_format($amount, 2) . " $action to $amount_column.";
    }
    $message .= " Status updated to '$status'.";
    
    // Add match_status info for seller orders
    if ($order_type === 'seller') {
        $message .= " Match status reset to 'pending'.";
    }
    
    echo json_encode([
        'success' => true,
        'message' => $message,
        'order_id' => $order_id,
        'order_type' => $order_type,
        'status' => $status
    ]);
    
} catch (Exception $e) {
    // Rollback transaction on error
    if ($pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    echo json_encode([
        'success' => false,
        'message' => 'Error: ' . $e->getMessage()
    ]);
    exit;
}
?>