<?php
// b_setting.php
session_start();

// Set headers first
header('Content-Type: application/json; charset=utf-8');

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Handle POST requests
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Get JSON input
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Try form data if JSON fails
        $input = $_POST;
    }
    
    $action = $input['action'] ?? '';
    
    try {
        switch ($action) {
            case 'add_settings':
                // Check if settings already exist
                $checkStmt = $pdo->prepare("SELECT COUNT(*) as count FROM fund_settings");
                $checkStmt->execute();
                $result = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if ($result['count'] > 0) {
                    throw new Exception('Settings already exist. You can only edit existing settings.');
                }
                
                // Validate required fields
                $required_fields = ['refund_fee', 'buyer_late_fee', 'seller_late_fee', 'percentage'];
                $missing_fields = [];
                foreach ($required_fields as $field) {
                    if (!isset($input[$field]) || !is_numeric($input[$field])) {
                        $missing_fields[] = $field;
                    }
                }
                
                if (!empty($missing_fields)) {
                    throw new Exception('Missing or invalid required fields: ' . implode(', ', $missing_fields));
                }
                
                // Validate values
                $refund_fee = intval($input['refund_fee']);
                $buyer_late_fee = intval($input['buyer_late_fee']);
                $seller_late_fee = intval($input['seller_late_fee']);
                $percentage = floatval($input['percentage']);
                
                // Check max values
                if ($refund_fee < 0 || $refund_fee > 100000) {
                    throw new Exception('Refund fee must be between 0 and 100,000');
                }
                
                if ($buyer_late_fee < 0 || $buyer_late_fee > 100000) {
                    throw new Exception('Buyer late fee must be between 0 and 100,000');
                }
                
                if ($seller_late_fee < 0 || $seller_late_fee > 100000) {
                    throw new Exception('Seller late fee must be between 0 and 100,000');
                }
                
                if ($percentage < 0 || $percentage > 100) {
                    throw new Exception('Percentage must be between 0 and 100');
                }
                
                // Insert settings
                $sql = "INSERT INTO fund_settings (`refund fee`, `buyer_late_fee`, `seller_late_fee`, `percentage`) 
                        VALUES (:refund_fee, :buyer_late_fee, :seller_late_fee, :percentage)";
                
                $stmt = $pdo->prepare($sql);
                $stmt->bindParam(':refund_fee', $refund_fee);
                $stmt->bindParam(':buyer_late_fee', $buyer_late_fee);
                $stmt->bindParam(':seller_late_fee', $seller_late_fee);
                $stmt->bindParam(':percentage', $percentage);
                
                $result = $stmt->execute();
                
                if ($result) {
                    echo json_encode(['success' => true, 'message' => 'Settings added successfully']);
                } else {
                    throw new Exception('Failed to add settings');
                }
                break;
                
            case 'update_settings':
                // Check if settings exist
                $checkStmt = $pdo->prepare("SELECT COUNT(*) as count FROM fund_settings");
                $checkStmt->execute();
                $result = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if ($result['count'] === 0) {
                    throw new Exception('No settings found. Please add settings first.');
                }
                
                // Validate required fields
                $required_fields = ['refund_fee', 'buyer_late_fee', 'seller_late_fee', 'percentage'];
                $missing_fields = [];
                foreach ($required_fields as $field) {
                    if (!isset($input[$field]) || !is_numeric($input[$field])) {
                        $missing_fields[] = $field;
                    }
                }
                
                if (!empty($missing_fields)) {
                    throw new Exception('Missing or invalid required fields: ' . implode(', ', $missing_fields));
                }
                
                // Validate values
                $refund_fee = intval($input['refund_fee']);
                $buyer_late_fee = intval($input['buyer_late_fee']);
                $seller_late_fee = intval($input['seller_late_fee']);
                $percentage = floatval($input['percentage']);
                
                // Check max values
                if ($refund_fee < 0 || $refund_fee > 100000) {
                    throw new Exception('Refund fee must be between 0 and 100,000');
                }
                
                if ($buyer_late_fee < 0 || $buyer_late_fee > 100000) {
                    throw new Exception('Buyer late fee must be between 0 and 100,000');
                }
                
                if ($seller_late_fee < 0 || $seller_late_fee > 100000) {
                    throw new Exception('Seller late fee must be between 0 and 100,000');
                }
                
                if ($percentage < 0 || $percentage > 100) {
                    throw new Exception('Percentage must be between 0 and 100');
                }
                
                // Update all settings
                $sql = "UPDATE fund_settings SET 
                        `refund fee` = :refund_fee,
                        `buyer_late_fee` = :buyer_late_fee,
                        `seller_late_fee` = :seller_late_fee,
                        `percentage` = :percentage,
                        `created at` = CURRENT_TIMESTAMP";
                
                $stmt = $pdo->prepare($sql);
                $stmt->bindParam(':refund_fee', $refund_fee);
                $stmt->bindParam(':buyer_late_fee', $buyer_late_fee);
                $stmt->bindParam(':seller_late_fee', $seller_late_fee);
                $stmt->bindParam(':percentage', $percentage);
                
                $result = $stmt->execute();
                
                if ($result && $stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Settings updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'No changes made']);
                }
                break;
                
            case 'update_single_setting':
                // Check if settings exist
                $checkStmt = $pdo->prepare("SELECT COUNT(*) as count FROM fund_settings");
                $checkStmt->execute();
                $result = $checkStmt->fetch(PDO::FETCH_ASSOC);
                
                if ($result['count'] === 0) {
                    throw new Exception('No settings found. Please add settings first.');
                }
                
                // Validate required fields
                $field_name = $input['field_name'] ?? '';
                $field_value = $input['field_value'] ?? null;
                
                if (empty($field_name) || !is_numeric($field_value)) {
                    throw new Exception('Field name and value are required');
                }
                
                // Validate field name
                $valid_fields = ['refund fee', 'buyer_late_fee', 'seller_late_fee', 'percentage'];
                if (!in_array($field_name, $valid_fields)) {
                    throw new Exception('Invalid field name');
                }
                
                // Validate value based on field type
                if ($field_name === 'percentage') {
                    $field_value = floatval($field_value);
                    if ($field_value < 0 || $field_value > 100) {
                        throw new Exception('Percentage must be between 0 and 100');
                    }
                } else {
                    $field_value = intval($field_value);
                    if ($field_value < 0 || $field_value > 100000) {
                        throw new Exception('Value must be between 0 and 100,000');
                    }
                }
                
                // Update single setting
                // Note: We need to use backticks for column names with spaces
                $column_name = $field_name === 'refund fee' ? '`refund fee`' : $field_name;
                
                $sql = "UPDATE fund_settings SET 
                        $column_name = :field_value,
                        `created at` = CURRENT_TIMESTAMP";
                
                $stmt = $pdo->prepare($sql);
                $stmt->bindParam(':field_value', $field_value);
                
                $result = $stmt->execute();
                
                if ($result && $stmt->rowCount() > 0) {
                    echo json_encode(['success' => true, 'message' => 'Setting updated successfully']);
                } else {
                    echo json_encode(['success' => false, 'message' => 'No changes made']);
                }
                break;
                
            default:
                echo json_encode(['success' => false, 'message' => 'Invalid action: ' . $action]);
                break;
        }
    } catch (PDOException $e) {
        // Log the error for debugging
        error_log("Database Error in b_setting.php: " . $e->getMessage());
        echo json_encode(['success' => false, 'message' => 'Database error occurred']);
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => $e->getMessage()]);
    }
    exit;
}

echo json_encode(['success' => false, 'message' => 'Invalid request method. Use POST.']);
exit;
?>