<?php
// b_stream.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Content-Type: application/json');
    echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
    exit;
}

// Include database connection
require_once 'php/db.php';

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

// Validate input
if (!$input || !isset($input['action'])) {
    echo json_encode(['success' => false, 'message' => 'Invalid request']);
    exit;
}

$action = $input['action'];
$response = ['success' => false, 'message' => 'Unknown action'];

try {
    switch ($action) {
        case 'update_stream':
            // Update stream schedule
            if (!isset($input['title']) || !isset($input['day']) || !isset($input['time'])) {
                $response['message'] = 'Missing required fields';
                break;
            }
            
            $title = trim($input['title']);
            $day = trim($input['day']);
            $time = trim($input['time']); // Time only (HH:MM:SS)
            $stream_id = isset($input['stream_id']) ? intval($input['stream_id']) : null;
            $status = isset($input['status']) ? trim($input['status']) : 'not started';
            
            // Validate day
            $valid_days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
            if (!in_array($day, $valid_days)) {
                $response['message'] = 'Invalid day selected';
                break;
            }
            
            // Validate status
            $valid_statuses = ['not started', 'started', 'ended', 'live'];
            if (!in_array($status, $valid_statuses)) {
                $response['message'] = 'Invalid status';
                break;
            }
            
            // Check if there's already a stream with active status (not ended)
            $stmt = $pdo->prepare("SELECT id FROM audio_conference WHERE status IN ('not started', 'started', 'live') LIMIT 1");
            $stmt->execute();
            $active_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($stream_id) {
                // Update specific stream by ID
                $stmt = $pdo->prepare("
                    UPDATE audio_conference 
                    SET title = ?, day = ?, time = ?, status = ? 
                    WHERE id = ?
                ");
                $stmt->execute([$title, $day, $time, $status, $stream_id]);
                
                if ($stmt->rowCount() > 0) {
                    $response = ['success' => true, 'message' => 'Stream updated successfully'];
                } else {
                    $response['message'] = 'No changes made or stream not found';
                }
            } else {
                // Check if we can create a new stream (only if all existing streams are 'ended')
                if ($active_stream && $active_stream['id']) {
                    $response['message'] = 'Cannot create new stream while there is an active (not ended) stream. Please end or delete the active stream first.';
                    break;
                }
                
                // Check if stream exists for this day with not started status
                $stmt = $pdo->prepare("SELECT id FROM audio_conference WHERE day = ? AND status = 'not started' LIMIT 1");
                $stmt->execute([$day]);
                $existing_stream = $stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($existing_stream) {
                    // Update existing stream
                    $stmt = $pdo->prepare("
                        UPDATE audio_conference 
                        SET title = ?, time = ?, created_at = NOW() 
                        WHERE id = ?
                    ");
                    $stmt->execute([$title, $time, $existing_stream['id']]);
                    
                    if ($stmt->rowCount() > 0) {
                        $response = ['success' => true, 'message' => 'Stream updated successfully'];
                    } else {
                        $response['message'] = 'No changes made';
                    }
                } else {
                    // Create new stream
                    $stmt = $pdo->prepare("
                        INSERT INTO audio_conference (title, day, time, status, created_at) 
                        VALUES (?, ?, ?, 'not started', NOW())
                    ");
                    $stmt->execute([$title, $day, $time]);
                    
                    $response = ['success' => true, 'message' => 'Stream created successfully', 'stream_id' => $pdo->lastInsertId()];
                }
            }
            break;
            
        case 'start_stream':
            // Start a new live stream from the next scheduled stream
            
            // Check if there's already a live stream
            $stmt = $pdo->prepare("SELECT id FROM audio_conference WHERE status = 'live' LIMIT 1");
            $stmt->execute();
            $live_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($live_stream) {
                $response['message'] = 'There is already a live stream running. Please end it first.';
                break;
            }
            
            // Get the next scheduled stream (not started)
            $stmt = $pdo->prepare("
                SELECT id, title FROM audio_conference 
                WHERE status = 'not started' 
                ORDER BY FIELD(day, 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'), time ASC 
                LIMIT 1
            ");
            $stmt->execute();
            $next_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$next_stream) {
                $response['message'] = 'No scheduled streams found. Please schedule a stream first.';
                break;
            }
            
            // Start the stream by marking it as 'live'
            $stmt = $pdo->prepare("UPDATE audio_conference SET status = 'live', created_at = NOW() WHERE id = ?");
            $stmt->execute([$next_stream['id']]);
            
            // Create today's stream count entry if not exists
            $today = date('Y-m-d');
            $stmt = $pdo->prepare("SELECT id FROM stream_counts WHERE DATE(date) = ?");
            $stmt->execute([$today]);
            
            if (!$stmt->fetch()) {
                $stmt = $pdo->prepare("INSERT INTO stream_counts (date, counts, created_at) VALUES (?, 0, NOW())");
                $stmt->execute([$today . ' 00:00:00']);
            }
            
            $response = ['success' => true, 'message' => 'Stream started successfully', 'stream_id' => $next_stream['id']];
            break;
            
        case 'stop_stream':
            // Stop current live stream
            $stmt = $pdo->prepare("SELECT id FROM audio_conference WHERE status = 'live' LIMIT 1");
            $stmt->execute();
            $live_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$live_stream) {
                $response['message'] = 'No live stream found';
                break;
            }
            
            $stream_id = $live_stream['id'];
            
            $stmt = $pdo->prepare("UPDATE audio_conference SET status = 'ended' WHERE id = ?");
            $stmt->execute([$stream_id]);
            
            if ($stmt->rowCount() > 0) {
                $response = ['success' => true, 'message' => 'Stream stopped successfully'];
            } else {
                $response['message'] = 'Stream not found or already stopped';
            }
            break;
            
        case 'get_stream_counts':
            // Get stream counts for a specific date
            $date = isset($input['date']) ? $input['date'] : date('Y-m-d');
            
            // Get total counts for the date
            $stmt = $pdo->prepare("SELECT SUM(counts) as total_counts FROM stream_counts WHERE DATE(date) = ?");
            $stmt->execute([$date]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'total_counts' => $result['total_counts'] ?? 0,
                'date' => $date
            ];
            break;
            
        case 'get_participants':
            // Get recent participants for active/ended stream (within 24 hours)
            $limit = isset($input['limit']) ? intval($input['limit']) : 50;
            
            // Get active or most recent stream (within 24 hours)
            $stmt = $pdo->prepare("
                SELECT id, status, created_at 
                FROM audio_conference 
                WHERE status IN ('live', 'ended') 
                AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY created_at DESC 
                LIMIT 1
            ");
            $stmt->execute();
            $active_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if (!$active_stream) {
                $response = [
                    'success' => true,
                    'participants' => [],
                    'count' => 0,
                    'stream_status' => 'no_stream'
                ];
                break;
            }
            
            // Get participants for this stream
            $stmt = $pdo->prepare("
                SELECT lsp.*, u.username, u.email, u.profile_url, u.first_name, u.last_name 
                FROM live_stream_participant lsp 
                JOIN users u ON lsp.user_id = u.user_id 
                WHERE lsp.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY lsp.created_at DESC 
                LIMIT ?
            ");
            $stmt->execute([$limit]);
            $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'participants' => $participants,
                'count' => count($participants),
                'stream_status' => $active_stream['status'],
                'stream_created_at' => $active_stream['created_at'],
                'stream_id' => $active_stream['id']
            ];
            break;
            
        case 'get_new_participants':
            // Get only new participants since last check
            $last_id = isset($input['last_id']) ? intval($input['last_id']) : 0;
            
            $stmt = $pdo->prepare("
                SELECT lsp.*, u.username, u.email, u.profile_url, u.first_name, u.last_name 
                FROM live_stream_participant lsp 
                JOIN users u ON lsp.user_id = u.user_id 
                WHERE lsp.id > ? 
                AND lsp.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY lsp.created_at DESC
            ");
            $stmt->execute([$last_id]);
            $new_participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $max_id = $last_id;
            if (!empty($new_participants)) {
                $max_id = max(array_column($new_participants, 'id'));
            }
            
            $response = [
                'success' => true,
                'new_participants' => $new_participants,
                'count' => count($new_participants),
                'last_id' => $max_id
            ];
            break;
            
        case 'get_stream_info':
            // Get current stream information
            $stmt = $pdo->prepare("
                SELECT *, TIME_TO_SEC(TIMEDIFF(NOW(), created_at)) as duration_seconds 
                FROM audio_conference 
                WHERE status IN ('live', 'ended') 
                AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
                ORDER BY created_at DESC 
                LIMIT 1
            ");
            $stmt->execute();
            $stream_info = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($stream_info) {
                $response = [
                    'success' => true,
                    'stream' => $stream_info,
                    'duration' => $stream_info['duration_seconds'] ?? 0
                ];
            } else {
                $response = [
                    'success' => true,
                    'stream' => null,
                    'message' => 'No active stream'
                ];
            }
            break;
            
        case 'get_next_stream':
            // Get the next scheduled stream for configuration
            $stmt = $pdo->prepare("
                SELECT * FROM audio_conference 
                WHERE status = 'not started' 
                ORDER BY FIELD(day, 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'), time ASC 
                LIMIT 1
            ");
            $stmt->execute();
            $next_stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($next_stream) {
                $response = [
                    'success' => true,
                    'stream' => $next_stream
                ];
            } else {
                $response = [
                    'success' => true,
                    'stream' => null,
                    'message' => 'No scheduled streams'
                ];
            }
            break;
            
        case 'get_upcoming_streams':
            // Get all upcoming streams (not started)
            $stmt = $pdo->prepare("
                SELECT * FROM audio_conference 
                WHERE status = 'not started' 
                ORDER BY FIELD(day, 'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'), time ASC
            ");
            $stmt->execute();
            $streams = $stmt->fetchAll(PDO::FETCH_ASSOC);
            
            $response = [
                'success' => true,
                'streams' => $streams,
                'count' => count($streams)
            ];
            break;
            
        case 'delete_stream':
            // Delete a stream
            if (!isset($input['stream_id'])) {
                $response['message'] = 'Stream ID required';
                break;
            }
            
            $stream_id = intval($input['stream_id']);
            
            $stmt = $pdo->prepare("DELETE FROM audio_conference WHERE id = ? AND status = 'not started'");
            $stmt->execute([$stream_id]);
            
            if ($stmt->rowCount() > 0) {
                $response = ['success' => true, 'message' => 'Stream deleted successfully'];
            } else {
                $response['message'] = 'Stream not found or cannot be deleted';
            }
            break;
            
        case 'get_stream_by_id':
            // Get stream by ID
            if (!isset($input['stream_id'])) {
                $response['message'] = 'Stream ID required';
                break;
            }
            
            $stream_id = intval($input['stream_id']);
            
            $stmt = $pdo->prepare("SELECT * FROM audio_conference WHERE id = ?");
            $stmt->execute([$stream_id]);
            $stream = $stmt->fetch(PDO::FETCH_ASSOC);
            
            if ($stream) {
                $response = [
                    'success' => true,
                    'stream' => $stream
                ];
            } else {
                $response['message'] = 'Stream not found';
            }
            break;
            
        default:
            $response['message'] = 'Invalid action';
            break;
    }
} catch (PDOException $e) {
    $response = ['success' => false, 'message' => 'Database error: ' . $e->getMessage()];
} catch (Exception $e) {
    $response = ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
}

header('Content-Type: application/json');
echo json_encode($response);
exit;