<?php
// verify.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Get verification stats
try {
    // Get total verified users count (users table verification = 'verified')
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM users WHERE verification = 'verified'");
    $stmt->execute();
    $verifiedUsers = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get under review stats (verifications table status = 'In Review')
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM verifications WHERE status = 'In Review'");
    $stmt->execute();
    $underReview = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get users under review with their details
    $stmt = $pdo->prepare("
        SELECT u.user_id, u.username, u.first_name, u.last_name, u.email, 
               u.phone_number, u.country, u.profile_url,
               v.id as verification_id, v.status, v.created_at as verification_date
        FROM users u
        INNER JOIN verifications v ON u.user_id = v.user_id
        WHERE v.status = 'In Review'
        ORDER BY v.created_at DESC
    ");
    $stmt->execute();
    $reviewUsers = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
} catch (Exception $e) {
    $error = "Error fetching data: " . $e->getMessage();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Verification Management - Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Stats Container */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 20px;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
        }
        
        .stat-icon {
            width: 70px;
            height: 70px;
            border-radius: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 30px;
        }
        
        .stat-icon.verified {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .stat-icon.review {
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .stat-content {
            flex: 1;
        }
        
        .stat-title {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 5px;
            font-weight: 500;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .stat-subtitle {
            font-size: 14px;
            color: var(--gray);
        }
        
        /* Users Container */
        .users-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .container-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .container-header h3 {
            font-size: 22px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .users-count {
            background: var(--light);
            color: var(--dark);
            padding: 6px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
        }
        
        /* Users List */
        .users-list {
            display: flex;
            flex-direction: column;
            gap: 20px;
        }
        
        .user-card {
            background: var(--light);
            border-radius: 12px;
            padding: 25px;
            transition: all 0.3s ease;
            border-left: 4px solid var(--warning);
        }
        
        .user-card:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .user-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 60px;
            height: 60px;
            border-radius: 50%;
            overflow: hidden;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 20px;
            flex-shrink: 0;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .user-details h4 {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .user-details p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 3px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .user-details p i {
            width: 16px;
        }
        
        .user-meta {
            display: flex;
            gap: 20px;
            margin-top: 10px;
        }
        
        .meta-item {
            display: flex;
            flex-direction: column;
        }
        
        .meta-label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 2px;
            font-weight: 500;
        }
        
        .meta-value {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .user-actions {
            display: flex;
            gap: 10px;
        }
        
        .approve-btn {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
            border: none;
            padding: 10px 25px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
            white-space: nowrap;
        }
        
        .approve-btn:hover {
            background: linear-gradient(90deg, var(--secondary-dark), #047857);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .review-date {
            background: var(--warning);
            color: white;
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
        }
        
        /* Error Message */
        .error-message {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
            padding: 20px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .user-card {
                flex-direction: column;
                align-items: flex-start;
                gap: 15px;
            }
            
            .user-header {
                flex-direction: column;
                align-items: flex-start;
                gap: 15px;
            }
            
            .user-actions {
                width: 100%;
                justify-content: flex-end;
            }
            
            .users-container {
                padding: 20px;
            }
        }
        
        @media (max-width: 480px) {
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .user-meta {
                flex-direction: column;
                gap: 10px;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Verification Management</h2>
                    <p>Monitor and manage user verifications</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <?php if (isset($error)): ?>
                <div class="error-message">
                    <i class="fas fa-exclamation-circle"></i> <?php echo $error; ?>
                </div>
            <?php endif; ?>
            
            <!-- Stats Container -->
            <div class="stats-container">
                <div class="stat-card">
                    <div class="stat-icon verified">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Verified Users</div>
                        <div class="stat-value"><?php echo $verifiedUsers['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Total verified accounts</div>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-icon review">
                        <i class="fas fa-hourglass-half"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-title">Under Review</div>
                        <div class="stat-value"><?php echo $underReview['total'] ?? 0; ?></div>
                        <div class="stat-subtitle">Verifications in review</div>
                    </div>
                </div>
            </div>
            
            <!-- Users Under Review Container -->
            <div class="users-container">
                <div class="container-header">
                    <h3>Users Under Review</h3>
                    <span class="users-count"><?php echo count($reviewUsers ?? []); ?> users</span>
                </div>
                
                <div class="users-list">
                    <?php if (!empty($reviewUsers)): ?>
                        <?php foreach ($reviewUsers as $user): ?>
                            <div class="user-card">
                                <div class="user-header">
                                    <div class="user-info">
                                        <div class="user-avatar">
                                            <?php if (!empty($user['profile_url'])): ?>
                                                <img src="<?php echo htmlspecialchars($user['profile_url']); ?>" 
                                                     alt="<?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?>">
                                            <?php else: ?>
                                                <?php echo strtoupper(substr($user['first_name'], 0, 1) . substr($user['last_name'], 0, 1)); ?>
                                            <?php endif; ?>
                                        </div>
                                        <div class="user-details">
                                            <h4><?php echo htmlspecialchars($user['first_name'] . ' ' . $user['last_name']); ?></h4>
                                            <p>
                                                <i class="fas fa-envelope"></i>
                                                <?php echo htmlspecialchars($user['email']); ?>
                                            </p>
                                            <p>
                                                <i class="fas fa-phone"></i>
                                                <?php echo htmlspecialchars($user['phone_number']); ?>
                                            </p>
                                            <p>
                                                <i class="fas fa-globe"></i>
                                                <?php echo htmlspecialchars($user['country']); ?>
                                            </p>
                                            <div class="user-meta">
                                                <div class="meta-item">
                                                    <span class="meta-label">Username</span>
                                                    <span class="meta-value"><?php echo htmlspecialchars($user['username']); ?></span>
                                                </div>
                                                <div class="meta-item">
                                                    <span class="meta-label">Verification ID</span>
                                                    <span class="meta-value">#<?php echo $user['verification_id']; ?></span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <div class="user-actions">
                                        <button class="approve-btn" onclick="approveVerification(<?php echo $user['user_id']; ?>)">
                                            <i class="fas fa-check-circle"></i> Approve
                                        </button>
                                    </div>
                                </div>
                                <div style="display: flex; justify-content: space-between; align-items: center; margin-top: 15px;">
                                    <div class="review-date">
                                        <i class="fas fa-clock"></i> 
                                        <?php echo date('M d, Y H:i', strtotime($user['verification_date'])); ?>
                                    </div>
                                    <div style="font-size: 12px; color: var(--warning);">
                                        <i class="fas fa-exclamation-triangle"></i> Awaiting Approval
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-users-slash"></i>
                            <h3>No Users Under Review</h3>
                            <p>There are currently no verifications pending review</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    
    <script>
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Approve verification function
        function approveVerification(userId) {
            // Redirect to external approval system
            window.open('https://business.didit.me/', '_blank');
        }
    </script>
</body>
</html>