<?php
// Start session and check authentication
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}

if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Function to format numbers with naira symbol
function formatNaira($amount) {
    return '₦' . number_format($amount, 2);
}

// Function to format date/time
function formatDateTime($datetime) {
    if (!$datetime) return 'Never';
    return date('M d, Y H:i:s', strtotime($datetime));
}

// Get current time for auction calculations
date_default_timezone_set('Africa/Lagos'); // Nigerian timezone
$current_time = date('H:i:s');
$current_datetime = date('Y-m-d H:i:s');

// Fetch all statistics from database
try {
    // Total Users
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
    $total_users = $stmt->fetch()['total'];
    
    // Active Users
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE account_status = 'active'");
    $active_users = $stmt->fetch()['total'];
    
    // Verified Users
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users WHERE verification = 'verified'");
    $verified_users = $stmt->fetch()['total'];
    
    // Total Buyer Orders
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM buyer_orders");
    $buyer_data = $stmt->fetch();
    $total_buyer_orders = $buyer_data['total'];
    $total_buyer_amount = $buyer_data['total_amount'] ?: 0;
    
    // Active Buyer Orders (not sold)
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM buyer_orders WHERE status != 'sold'");
    $active_buyer_data = $stmt->fetch();
    $active_buyer_orders = $active_buyer_data['total'];
    $active_buyer_amount = $active_buyer_data['total_amount'] ?: 0;
    
    // Total Seller Orders
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM seller_orders");
    $seller_data = $stmt->fetch();
    $total_seller_orders = $seller_data['total'];
    $total_seller_amount = $seller_data['total_amount'] ?: 0;
    
    // Active Seller Orders (in progress)
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM seller_orders WHERE status = 'in progress'");
    $active_seller_data = $stmt->fetch();
    $active_seller_orders = $active_seller_data['total'];
    $active_seller_amount = $active_seller_data['total_amount'] ?: 0;
    
    // Security Deposits Statistics - UPDATED
    // Total Paid Deposits
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM security_deposit WHERE status = 'paid'");
    $paid_deposit_data = $stmt->fetch();
    $paid_deposits = $paid_deposit_data['total'];
    $paid_deposit_amount = $paid_deposit_data['total_amount'] ?: 0;
    
    // Total Refunded Deposits
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM security_deposit WHERE status = 'refunded'");
    $refunded_deposit_data = $stmt->fetch();
    $refunded_deposits = $refunded_deposit_data['total'];
    $refunded_deposit_amount = $refunded_deposit_data['total_amount'] ?: 0;
    
    // Total Not Paid Deposits
    $stmt = $pdo->query("SELECT COUNT(*) as total, SUM(amount) as total_amount FROM security_deposit WHERE status = 'not paid'");
    $not_paid_deposit_data = $stmt->fetch();
    $not_paid_deposits = $not_paid_deposit_data['total'];
    $not_paid_deposit_amount = $not_paid_deposit_data['total_amount'] ?: 0;
    
    // Total All Deposits
    $total_deposits = $paid_deposits + $refunded_deposits + $not_paid_deposits;
    $total_deposit_amount = $paid_deposit_amount + $refunded_deposit_amount + $not_paid_deposit_amount;
    
    // Pending Testimonies
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM testimony WHERE status = 'pending'");
    $pending_testimonies = $stmt->fetch()['total'];
    
    // Total Referrals
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM referrals");
    $total_referrals = $stmt->fetch()['total'];
    
    // Get Auction Times
    $stmt = $pdo->query("SELECT * FROM auction_time ORDER BY open_at ASC");
    $auction_times = $stmt->fetchAll();
    
    // Calculate next auction time
    $next_auction = null;
    $next_auction_time = null;
    foreach ($auction_times as $auction) {
        $auction_time = strtotime($auction['open_at']);
        $current_time_stamp = strtotime($current_time);
        
        if ($auction_time > $current_time_stamp) {
            $next_auction = $auction;
            $next_auction_time = date('H:i:s', $auction_time);
            break;
        }
    }
    
    // If no auction found for today, get first one for tomorrow
    if (!$next_auction && count($auction_times) > 0) {
        $next_auction = $auction_times[0];
        $next_auction_time = date('H:i:s', strtotime($next_auction['open_at'] . ' +1 day'));
    }
    
    // Recent Activities (last 10) - FIXED SQL QUERY
    $stmt = $pdo->query("
        (SELECT 'buyer_order' as type, id, amount, created_at 
         FROM buyer_orders 
         ORDER BY created_at DESC LIMIT 5)
        UNION ALL
        (SELECT 'seller_order' as type, id, amount, created_at 
         FROM seller_orders 
         ORDER BY created_at DESC LIMIT 5)
        ORDER BY created_at DESC LIMIT 10
    ");
    $recent_activities = $stmt->fetchAll();
    
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Overview - BCA Secure Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h2>Dashboard Overview</h2>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Stats Grid -->
            <div class="stats-grid">
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-users">
                            <i class="fas fa-users"></i>
                        </div>
                        <span class="stat-trend trend-up">+12%</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($total_users); ?></div>
                    <div class="stat-label">Total Users</div>
                    <div class="stat-details">
                        <small><?php echo number_format($active_users); ?> Active • <?php echo number_format($verified_users); ?> Verified</small>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-money">
                            <i class="fas fa-money-bill-wave"></i>
                        </div>
                        <span class="stat-trend trend-up">+24%</span>
                    </div>
                    <div class="stat-value"><?php echo formatNaira($total_buyer_amount + $total_seller_amount); ?></div>
                    <div class="stat-label">Total Orders Value</div>
                    <div class="stat-details">
                        <small>Buy: <?php echo formatNaira($total_buyer_amount); ?> • Sell: <?php echo formatNaira($total_seller_amount); ?></small>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-orders">
                            <i class="fas fa-exchange-alt"></i>
                        </div>
                        <span class="stat-trend trend-up">+18%</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($total_buyer_orders + $total_seller_orders); ?></div>
                    <div class="stat-label">Total Orders</div>
                    <div class="stat-details">
                        <small><?php echo number_format($total_buyer_orders); ?> Buy • <?php echo number_format($total_seller_orders); ?> Sell</small>
                    </div>
                </div>
                
                <!-- UPDATED: Security Deposits Card -->
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-deposit">
                            <i class="fas fa-shield-alt"></i>
                        </div>
                        <span class="stat-trend trend-up">+8%</span>
                    </div>
                    <div class="stat-value"><?php echo formatNaira($total_deposit_amount); ?></div>
                    <div class="stat-label">Security Deposits</div>
                    <div class="stat-details">
                        <small>
                            Paid: <?php echo number_format($paid_deposits); ?> (<?php echo formatNaira($paid_deposit_amount); ?>)<br>
                            Refunded: <?php echo number_format($refunded_deposits); ?> (<?php echo formatNaira($refunded_deposit_amount); ?>)<br>
                            Not Paid: <?php echo number_format($not_paid_deposits); ?> (<?php echo formatNaira($not_paid_deposit_amount); ?>)
                        </small>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-testimony">
                            <i class="fas fa-comment-dots"></i>
                        </div>
                        <span class="stat-trend trend-up">+15%</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($pending_testimonies); ?></div>
                    <div class="stat-label">Pending Testimonies</div>
                    <div class="stat-details">
                        <small>Awaiting approval</small>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-icon icon-referral">
                            <i class="fas fa-user-plus"></i>
                        </div>
                        <span class="stat-trend trend-up">+32%</span>
                    </div>
                    <div class="stat-value"><?php echo number_format($total_referrals); ?></div>
                    <div class="stat-label">Total Referrals</div>
                    <div class="stat-details">
                        <small>User Referral Network</small>
                    </div>
                </div>
            </div>
            
            <!-- Auction Timer -->
            <div class="auction-section">
                <h3 class="section-title">
                    <i class="fas fa-gavel"></i>
                    Auction Schedule
                </h3>
                
                <div class="timer-container">
                    <div class="timer" id="auctionTimer">
                        <?php if ($next_auction): ?>
                        <span id="hours">00</span>:<span id="minutes">00</span>:<span id="seconds">00</span>
                        <?php else: ?>
                        No Auction Scheduled
                        <?php endif; ?>
                    </div>
                    <div class="timer-info">
                        <h4>Next Auction</h4>
                        <?php if ($next_auction): ?>
                        <p><strong>Time:</strong> <?php echo date('h:i A', strtotime($next_auction['open_at'])); ?></p>
                        <p><strong>Duration:</strong> <?php echo $next_auction['close_after']; ?> minutes</p>
                        <p><strong>Status:</strong> <span class="trend-up" style="display: inline-block; padding: 2px 8px;">Upcoming</span></p>
                        <?php else: ?>
                        <p>No auction times configured</p>
                        <?php endif; ?>
                    </div>
                </div>
                
                <h4 style="margin-bottom: 15px; color: var(--gray);">Today's Auction Schedule</h4>
                <div class="auction-schedule">
                    <?php foreach ($auction_times as $index => $auction): 
                        $auction_time = strtotime($auction['open_at']);
                        $is_past = $auction_time < strtotime($current_time);
                        $is_next = $next_auction && $next_auction['id'] == $auction['id'];
                    ?>
                    <div class="auction-time <?php echo $is_next ? 'active' : ''; ?>">
                        <h5>Auction <?php echo $index + 1; ?></h5>
                        <p><strong>Open:</strong> <?php echo date('h:i A', $auction_time); ?></p>
                        <p><strong>Close After:</strong> <?php echo $auction['close_after']; ?> min</p>
                        <p><strong>Status:</strong> 
                            <?php if ($is_next): ?>
                            <span style="color: var(--success); font-weight: 500;">Next</span>
                            <?php elseif ($is_past): ?>
                            <span style="color: var(--gray);">Completed</span>
                            <?php else: ?>
                            <span style="color: var(--warning);">Upcoming</span>
                            <?php endif; ?>
                        </p>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
            
            <!-- Recent Activities -->
            <div class="activities-section">
                <h3 class="section-title">
                    <i class="fas fa-history"></i>
                    Recent Activities
                </h3>
                
                <table class="activities-table">
                    <thead>
                        <tr>
                            <th>Type</th>
                            <th>Order ID</th>
                            <th>Amount</th>
                            <th>Time</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (count($recent_activities) > 0): ?>
                            <?php foreach ($recent_activities as $activity): ?>
                            <tr>
                                <td>
                                    <span class="activity-type <?php echo $activity['type'] == 'buyer_order' ? 'type-buy' : 'type-sell'; ?>">
                                        <?php echo $activity['type'] == 'buyer_order' ? 'BUY' : 'SELL'; ?>
                                    </span>
                                </td>
                                <td><?php echo $activity['id']; ?></td>
                                <td class="activity-amount"><?php echo formatNaira($activity['amount']); ?></td>
                                <td class="activity-time"><?php echo formatDateTime($activity['created_at']); ?></td>
                            </tr>
                            <?php endforeach; ?>
                        <?php else: ?>
                            <tr>
                                <td colspan="4" style="text-align: center; padding: 40px; color: var(--gray);">
                                    No recent activities found
                                </td>
                            </tr>
                        <?php endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <style>
        /* Additional styles for main content */
        .header-actions {
            display: flex;
            gap: 15px;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Stats Grid */
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
            gap: 25px;
            margin-bottom: 40px;
        }
        
        .stat-card {
            background: white;
            padding: 25px;
            border-radius: 15px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
            border: 1px solid var(--gray-light);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 20px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            color: white;
        }
        
        .icon-users { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
        .icon-money { background: linear-gradient(135deg, #10b981, #059669); }
        .icon-orders { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .icon-deposit { background: linear-gradient(135deg, #3b82f6, #1d4ed8); }
        .icon-testimony { background: linear-gradient(135deg, #ec4899, #db2777); }
        .icon-referral { background: linear-gradient(135deg, #06b6d4, #0891b2); }
        
        .stat-trend {
            font-size: 14px;
            padding: 4px 12px;
            border-radius: 20px;
            font-weight: 500;
        }
        
        .trend-up {
            background: #d1fae5;
            color: #065f46;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .stat-label {
            font-size: 14px;
            color: var(--gray);
            font-weight: 500;
        }
        
        .stat-details small {
            font-size: 12px;
            color: var(--gray);
            line-height: 1.5;
        }
        
        /* Auction Timer */
        .auction-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            margin-bottom: 40px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .timer-container {
            display: flex;
            align-items: center;
            gap: 30px;
            margin-bottom: 30px;
            padding: 25px;
            background: linear-gradient(135deg, #f0f9ff, #e0f2fe);
            border-radius: 15px;
        }
        
        .timer {
            font-family: 'Courier New', monospace;
            font-size: 48px;
            font-weight: 700;
            color: var(--primary-dark);
            background: white;
            padding: 15px 25px;
            border-radius: 10px;
            box-shadow: 0 4px 15px rgba(59, 130, 246, 0.15);
            min-width: 280px;
            text-align: center;
        }
        
        .timer-info {
            flex: 1;
        }
        
        .timer-info h4 {
            font-size: 18px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 10px;
        }
        
        .timer-info p {
            color: var(--gray);
            margin-bottom: 5px;
        }
        
        .auction-schedule {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
        }
        
        .auction-time {
            background: var(--light);
            padding: 20px;
            border-radius: 10px;
            border-left: 4px solid var(--primary);
        }
        
        .auction-time.active {
            background: linear-gradient(135deg, #dbeafe, #eff6ff);
            border-left-color: var(--secondary);
        }
        
        .auction-time h5 {
            font-size: 16px;
            font-weight: 600;
            margin-bottom: 8px;
            color: var(--dark);
        }
        
        .auction-time p {
            font-size: 14px;
            color: var(--gray);
        }
        
        /* Recent Activities */
        .activities-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .activities-table {
            width: 100%;
            border-collapse: collapse;
        }
        
        .activities-table th {
            text-align: left;
            padding: 15px;
            font-weight: 600;
            color: var(--gray);
            border-bottom: 2px solid var(--gray-light);
            font-size: 14px;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .activities-table td {
            padding: 15px;
            border-bottom: 1px solid var(--gray-light);
            vertical-align: middle;
        }
        
        .activity-type {
            display: inline-block;
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 500;
        }
        
        .type-buy {
            background: #d1fae5;
            color: #065f46;
        }
        
        .type-sell {
            background: #dbeafe;
            color: #1e40af;
        }
        
        .activity-amount {
            font-weight: 600;
            color: var(--dark);
        }
        
        .activity-time {
            font-size: 13px;
            color: var(--gray);
        }
    </style>

    <script>
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        // Update time every second
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        <?php if ($next_auction): ?>
        // Auction Timer Countdown
        function updateAuctionTimer() {
            const nextAuctionTime = new Date('<?php echo date("Y-m-d") . " " . $next_auction_time; ?>');
            const now = new Date();
            
            // If the auction time has passed for today, set it for tomorrow
            if (nextAuctionTime < now) {
                nextAuctionTime.setDate(nextAuctionTime.getDate() + 1);
            }
            
            const diff = nextAuctionTime - now;
            
            if (diff > 0) {
                const hours = Math.floor(diff / (1000 * 60 * 60));
                const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                const seconds = Math.floor((diff % (1000 * 60)) / 1000);
                
                document.getElementById('hours').textContent = hours.toString().padStart(2, '0');
                document.getElementById('minutes').textContent = minutes.toString().padStart(2, '0');
                document.getElementById('seconds').textContent = seconds.toString().padStart(2, '0');
            } else {
                document.getElementById('auctionTimer').innerHTML = '<span style="color: var(--success);">Auction Started!</span>';
            }
        }
        
        // Update timer every second
        setInterval(updateAuctionTimer, 1000);
        updateAuctionTimer();
        <?php endif; ?>
        
        // Auto-refresh page every 5 minutes to update data
        setTimeout(function() {
            location.reload();
        }, 5 * 60 * 1000); // 5 minutes
    </script>
</body>
</html>