<?php
// setting.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Fetch existing settings
try {
    $stmt = $pdo->prepare("SELECT * FROM fund_settings LIMIT 1");
    $stmt->execute();
    $settings = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Fund Settings - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Settings Container */
        .settings-container {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .settings-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
            padding-bottom: 15px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .settings-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .settings-title i {
            color: var(--primary);
        }
        
        .add-settings-btn {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .add-settings-btn:hover {
            background: linear-gradient(90deg, var(--primary-dark), #1e40af);
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Settings Grid */
        .settings-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .setting-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
        }
        
        .setting-item:hover {
            background: #f1f5f9;
            transform: translateY(-2px);
        }
        
        .setting-info {
            flex: 1;
        }
        
        .setting-label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
            text-transform: capitalize;
        }
        
        .setting-value {
            font-size: 18px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .setting-unit {
            font-size: 12px;
            color: var(--gray);
            margin-left: 4px;
            font-weight: 500;
        }
        
        .setting-actions {
            display: flex;
            gap: 8px;
        }
        
        .edit-setting-btn, .delete-setting-btn {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
        }
        
        .edit-setting-btn {
            background: var(--primary);
            color: white;
        }
        
        .edit-setting-btn:hover {
            background: var(--primary-dark);
        }
        
        .delete-setting-btn {
            background: var(--danger);
            color: white;
        }
        
        .delete-setting-btn:hover {
            background: #dc2626;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            background: var(--light);
            border-radius: 12px;
            margin-top: 20px;
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            color: var(--gray-light);
        }
        
        .empty-state h3 {
            font-size: 20px;
            margin-bottom: 10px;
            color: var(--dark);
        }
        
        .empty-state p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 25px;
        }
        
        /* Modals */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .setting-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active .setting-modal {
            transform: translateY(0);
        }
        
        .setting-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            text-align: center;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-group label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .form-group label.required::after {
            content: ' *';
            color: var(--danger);
        }
        
        .form-control {
            width: 100%;
            padding: 12px 16px;
            border: 1px solid var(--gray-light);
            border-radius: 8px;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .form-control:focus {
            outline: none;
            border-color: var(--primary);
            box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
        }
        
        .form-hint {
            font-size: 12px;
            color: var(--gray);
            margin-top: 5px;
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
            margin-top: 30px;
        }
        
        .form-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .form-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .form-btn.primary:hover:not(:disabled) {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .form-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .form-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .form-btn:disabled {
            opacity: 0.6;
            cursor: not-allowed;
        }
        
        /* Loading Spinner */
        .spinner {
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Message Modal */
        .message-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .message-modal {
            transform: translateY(0);
        }
        
        .message-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .message-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .message-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .message-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .message-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .message-actions {
            display: flex;
            justify-content: center;
        }
        
        .message-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .message-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Confirmation Modal */
        .confirm-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .confirm-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .confirm-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .confirm-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .confirm-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .confirm-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .confirm-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .confirm-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .confirm-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .confirm-btn.secondary:hover {
            background: #d1d5db;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .settings-grid {
                grid-template-columns: 1fr;
            }
            
            .setting-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .message-modal, .confirm-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
        
        @media (max-width: 480px) {
            .form-actions, .confirm-actions {
                flex-direction: column;
            }
            
            .form-btn, .confirm-btn {
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Fund Settings</h2>
                    <p>Manage fund settings and percentages</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Settings Container -->
            <div class="settings-container">
                <div class="settings-header">
                    <h3 class="settings-title">
                        <i class="fas fa-cog"></i>
                        Fund Settings
                    </h3>
                    
                    <?php if (!$settings): ?>
                        <button class="add-settings-btn" id="addSettingsBtn">
                            <i class="fas fa-plus"></i> Add Settings
                        </button>
                    <?php endif; ?>
                </div>
                
                <?php if ($settings): ?>
                    <div class="settings-grid" id="settingsList">
                        <?php 
                        // Define settings with their labels and units
                        $setting_fields = [
                            'refund fee' => ['label' => 'Refund Fee', 'unit' => '', 'max' => 100000],
                            'buyer_late_fee' => ['label' => 'Buyer Late Fee', 'unit' => '', 'max' => 100000],
                            'seller_late_fee' => ['label' => 'Seller Late Fee', 'unit' => '', 'max' => 100000],
                            'percentage' => ['label' => 'Percentage', 'unit' => '%', 'max' => 100]
                        ];
                        
                        foreach ($setting_fields as $key => $info): 
                            $value = $settings[$key] ?? '';
                        ?>
                            <div class="setting-item" data-field="<?php echo $key; ?>">
                                <div class="setting-info">
                                    <div class="setting-label"><?php echo $info['label']; ?></div>
                                    <div class="setting-value">
                                        <?php echo htmlspecialchars($value); ?>
                                        <?php if ($info['unit']): ?>
                                            <span class="setting-unit"><?php echo $info['unit']; ?></span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="setting-actions">
                                    <button class="edit-setting-btn" onclick="editSetting('<?php echo $key; ?>', '<?php echo htmlspecialchars($value); ?>', '<?php echo $info['max']; ?>')">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                    
                    <!-- Last Updated -->
                    <div style="margin-top: 25px; padding-top: 15px; border-top: 1px solid var(--gray-light); font-size: 12px; color: var(--gray);">
                        <i class="fas fa-clock"></i>
                        Last updated: <?php echo date('F j, Y H:i:s', strtotime($settings['created at'])); ?>
                    </div>
                    
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-cog"></i>
                        <h3>No Settings Found</h3>
                        <p>Fund settings have not been configured yet</p>
                        <button class="add-settings-btn" id="addSettingsBtn2">
                            <i class="fas fa-plus"></i> Add Settings
                        </button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <!-- Add/Edit Setting Modal -->
    <div class="modal-overlay" id="settingModal">
        <div class="setting-modal">
            <h3 id="modalTitle">Add Fund Settings</h3>
            <form id="settingForm">
                <input type="hidden" id="actionType" name="action_type" value="add">
                
                <div class="form-group">
                    <label for="refundFee" class="required">Refund Fee</label>
                    <input type="number" class="form-control" id="refundFee" name="refund_fee" min="0" max="100000" step="1" required>
                    <div class="form-hint">Enter a number between 0 and 100,000</div>
                </div>
                
                <div class="form-group">
                    <label for="buyerLateFee" class="required">Buyer Late Fee</label>
                    <input type="number" class="form-control" id="buyerLateFee" name="buyer_late_fee" min="0" max="100000" step="1" required>
                    <div class="form-hint">Enter a number between 0 and 100,000</div>
                </div>
                
                <div class="form-group">
                    <label for="sellerLateFee" class="required">Seller Late Fee</label>
                    <input type="number" class="form-control" id="sellerLateFee" name="seller_late_fee" min="0" max="100000" step="1" required>
                    <div class="form-hint">Enter a number between 0 and 100,000</div>
                </div>
                
                <div class="form-group">
                    <label for="percentage" class="required">Percentage</label>
                    <input type="number" class="form-control" id="percentage" name="percentage" min="0" max="100" step="0.01" required>
                    <div class="form-hint">Enter a percentage between 0 and 100%</div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelSettingBtn">Cancel</button>
                    <button type="submit" class="form-btn primary" id="saveSettingBtn">
                        <span id="saveBtnText">Save Settings</span>
                        <div class="spinner" style="display: none;"></div>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Edit Single Setting Modal -->
    <div class="modal-overlay" id="editSettingModal">
        <div class="setting-modal">
            <h3 id="editModalTitle">Edit Setting</h3>
            <form id="editSettingForm">
                <input type="hidden" id="editFieldName" name="field_name" value="">
                <input type="hidden" id="editMaxValue" name="max_value" value="">
                
                <div class="form-group">
                    <label id="editFieldLabel" class="required">Setting Value</label>
                    <input type="number" class="form-control" id="editFieldValue" name="field_value" required>
                    <div class="form-hint" id="editFieldHint">Enter a valid number</div>
                </div>
                
                <div class="form-actions">
                    <button type="button" class="form-btn secondary" id="cancelEditSettingBtn">Cancel</button>
                    <button type="submit" class="form-btn primary" id="saveEditSettingBtn">
                        <span id="saveEditBtnText">Update Setting</span>
                        <div class="spinner" style="display: none;"></div>
                    </button>
                </div>
            </form>
        </div>
    </div>
    
    <!-- Success/Error Message Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="message-modal">
            <div class="message-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="message-actions">
                <button class="message-btn" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <script>
        // Global variables
        let currentFieldName = '';
        let currentMaxValue = 0;
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon based on type
            icon.className = 'message-icon ' + type;
            if (type === 'success') {
                icon.innerHTML = '<i class="fas fa-check-circle"></i>';
            } else if (type === 'error') {
                icon.innerHTML = '<i class="fas fa-times-circle"></i>';
            }
            
            modal.classList.add('active');
        }
        
        // Add settings button click
        document.getElementById('addSettingsBtn')?.addEventListener('click', () => {
            openAddSettingsModal();
        });
        
        document.getElementById('addSettingsBtn2')?.addEventListener('click', () => {
            openAddSettingsModal();
        });
        
        // Open add settings modal
        function openAddSettingsModal() {
            document.getElementById('modalTitle').textContent = 'Add Fund Settings';
            document.getElementById('actionType').value = 'add';
            document.getElementById('settingForm').reset();
            document.getElementById('saveBtnText').textContent = 'Save Settings';
            document.getElementById('settingModal').classList.add('active');
        }
        
        // Edit single setting
        function editSetting(fieldName, currentValue, maxValue) {
            currentFieldName = fieldName;
            currentMaxValue = parseInt(maxValue);
            
            // Field label mapping
            const fieldLabels = {
                'refund fee': 'Refund Fee',
                'buyer_late_fee': 'Buyer Late Fee',
                'seller_late_fee': 'Seller Late Fee',
                'percentage': 'Percentage'
            };
            
            // Unit mapping
            const fieldUnits = {
                'percentage': '%'
            };
            
            // Set modal values
            document.getElementById('editModalTitle').textContent = 'Edit ' + fieldLabels[fieldName];
            document.getElementById('editFieldName').value = fieldName;
            document.getElementById('editMaxValue').value = maxValue;
            document.getElementById('editFieldLabel').textContent = fieldLabels[fieldName];
            document.getElementById('editFieldValue').value = currentValue;
            
            // Set min, max, and step attributes
            const inputField = document.getElementById('editFieldValue');
            inputField.min = 0;
            inputField.max = maxValue;
            inputField.step = fieldName === 'percentage' ? '0.01' : '1';
            
            // Set hint text
            const hintText = fieldName === 'percentage' 
                ? `Enter a percentage between 0 and ${maxValue}%`
                : `Enter a number between 0 and ${maxValue.toLocaleString()}`;
            
            document.getElementById('editFieldHint').textContent = hintText;
            document.getElementById('saveEditBtnText').textContent = 'Update Setting';
            
            // Show modal
            document.getElementById('editSettingModal').classList.add('active');
        }
        
        // Add settings form submission
        document.getElementById('settingForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            
            // Get form elements
            const saveBtn = document.getElementById('saveSettingBtn');
            const saveBtnText = document.getElementById('saveBtnText');
            const spinner = saveBtn.querySelector('.spinner');
            
            try {
                // Show loading state
                saveBtn.disabled = true;
                saveBtnText.textContent = 'Saving...';
                spinner.style.display = 'block';
                
                const response = await fetch('b_setting.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: data.action_type === 'add' ? 'add_settings' : 'update_settings',
                        refund_fee: data.refund_fee,
                        buyer_late_fee: data.buyer_late_fee,
                        seller_late_fee: data.seller_late_fee,
                        percentage: data.percentage
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Close modal
                    document.getElementById('settingModal').classList.remove('active');
                    
                    // Show success message
                    showMessage('Success', result.message, 'success');
                    
                    // Reload page after 2 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to save settings', 'error');
                }
            } catch (error) {
                console.error('Error saving settings:', error);
                showMessage('Error', 'Error saving settings', 'error');
            } finally {
                // Reset button state
                saveBtn.disabled = false;
                saveBtnText.textContent = 'Save Settings';
                spinner.style.display = 'none';
            }
        });
        
        // Edit setting form submission
        document.getElementById('editSettingForm').addEventListener('submit', async (e) => {
            e.preventDefault();
            
            const formData = new FormData(e.target);
            const data = Object.fromEntries(formData.entries());
            const fieldValue = parseFloat(data.field_value);
            
            // Validate value
            if (fieldValue < 0 || fieldValue > currentMaxValue) {
                showMessage('Error', `Value must be between 0 and ${currentMaxValue}`, 'error');
                return;
            }
            
            // Get form elements
            const saveBtn = document.getElementById('saveEditSettingBtn');
            const saveBtnText = document.getElementById('saveEditBtnText');
            const spinner = saveBtn.querySelector('.spinner');
            
            try {
                // Show loading state
                saveBtn.disabled = true;
                saveBtnText.textContent = 'Updating...';
                spinner.style.display = 'block';
                
                const response = await fetch('b_setting.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'update_single_setting',
                        field_name: data.field_name,
                        field_value: fieldValue
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    // Close modal
                    document.getElementById('editSettingModal').classList.remove('active');
                    
                    // Update the value in the UI
                    const settingItem = document.querySelector(`.setting-item[data-field="${data.field_name}"]`);
                    if (settingItem) {
                        const valueElement = settingItem.querySelector('.setting-value');
                        
                        // Field label mapping for units
                        const fieldUnits = {
                            'percentage': '%'
                        };
                        
                        const unit = fieldUnits[data.field_name] || '';
                        valueElement.innerHTML = `${fieldValue}${unit ? ` <span class="setting-unit">${unit}</span>` : ''}`;
                    }
                    
                    // Show success message
                    showMessage('Success', result.message, 'success');
                } else {
                    showMessage('Error', result.message || 'Failed to update setting', 'error');
                }
            } catch (error) {
                console.error('Error updating setting:', error);
                showMessage('Error', 'Error updating setting', 'error');
            } finally {
                // Reset button state
                saveBtn.disabled = false;
                saveBtnText.textContent = 'Update Setting';
                spinner.style.display = 'none';
            }
        });
        
        // Modal close buttons
        document.getElementById('cancelSettingBtn').addEventListener('click', () => {
            document.getElementById('settingModal').classList.remove('active');
        });
        
        document.getElementById('cancelEditSettingBtn').addEventListener('click', () => {
            document.getElementById('editSettingModal').classList.remove('active');
        });
        
        document.getElementById('messageOkBtn').addEventListener('click', () => {
            document.getElementById('messageModal').classList.remove('active');
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                }
            });
        });
    </script>
</body>
</html>