<?php
// stream.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
if (!isset($_SESSION['admin_role']) || $_SESSION['admin_role'] !== 'root') {
header('Location: auction.php');
exit;
}

// Include database connection
require_once 'php/db.php';

// Fetch the next scheduled stream for configuration
try {
    $stmt = $pdo->prepare("SELECT * FROM audio_conference WHERE status = 'not started' ORDER BY FIELD(day, 
        'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'), time ASC LIMIT 1");
    $stmt->execute();
    $next_stream = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Fetch all upcoming streams for display
try {
    $stmt = $pdo->prepare("SELECT * FROM audio_conference WHERE status = 'not started' ORDER BY FIELD(day, 
        'sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'), time ASC");
    $stmt->execute();
    $upcoming_streams = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Fetch current live stream
try {
    $stmt = $pdo->prepare("SELECT * FROM audio_conference WHERE status = 'live' ORDER BY created_at DESC LIMIT 1");
    $stmt->execute();
    $current_stream = $stmt->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Get today's stream count
$today_date = date('Y-m-d');
try {
    $stmt = $pdo->prepare("SELECT SUM(counts) as total_counts FROM stream_counts WHERE DATE(date) = ?");
    $stmt->execute([$today_date]);
    $today_count = $stmt->fetch(PDO::FETCH_ASSOC);
    $total_users_today = $today_count['total_counts'] ?? 0;
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Get total participants count
try {
    $stmt = $pdo->prepare("SELECT COUNT(*) as total FROM live_stream_participant");
    $stmt->execute();
    $participant_count = $stmt->fetch(PDO::FETCH_ASSOC);
    $total_participants = $participant_count['total'] ?? 0;
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Get participants if there's an active stream within 24 hours
$participants = [];
$show_participants = false;
$stream_for_participants = null;

try {
    // Get active or recently ended stream (within 24 hours)
    $stmt = $pdo->prepare("
        SELECT * FROM audio_conference 
        WHERE status IN ('live', 'ended') 
        AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ORDER BY created_at DESC 
        LIMIT 1
    ");
    $stmt->execute();
    $stream_for_participants = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($stream_for_participants) {
        $show_participants = true;
        
        // Get participants (within 24 hours)
        $stmt = $pdo->prepare("
            SELECT lsp.*, u.username, u.email, u.profile_url, u.first_name, u.last_name 
            FROM live_stream_participant lsp 
            JOIN users u ON lsp.user_id = u.user_id 
            WHERE lsp.created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ORDER BY lsp.created_at DESC 
            LIMIT 20
        ");
        $stmt->execute();
        $participants = $stmt->fetchAll(PDO::FETCH_ASSOC);
    }
} catch (PDOException $e) {
    die("Database error: " . $e->getMessage());
}

// Calculate countdown for the very next stream (for stats)
$countdown_hours = 0;
$countdown_minutes = 0;
$countdown_seconds = 0;

if ($next_stream && isset($next_stream['time'])) {
    // For countdown, we need to calculate based on next occurrence of the day/time
    $current_day = strtolower(date('l'));
    $target_day = $next_stream['day'];
    
    $days = ['sunday', 'monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday'];
    $current_day_index = array_search($current_day, $days);
    $target_day_index = array_search($target_day, $days);
    
    // Calculate days until target day
    $days_until_target = $target_day_index - $current_day_index;
    if ($days_until_target < 0) {
        $days_until_target += 7;
    }
    
    // Get current time and target time
    $current_time = time();
    $target_time_today = strtotime($next_stream['time']);
    $current_time_of_day = $current_time - strtotime('today');
    
    // If it's today and time hasn't passed yet
    if ($days_until_target == 0 && $current_time_of_day < $target_time_today) {
        $time_diff = $target_time_today - $current_time_of_day;
    } else {
        // Next week's occurrence
        if ($days_until_target == 0) {
            $days_until_target = 7;
        }
        $time_diff = ($days_until_target * 86400) + $target_time_today - $current_time_of_day;
    }
    
    if ($time_diff > 0) {
        $countdown_hours = floor($time_diff / 3600);
        $countdown_minutes = floor(($time_diff % 3600) / 60);
        $countdown_seconds = $time_diff % 60;
    }
}

// Calculate stream duration for live stream
$stream_duration = '';
if ($current_stream && isset($current_stream['created_at'])) {
    $start_time = strtotime($current_stream['created_at']);
    $current_time = time();
    $duration = $current_time - $start_time;
    
    $hours = floor($duration / 3600);
    $minutes = floor(($duration % 3600) / 60);
    
    if ($hours > 0) {
        $stream_duration = $hours . "h " . $minutes . "m";
    } else {
        $stream_duration = $minutes . "m";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Live Stream Admin - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
            --live-red: #ef4444;
            --upcoming-orange: #f59e0b;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #0f172a 0%, #1e293b 100%);
            min-height: 100vh;
            color: white;
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: white;
        }
        
        .header-subtitle {
            font-size: 16px;
            color: #94a3b8;
            margin-top: 5px;
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: rgba(255, 255, 255, 0.1);
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: white;
            display: flex;
            align-items: center;
            gap: 8px;
            backdrop-filter: blur(10px);
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        /* Stats Container */
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 25px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            background: rgba(255, 255, 255, 0.08);
            transform: translateY(-5px);
        }
        
        .stat-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }
        
        .stat-title {
            font-size: 14px;
            color: #94a3b8;
            font-weight: 500;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
        }
        
        .stat-icon.users {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .stat-icon.live {
            background: linear-gradient(135deg, var(--live-red), #dc2626);
            color: white;
        }
        
        .stat-icon.upcoming {
            background: linear-gradient(135deg, var(--upcoming-orange), #d97706);
            color: white;
        }
        
        .stat-value {
            font-size: 32px;
            font-weight: 700;
            color: white;
            margin-bottom: 5px;
        }
        
        .stat-trend {
            font-size: 14px;
            color: var(--secondary);
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        /* Stream Manager */
        .stream-manager {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 30px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 30px;
        }
        
        .manager-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }
        
        .manager-title {
            font-size: 20px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .manager-title i {
            color: var(--primary);
        }
        
        .live-badge {
            background: var(--live-red);
            color: white;
            padding: 6px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 6px;
            animation: pulse 2s infinite;
        }
        
        @keyframes pulse {
            0% {
                opacity: 1;
            }
            50% {
                opacity: 0.7;
            }
            100% {
                opacity: 1;
            }
        }
        
        /* Countdown Timer */
        .countdown-container {
            background: rgba(0, 0, 0, 0.3);
            border-radius: 12px;
            padding: 25px;
            text-align: center;
            margin-bottom: 30px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .countdown-title {
            font-size: 16px;
            color: #94a3b8;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .countdown-timer {
            display: flex;
            justify-content: center;
            gap: 20px;
            margin-bottom: 20px;
        }
        
        .countdown-unit {
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        
        .countdown-value {
            font-size: 48px;
            font-weight: 800;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            line-height: 1;
        }
        
        .countdown-label {
            font-size: 12px;
            color: #94a3b8;
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 5px;
        }
        
        .next-stream-info {
            font-size: 14px;
            color: #94a3b8;
        }
        
        .next-stream-time {
            color: var(--secondary);
            font-weight: 600;
        }
        
        /* Upcoming Streams Section */
        .upcoming-streams {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 30px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 30px;
        }
        
        .upcoming-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .upcoming-title {
            font-size: 20px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .upcoming-title i {
            color: var(--upcoming-orange);
        }
        
        .streams-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .stream-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            padding: 20px;
            border: 1px solid rgba(255, 255, 255, 0.1);
            transition: all 0.3s ease;
            position: relative;
        }
        
        .stream-card:hover {
            background: rgba(255, 255, 255, 0.08);
            transform: translateY(-3px);
            border-color: var(--upcoming-orange);
        }
        
        .stream-card-header {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 15px;
        }
        
        .stream-card-title {
            font-size: 16px;
            font-weight: 600;
            color: white;
            margin-bottom: 5px;
        }
        
        .stream-card-actions {
            display: flex;
            gap: 8px;
        }
        
        .stream-action-btn {
            width: 36px;
            height: 36px;
            border-radius: 8px;
            border: none;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 14px;
            background: rgba(255, 255, 255, 0.1);
            color: white;
        }
        
        .stream-action-btn:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .stream-action-btn.edit {
            background: rgba(59, 130, 246, 0.2);
            color: var(--primary);
        }
        
        .stream-action-btn.edit:hover {
            background: rgba(59, 130, 246, 0.3);
        }
        
        .stream-action-btn.delete {
            background: rgba(239, 68, 68, 0.2);
            color: var(--danger);
        }
        
        .stream-action-btn.delete:hover {
            background: rgba(239, 68, 68, 0.3);
        }
        
        .stream-card-details {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
            margin-bottom: 15px;
        }
        
        .stream-detail {
            display: flex;
            flex-direction: column;
        }
        
        .stream-detail-label {
            font-size: 12px;
            color: #94a3b8;
            margin-bottom: 3px;
        }
        
        .stream-detail-value {
            font-size: 14px;
            font-weight: 600;
            color: white;
        }
        
        .stream-card-footer {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding-top: 15px;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .stream-status {
            font-size: 12px;
            padding: 4px 10px;
            border-radius: 20px;
            font-weight: 600;
        }
        
        .stream-status.not-started {
            background: rgba(245, 158, 11, 0.2);
            color: var(--upcoming-orange);
        }
        
        .stream-created {
            font-size: 11px;
            color: #94a3b8;
        }
        
        /* Stream Duration (Count Up) */
        .duration-container {
            background: linear-gradient(135deg, rgba(59, 130, 246, 0.1), rgba(29, 78, 216, 0.1));
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            margin-bottom: 30px;
            border: 1px solid rgba(59, 130, 246, 0.2);
        }
        
        .duration-title {
            font-size: 16px;
            color: #94a3b8;
            margin-bottom: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }
        
        .duration-timer {
            display: flex;
            justify-content: center;
            gap: 20px;
        }
        
        .duration-unit {
            display: flex;
            flex-direction: column;
            align-items: center;
        }
        
        .duration-value {
            font-size: 36px;
            font-weight: 800;
            color: var(--primary);
            line-height: 1;
        }
        
        .duration-label {
            font-size: 12px;
            color: #94a3b8;
            text-transform: uppercase;
            letter-spacing: 1px;
            margin-top: 5px;
        }
        
        /* Stream Controls */
        .stream-controls {
            background: rgba(0, 0, 0, 0.3);
            border-radius: 12px;
            padding: 25px;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .controls-title {
            font-size: 18px;
            font-weight: 600;
            color: white;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .controls-title i {
            color: var(--secondary);
        }
        
        .controls-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
            gap: 20px;
        }
        
        .control-group {
            margin-bottom: 20px;
        }
        
        .control-label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: white;
            margin-bottom: 8px;
        }
        
        .control-input {
            width: 100%;
            padding: 12px 16px;
            background: rgba(255, 255, 255, 0.1);
            border: 1px solid rgba(255, 255, 255, 0.2);
            border-radius: 8px;
            color: white;
            font-size: 14px;
            font-family: 'Inter', sans-serif;
            transition: all 0.3s ease;
        }
        
        .control-input:focus {
            outline: none;
            border-color: var(--primary);
            background: rgba(255, 255, 255, 0.15);
        }
        
        .control-input::placeholder {
            color: rgba(255, 255, 255, 0.5);
        }
        
        .control-hint {
            font-size: 12px;
            color: #94a3b8;
            margin-top: 5px;
        }
        
        select.control-input {
            cursor: pointer;
        }
        
        /* Control Buttons */
        .control-buttons {
            display: flex;
            gap: 15px;
            margin-top: 30px;
            padding-top: 25px;
            border-top: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .control-btn {
            padding: 14px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
            min-width: 160px;
        }
        
        .control-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .control-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .control-btn.success {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .control-btn.success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .control-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        .control-btn.danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        .control-btn.secondary {
            background: rgba(255, 255, 255, 0.1);
            color: white;
        }
        
        .control-btn.secondary:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        /* Participants Section */
        .participants-section {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 30px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
            margin-bottom: 30px;
        }
        
        .section-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        /* Participants Grid */
        .participants-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(200px, 1fr));
            gap: 15px;
        }
        
        .participant-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 12px;
            padding: 15px;
            display: flex;
            align-items: center;
            gap: 15px;
            transition: all 0.3s ease;
            cursor: pointer;
            border: 1px solid rgba(255, 255, 255, 0.1);
            text-decoration: none;
        }
        
        .participant-card:hover {
            background: rgba(255, 255, 255, 0.1);
            transform: translateY(-3px);
            border-color: var(--primary);
        }
        
        .participant-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            overflow: hidden;
            flex-shrink: 0;
        }
        
        .participant-avatar img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        
        .default-avatar {
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
        }
        
        .participant-info {
            flex: 1;
            min-width: 0;
        }
        
        .participant-name {
            font-size: 14px;
            font-weight: 600;
            color: white;
            margin-bottom: 3px;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .participant-email {
            font-size: 12px;
            color: #94a3b8;
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
        }
        
        .joined-time {
            font-size: 11px;
            color: var(--secondary);
            margin-top: 3px;
        }
        
        /* Activity Log */
        .activity-log {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 15px;
            padding: 30px;
            backdrop-filter: blur(10px);
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .log-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 25px;
        }
        
        .log-title {
            font-size: 18px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .log-title i {
            color: var(--warning);
        }
        
        .refresh-btn {
            background: rgba(255, 255, 255, 0.1);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .refresh-btn:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .log-list {
            max-height: 300px;
            overflow-y: auto;
        }
        
        .log-item {
            display: flex;
            align-items: center;
            gap: 15px;
            padding: 15px;
            background: rgba(255, 255, 255, 0.05);
            border-radius: 10px;
            margin-bottom: 10px;
            transition: all 0.3s ease;
        }
        
        .log-item:hover {
            background: rgba(255, 255, 255, 0.1);
        }
        
        .log-item:last-child {
            margin-bottom: 0;
        }
        
        .log-icon {
            width: 40px;
            height: 40px;
            border-radius: 10px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 16px;
        }
        
        .log-icon.join {
            background: rgba(16, 185, 129, 0.2);
            color: var(--success);
        }
        
        .log-icon.leave {
            background: rgba(239, 68, 68, 0.2);
            color: var(--danger);
        }
        
        .log-icon.stream {
            background: rgba(59, 130, 246, 0.2);
            color: var(--primary);
        }
        
        .log-content {
            flex: 1;
        }
        
        .log-message {
            font-size: 14px;
            color: white;
            margin-bottom: 3px;
        }
        
        .log-time {
            font-size: 12px;
            color: #94a3b8;
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 40px 20px;
            background: rgba(255, 255, 255, 0.03);
            border-radius: 12px;
            margin-top: 20px;
        }
        
        .empty-state i {
            font-size: 48px;
            margin-bottom: 15px;
            color: rgba(255, 255, 255, 0.2);
        }
        
        .empty-state h3 {
            font-size: 18px;
            margin-bottom: 10px;
            color: white;
        }
        
        .empty-state p {
            font-size: 14px;
            color: #94a3b8;
            margin-bottom: 20px;
        }
        
        /* Modal Styles */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.7);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 10000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
            backdrop-filter: blur(5px);
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        .modal {
            background: rgba(30, 41, 59, 0.95);
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 500px;
            max-height: 80vh;
            overflow-y: auto;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.1);
        }
        
        .modal-overlay.active .modal {
            transform: translateY(0);
        }
        
        .modal-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
        }
        
        .modal-title {
            font-size: 24px;
            font-weight: 600;
            color: white;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .modal-title i {
            color: var(--primary);
        }
        
        .modal-close {
            background: rgba(255, 255, 255, 0.1);
            border: none;
            color: white;
            width: 40px;
            height: 40px;
            border-radius: 10px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .modal-close:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        .modal-content {
            margin-bottom: 30px;
        }
        
        .modal-form .control-group {
            margin-bottom: 20px;
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
            justify-content: flex-end;
        }
        
        .modal-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            min-width: 120px;
        }
        
        .modal-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .modal-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .modal-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        .modal-btn.danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        .modal-btn.secondary {
            background: rgba(255, 255, 255, 0.1);
            color: white;
        }
        
        .modal-btn.secondary:hover {
            background: rgba(255, 255, 255, 0.2);
        }
        
        /* Message Modal */
        .message-modal {
            background: rgba(30, 41, 59, 0.95);
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.1);
            text-align: center;
        }
        
        .message-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .message-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .message-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .message-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: white;
            margin-bottom: 15px;
        }
        
        .message-modal p {
            font-size: 16px;
            color: #94a3b8;
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .message-actions {
            display: flex;
            justify-content: center;
        }
        
        .message-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .message-btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        /* Confirmation Modal */
        .confirm-modal {
            background: rgba(30, 41, 59, 0.95);
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            border: 1px solid rgba(255, 255, 255, 0.1);
            text-align: center;
        }
        
        .confirm-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .confirm-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: white;
            margin-bottom: 15px;
        }
        
        .confirm-modal p {
            font-size: 16px;
            color: #94a3b8;
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .confirm-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        /* Loading Spinner */
        .spinner {
            width: 20px;
            height: 20px;
            border: 3px solid rgba(255, 255, 255, 0.3);
            border-radius: 50%;
            border-top-color: white;
            animation: spin 1s ease-in-out infinite;
        }
        
        @keyframes spin {
            to { transform: rotate(360deg); }
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .countdown-timer, .duration-timer {
                gap: 10px;
            }
            
            .countdown-value, .duration-value {
                font-size: 36px;
            }
            
            .control-buttons, .modal-actions, .confirm-actions {
                flex-direction: column;
            }
            
            .control-btn, .modal-btn {
                width: 100%;
            }
            
            .controls-grid {
                grid-template-columns: 1fr;
            }
            
            .streams-grid {
                grid-template-columns: 1fr;
            }
            
            .participants-grid {
                grid-template-columns: repeat(auto-fill, minmax(150px, 1fr));
            }
            
            .modal, .message-modal, .confirm-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
        }
        
        @media (max-width: 480px) {
            .countdown-value, .duration-value {
                font-size: 28px;
            }
            
            .countdown-label, .duration-label {
                font-size: 10px;
            }
            
            .participants-grid {
                grid-template-columns: 1fr;
            }
            
            .participant-card {
                max-width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <div>
                    <h2>Live Stream Manager</h2>
                    <p class="header-subtitle">Manage audio conference live streams and participants</p>
                </div>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Stats Overview -->
            <div class="stats-container">
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-title">Total Participants</div>
                        <div class="stat-icon users">
                            <i class="fas fa-users"></i>
                        </div>
                    </div>
                    <div class="stat-value" id="totalUsers"><?php echo $total_participants; ?></div>
                    <div class="stat-trend">
                        <i class="fas fa-calendar-day"></i>
                        <span><?php echo $total_users_today; ?> today</span>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-title">Current Status</div>
                        <div class="stat-icon live">
                            <i class="fas fa-broadcast-tower"></i>
                        </div>
                    </div>
                    <div class="stat-value">
                        <?php echo $current_stream ? strtoupper($current_stream['status']) : 'NO STREAM'; ?>
                    </div>
                    <div class="stat-trend">
                        <i class="fas fa-clock"></i>
                        <span id="streamDurationText">
                            <?php echo $current_stream ? $stream_duration : 'No active stream'; ?>
                        </span>
                    </div>
                </div>
                
                <div class="stat-card">
                    <div class="stat-header">
                        <div class="stat-title">Next Stream</div>
                        <div class="stat-icon upcoming">
                            <i class="fas fa-calendar-alt"></i>
                        </div>
                    </div>
                    <div class="stat-value" id="nextStreamTime">
                        <?php echo $next_stream ? date('H:i', strtotime($next_stream['time'])) : '--:--'; ?>
                    </div>
                    <div class="stat-trend">
                        <i class="fas fa-clock"></i>
                        <span id="countdownStatus">
                            <?php if ($next_stream): ?>
                                <?php 
                                if ($countdown_hours == 0 && $countdown_minutes == 0 && $countdown_seconds == 0) {
                                    echo 'Stream time passed';
                                } else {
                                    echo 'Starts in ' . sprintf('%02d:%02d:%02d', $countdown_hours, $countdown_minutes, $countdown_seconds);
                                }
                                ?>
                            <?php else: ?>
                                No upcoming stream
                            <?php endif; ?>
                        </span>
                    </div>
                </div>
            </div>
            
            <!-- Stream Manager -->
            <div class="stream-manager">
                <div class="manager-header">
                    <h3 class="manager-title">
                        <i class="fas fa-sliders-h"></i>
                        Stream Controls
                    </h3>
                    <?php if ($current_stream): ?>
                        <div class="live-badge">
                            <i class="fas fa-circle"></i>
                            <span>LIVE NOW</span>
                        </div>
                    <?php endif; ?>
                </div>
                
                <?php if ($next_stream && $countdown_hours + $countdown_minutes + $countdown_seconds > 0): ?>
                <!-- Countdown Timer -->
                <div class="countdown-container">
                    <div class="countdown-title">
                        <i class="fas fa-hourglass-half"></i>
                        Next Stream Countdown
                    </div>
                    <div class="countdown-timer" id="countdownTimer">
                        <div class="countdown-unit">
                            <div class="countdown-value" id="countdownHours"><?php echo sprintf('%02d', $countdown_hours); ?></div>
                            <div class="countdown-label">Hours</div>
                        </div>
                        <div class="countdown-unit">
                            <div class="countdown-value" id="countdownMinutes"><?php echo sprintf('%02d', $countdown_minutes); ?></div>
                            <div class="countdown-label">Minutes</div>
                        </div>
                        <div class="countdown-unit">
                            <div class="countdown-value" id="countdownSeconds"><?php echo sprintf('%02d', $countdown_seconds); ?></div>
                            <div class="countdown-label">Seconds</div>
                        </div>
                    </div>
                    <div class="next-stream-info">
                        Next stream: 
                        <span class="next-stream-time">
                            <?php echo ucfirst($next_stream['day']); ?> at 
                            <?php echo date('H:i', strtotime($next_stream['time'])); ?>
                        </span>
                    </div>
                </div>
                <?php endif; ?>
                
                <?php if ($current_stream): ?>
                <!-- Stream Duration (Count Up) -->
                <div class="duration-container" id="durationContainer">
                    <div class="duration-title">
                        <i class="fas fa-hourglass-start"></i>
                        Stream Duration
                    </div>
                    <div class="duration-timer">
                        <div class="duration-unit">
                            <div class="duration-value" id="durationHours">00</div>
                            <div class="duration-label">Hours</div>
                        </div>
                        <div class="duration-unit">
                            <div class="duration-value" id="durationMinutes">00</div>
                            <div class="duration-label">Minutes</div>
                        </div>
                        <div class="duration-unit">
                            <div class="duration-value" id="durationSeconds">00</div>
                            <div class="duration-label">Seconds</div>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
                
                <!-- Stream Controls -->
                <div class="stream-controls">
                    <h4 class="controls-title">
                        <i class="fas fa-cogs"></i>
                        Next Stream Configuration
                    </h4>
                    
                    <div class="controls-grid">
                        <div class="control-group">
                            <label class="control-label">Stream Title</label>
                            <input type="text" class="control-input" id="streamTitle" 
                                   placeholder="Enter stream title" 
                                   value="<?php echo htmlspecialchars($next_stream['title'] ?? 'Weekly Audio Conference'); ?>">
                            <div class="control-hint">Maximum 100 characters</div>
                        </div>
                        
                        <div class="control-group">
                            <label class="control-label">Day of Week</label>
                            <select class="control-input" id="streamDay">
                                <option value="sunday" <?php echo ($next_stream['day'] ?? 'sunday') == 'sunday' ? 'selected' : ''; ?>>Sunday</option>
                                <option value="monday" <?php echo ($next_stream['day'] ?? 'sunday') == 'monday' ? 'selected' : ''; ?>>Monday</option>
                                <option value="tuesday" <?php echo ($next_stream['day'] ?? 'sunday') == 'tuesday' ? 'selected' : ''; ?>>Tuesday</option>
                                <option value="wednesday" <?php echo ($next_stream['day'] ?? 'sunday') == 'wednesday' ? 'selected' : ''; ?>>Wednesday</option>
                                <option value="thursday" <?php echo ($next_stream['day'] ?? 'sunday') == 'thursday' ? 'selected' : ''; ?>>Thursday</option>
                                <option value="friday" <?php echo ($next_stream['day'] ?? 'sunday') == 'friday' ? 'selected' : ''; ?>>Friday</option>
                                <option value="saturday" <?php echo ($next_stream['day'] ?? 'sunday') == 'saturday' ? 'selected' : ''; ?>>Saturday</option>
                            </select>
                            <div class="control-hint">Select day for the stream</div>
                        </div>
                        
                        <div class="control-group">
                            <label class="control-label">Stream Time</label>
                            <input type="time" class="control-input" id="streamTime" 
                                   value="<?php echo $next_stream ? date('H:i', strtotime($next_stream['time'])) : '18:00'; ?>">
                            <div class="control-hint">Set time for the stream (24-hour format)</div>
                        </div>
                    </div>
                    
                    <div class="control-buttons">
                        <?php if ($current_stream): ?>
                            <button class="control-btn danger" id="stopStreamBtn">
                                <i class="fas fa-stop-circle"></i> End Live Stream
                            </button>
                        <?php else: ?>
                            <button class="control-btn primary" id="startStreamBtn">
                                <i class="fas fa-play-circle"></i> Start Live Stream
                            </button>
                        <?php endif; ?>
                        
                        <button class="control-btn success" id="saveStreamBtn">
                            <i class="fas fa-save"></i> <?php echo $next_stream ? 'Update Schedule' : 'Schedule Stream'; ?>
                        </button>
                        
                        <button class="control-btn secondary" id="refreshBtn">
                            <i class="fas fa-sync-alt"></i> Refresh
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Upcoming Streams Section -->
            <div class="upcoming-streams">
                <div class="upcoming-header">
                    <h4 class="upcoming-title">
                        <i class="fas fa-calendar-alt"></i>
                        Upcoming Streams
                    </h4>
                    <div class="stat-trend">
                        <i class="fas fa-broadcast-tower"></i>
                        <span><?php echo count($upcoming_streams); ?> scheduled</span>
                    </div>
                </div>
                
                <?php if (!empty($upcoming_streams)): ?>
                    <div class="streams-grid" id="upcomingStreamsList">
                        <?php foreach ($upcoming_streams as $stream): ?>
                            <div class="stream-card" data-stream-id="<?php echo $stream['id']; ?>">
                                <div class="stream-card-header">
                                    <div>
                                        <div class="stream-card-title"><?php echo htmlspecialchars($stream['title']); ?></div>
                                    </div>
                                    <div class="stream-card-actions">
                                        <button class="stream-action-btn edit" onclick="editStream(<?php echo $stream['id']; ?>)">
                                            <i class="fas fa-edit"></i>
                                        </button>
                                        <button class="stream-action-btn delete" onclick="confirmDeleteStream(<?php echo $stream['id']; ?>, '<?php echo htmlspecialchars($stream['title']); ?>')">
                                            <i class="fas fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                                <div class="stream-card-details">
                                    <div class="stream-detail">
                                        <div class="stream-detail-label">Day</div>
                                        <div class="stream-detail-value"><?php echo ucfirst($stream['day']); ?></div>
                                    </div>
                                    <div class="stream-detail">
                                        <div class="stream-detail-label">Time</div>
                                        <div class="stream-detail-value"><?php echo date('H:i', strtotime($stream['time'])); ?></div>
                                    </div>
                                </div>
                                <div class="stream-card-footer">
                                    <div class="stream-status not-started">Not Started</div>
                                    <div class="stream-created">
                                        Created <?php echo date('M j, Y', strtotime($stream['created_at'])); ?>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php else: ?>
                    <div class="empty-state">
                        <i class="fas fa-calendar-times"></i>
                        <h3>No Upcoming Streams</h3>
                        <p>Schedule a new stream using the form above</p>
                    </div>
                <?php endif; ?>
            </div>
            
            <!-- Participants Section -->
            <div class="participants-section" id="participantsSection" <?php echo !$show_participants ? 'style="display: none;"' : ''; ?>>
                <div class="section-header">
                    <h4 class="section-title">
                        <i class="fas fa-user-friends"></i>
                        Recent Participants
                        <?php if ($stream_for_participants && $stream_for_participants['status'] == 'live'): ?>
                            <span class="live-badge" style="font-size: 10px; padding: 4px 8px;">
                                <i class="fas fa-circle"></i>
                                <span>LIVE UPDATING</span>
                            </span>
                        <?php endif; ?>
                    </h4>
                    <div class="stat-trend" id="participantsCount">
                        <i class="fas fa-users"></i>
                        <span><?php echo count($participants); ?> participants shown</span>
                    </div>
                </div>
                
                <div class="participants-grid" id="participantsList">
                    <?php if (!empty($participants)): ?>
                        <?php foreach ($participants as $participant): ?>
                            <a href="profile.php?user_id=<?php echo $participant['user_id']; ?>" class="participant-card">
                                <div class="participant-avatar">
                                    <?php if (!empty($participant['profile_url'])): ?>
                                        <img src="<?php echo htmlspecialchars($participant['profile_url']); ?>" 
                                             alt="<?php echo htmlspecialchars($participant['first_name'] . ' ' . $participant['last_name']); ?>">
                                    <?php else: ?>
                                        <div class="default-avatar">
                                            <?php echo strtoupper(substr($participant['first_name'], 0, 1) . substr($participant['last_name'], 0, 1)); ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="participant-info">
                                    <div class="participant-name">
                                        <?php echo htmlspecialchars($participant['first_name'] . ' ' . $participant['last_name']); ?>
                                    </div>
                                    <div class="participant-email">
                                        <?php echo htmlspecialchars($participant['email']); ?>
                                    </div>
                                    <div class="joined-time">
                                        Joined <?php echo date('M j, H:i', strtotime($participant['created_at'])); ?>
                                    </div>
                                </div>
                            </a>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="empty-state">
                            <i class="fas fa-users-slash"></i>
                            <h3>No Participants Yet</h3>
                            <p>Participants will appear here once they join the stream</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
            
            <!-- Activity Log -->
            <div class="activity-log">
                <div class="log-header">
                    <h4 class="log-title">
                        <i class="fas fa-history"></i>
                        Recent Activity
                    </h4>
                    <button class="refresh-btn" id="refreshLogBtn">
                        <i class="fas fa-sync-alt"></i> Refresh
                    </button>
                </div>
                
                <div class="log-list" id="activityLog">
                    <!-- Activity log items will be generated here -->
                </div>
            </div>
        </div>
    </div>
    
    <!-- Edit Stream Modal -->
    <div class="modal-overlay" id="editStreamModal">
        <div class="modal">
            <div class="modal-header">
                <h3 class="modal-title">
                    <i class="fas fa-edit"></i>
                    Edit Stream
                </h3>
                <button class="modal-close" id="closeEditModal">
                    <i class="fas fa-times"></i>
                </button>
            </div>
            <div class="modal-content">
                <form id="editStreamForm" class="modal-form">
                    <input type="hidden" id="editStreamId" name="stream_id">
                    
                    <div class="control-group">
                        <label class="control-label">Stream Title</label>
                        <input type="text" class="control-input" id="editStreamTitle" name="title" required>
                        <div class="control-hint">Maximum 100 characters</div>
                    </div>
                    
                    <div class="control-group">
                        <label class="control-label">Day of Week</label>
                        <select class="control-input" id="editStreamDay" name="day" required>
                            <option value="sunday">Sunday</option>
                            <option value="monday">Monday</option>
                            <option value="tuesday">Tuesday</option>
                            <option value="wednesday">Wednesday</option>
                            <option value="thursday">Thursday</option>
                            <option value="friday">Friday</option>
                            <option value="saturday">Saturday</option>
                        </select>
                        <div class="control-hint">Select day for the stream</div>
                    </div>
                    
                    <div class="control-group">
                        <label class="control-label">Stream Time</label>
                        <input type="time" class="control-input" id="editStreamTime" name="time" required>
                        <div class="control-hint">Set time for the stream (24-hour format)</div>
                    </div>
                    
                    <div class="modal-actions">
                        <button type="button" class="modal-btn secondary" id="cancelEditBtn">Cancel</button>
                        <button type="submit" class="modal-btn primary" id="saveEditBtn">
                            <span id="saveEditBtnText">Save Changes</span>
                            <div class="spinner" style="display: none;"></div>
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <!-- Delete Confirmation Modal -->
    <div class="modal-overlay" id="deleteConfirmModal">
        <div class="confirm-modal">
            <div class="confirm-icon">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="deleteConfirmTitle">Delete Stream</h3>
            <p id="deleteConfirmMessage">Are you sure you want to delete this stream? This action cannot be undone.</p>
            <div class="confirm-actions">
                <button class="modal-btn secondary" id="cancelDeleteBtn">Cancel</button>
                <button class="modal-btn danger" id="confirmDeleteBtn">
                    <span id="confirmDeleteBtnText">Delete</span>
                    <div class="spinner" style="display: none;"></div>
                </button>
            </div>
        </div>
    </div>
    
    <!-- Message Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="message-modal">
            <div class="message-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="message-actions">
                <button class="message-btn" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <script>
        // Global variables
        let totalUsers = <?php echo $total_participants; ?>;
        let countdownHours = <?php echo $countdown_hours; ?>;
        let countdownMinutes = <?php echo $countdown_minutes; ?>;
        let countdownSeconds = <?php echo $countdown_seconds; ?>;
        let isStreamActive = <?php echo $current_stream ? 'true' : 'false'; ?>;
        let lastParticipantId = <?php echo !empty($participants) ? max(array_column($participants, 'id')) : 0; ?>;
        let streamDurationSeconds = <?php echo $current_stream ? (time() - strtotime($current_stream['created_at'])) : 0; ?>;
        let streamToDelete = null;
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        // Update countdown timer
        function updateCountdown() {
            if (countdownSeconds > 0) {
                countdownSeconds--;
            } else {
                countdownSeconds = 59;
                if (countdownMinutes > 0) {
                    countdownMinutes--;
                } else {
                    countdownMinutes = 59;
                    if (countdownHours > 0) {
                        countdownHours--;
                    } else {
                        // Countdown finished - reload page to get new data
                        window.location.reload();
                        return;
                    }
                }
            }
            
            // Update display if elements exist
            const hoursEl = document.getElementById('countdownHours');
            const minutesEl = document.getElementById('countdownMinutes');
            const secondsEl = document.getElementById('countdownSeconds');
            const statusEl = document.getElementById('countdownStatus');
            
            if (hoursEl) hoursEl.textContent = countdownHours.toString().padStart(2, '0');
            if (minutesEl) minutesEl.textContent = countdownMinutes.toString().padStart(2, '0');
            if (secondsEl) secondsEl.textContent = countdownSeconds.toString().padStart(2, '0');
            
            if (statusEl && countdownHours + countdownMinutes + countdownSeconds > 0) {
                statusEl.textContent = `Starts in ${countdownHours.toString().padStart(2, '0')}:${countdownMinutes.toString().padStart(2, '0')}:${countdownSeconds.toString().padStart(2, '0')}`;
            }
        }
        
        // Update stream duration (count up)
        function updateStreamDuration() {
            if (!isStreamActive) return;
            
            streamDurationSeconds++;
            
            const hours = Math.floor(streamDurationSeconds / 3600);
            const minutes = Math.floor((streamDurationSeconds % 3600) / 60);
            const seconds = streamDurationSeconds % 60;
            
            // Update display
            const hoursEl = document.getElementById('durationHours');
            const minutesEl = document.getElementById('durationMinutes');
            const secondsEl = document.getElementById('durationSeconds');
            const durationText = document.getElementById('streamDurationText');
            
            if (hoursEl) hoursEl.textContent = hours.toString().padStart(2, '0');
            if (minutesEl) minutesEl.textContent = minutes.toString().padStart(2, '0');
            if (secondsEl) secondsEl.textContent = seconds.toString().padStart(2, '0');
            
            if (durationText) {
                if (hours > 0) {
                    durationText.textContent = hours + "h " + minutes + "m";
                } else {
                    durationText.textContent = minutes + "m";
                }
            }
        }
        
        // Fetch stream counts
        async function fetchStreamCounts() {
            try {
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'get_stream_counts',
                        date: new Date().toISOString().split('T')[0]
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    const totalCount = result.total_counts || 0;
                    document.getElementById('totalUsers').textContent = totalCount;
                }
            } catch (error) {
                console.error('Error fetching stream counts:', error);
            }
        }
        
        // Save stream changes
        async function saveStreamChanges() {
            const streamTitle = document.getElementById('streamTitle').value;
            const streamDay = document.getElementById('streamDay').value;
            const streamTime = document.getElementById('streamTime').value;
            
            if (!streamTitle || !streamDay || !streamTime) {
                showMessage('Error', 'Please fill in all fields!', 'error');
                return;
            }
            
            // Format time properly (HH:MM:SS)
            const timeParts = streamTime.split(':');
            if (timeParts.length < 2) {
                showMessage('Error', 'Invalid time format!', 'error');
                return;
            }
            
            const formattedTime = timeParts.length === 2 ? streamTime + ':00' : streamTime;
            
            const saveBtn = document.getElementById('saveStreamBtn');
            const saveBtnText = saveBtn.querySelector('span') || saveBtn;
            const originalText = saveBtnText.textContent;
            
            try {
                // Show loading state
                saveBtn.disabled = true;
                saveBtnText.textContent = 'Saving...';
                
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'update_stream',
                        title: streamTitle,
                        day: streamDay,
                        time: formattedTime,
                        status: 'not started'
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Stream schedule saved successfully!', 'success');
                    // Reload page after 2 seconds to get updated data
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to save stream schedule', 'error');
                }
            } catch (error) {
                console.error('Error saving stream:', error);
                showMessage('Error', 'Error saving stream schedule', 'error');
            } finally {
                // Reset button state
                saveBtn.disabled = false;
                saveBtnText.textContent = originalText;
            }
        }
        
        // Start live stream
        async function startStream() {
            const startBtn = document.getElementById('startStreamBtn');
            const startBtnText = startBtn.querySelector('span') || startBtn;
            const originalText = startBtnText.textContent;
            
            try {
                // Show loading state
                startBtn.disabled = true;
                startBtnText.textContent = 'Starting...';
                
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'start_stream'
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Live stream started successfully!', 'success');
                    // Reload page after 2 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to start live stream', 'error');
                }
            } catch (error) {
                console.error('Error starting stream:', error);
                showMessage('Error', 'Error starting live stream', 'error');
            } finally {
                // Reset button state
                startBtn.disabled = false;
                startBtnText.textContent = originalText;
            }
        }
        
        // Stop live stream
        async function stopStream() {
            const stopBtn = document.getElementById('stopStreamBtn');
            const stopBtnText = stopBtn.querySelector('span') || stopBtn;
            const originalText = stopBtnText.textContent;
            
            try {
                // Show loading state
                stopBtn.disabled = true;
                stopBtnText.textContent = 'Ending...';
                
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'stop_stream'
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Live stream ended successfully!', 'success');
                    // Reload page after 2 seconds
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to end live stream', 'error');
                }
            } catch (error) {
                console.error('Error stopping stream:', error);
                showMessage('Error', 'Error ending live stream', 'error');
            } finally {
                // Reset button state
                stopBtn.disabled = false;
                stopBtnText.textContent = originalText;
            }
        }
        
        // Check for active stream and participants
        async function checkActiveStream() {
            try {
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'get_stream_info'
                    })
                });
                
                const result = await response.json();
                
                if (result.success && result.stream) {
                    const stream = result.stream;
                    
                    // Update stream duration if live
                    if (stream.status === 'live' && !isStreamActive) {
                        // Stream became live, reload page
                        window.location.reload();
                    } else if (stream.status !== 'live' && isStreamActive) {
                        // Stream ended, reload page
                        window.location.reload();
                    }
                    
                    // Show/hide participants section based on stream status and age
                    const participantsSection = document.getElementById('participantsSection');
                    if (stream.status === 'live' || stream.status === 'ended') {
                        participantsSection.style.display = 'block';
                    }
                }
            } catch (error) {
                console.error('Error checking stream info:', error);
            }
        }
        
        // Fetch new participants (for live stream)
        async function fetchNewParticipants() {
            try {
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'get_new_participants',
                        last_id: lastParticipantId
                    })
                });
                
                const result = await response.json();
                
                if (result.success && result.new_participants && result.new_participants.length > 0) {
                    const participantsList = document.getElementById('participantsList');
                    const participantsCount = document.getElementById('participantsCount');
                    
                    // Update last participant ID
                    lastParticipantId = result.last_id;
                    
                    // Append new participants
                    result.new_participants.forEach(participant => {
                        const participantCard = createParticipantCard(participant);
                        participantsList.insertBefore(participantCard, participantsList.firstChild);
                    });
                    
                    // Update count
                    if (participantsCount) {
                        const currentText = participantsCount.querySelector('span');
                        const currentCount = parseInt(currentText.textContent.match(/\d+/)[0]) || 0;
                        const newCount = currentCount + result.new_participants.length;
                        currentText.textContent = newCount + ' participants shown';
                    }
                    
                    // Show message for new participants
                    if (result.new_participants.length === 1) {
                        showMessage('New Participant', result.new_participants[0].first_name + ' joined the stream!', 'info');
                    } else if (result.new_participants.length > 1) {
                        showMessage('New Participants', result.new_participants.length + ' new participants joined!', 'info');
                    }
                }
            } catch (error) {
                console.error('Error fetching new participants:', error);
            }
        }
        
        // Create participant card HTML
        function createParticipantCard(participant) {
            const initials = (participant.first_name.charAt(0) + participant.last_name.charAt(0)).toUpperCase();
            const joinedTime = new Date(participant.created_at).toLocaleString('en-US', {
                month: 'short',
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            });
            
            const card = document.createElement('a');
            card.className = 'participant-card';
            card.href = 'profile.php?user_id=' + participant.user_id;
            card.innerHTML = `
                <div class="participant-avatar">
                    ${participant.profile_url ? 
                        `<img src="${participant.profile_url}" alt="${participant.first_name} ${participant.last_name}">` : 
                        `<div class="default-avatar">${initials}</div>`
                    }
                </div>
                <div class="participant-info">
                    <div class="participant-name">${participant.first_name} ${participant.last_name}</div>
                    <div class="participant-email">${participant.email}</div>
                    <div class="joined-time">Joined ${joinedTime}</div>
                </div>
            `;
            
            return card;
        }
        
        // Edit stream function
        async function editStream(streamId) {
            try {
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'get_stream_by_id',
                        stream_id: streamId
                    })
                });
                
                const result = await response.json();
                
                if (result.success && result.stream) {
                    const stream = result.stream;
                    
                    // Fill the form with stream data
                    document.getElementById('editStreamId').value = stream.id;
                    document.getElementById('editStreamTitle').value = stream.title;
                    document.getElementById('editStreamDay').value = stream.day;
                    document.getElementById('editStreamTime').value = stream.time.substring(0, 5); // Convert HH:MM:SS to HH:MM
                    
                    // Show the modal
                    document.getElementById('editStreamModal').classList.add('active');
                } else {
                    showMessage('Error', 'Failed to load stream data', 'error');
                }
            } catch (error) {
                console.error('Error loading stream:', error);
                showMessage('Error', 'Error loading stream data', 'error');
            }
        }
        
        // Confirm delete stream
        function confirmDeleteStream(streamId, streamTitle) {
            streamToDelete = streamId;
            document.getElementById('deleteConfirmTitle').textContent = 'Delete Stream';
            document.getElementById('deleteConfirmMessage').textContent = `Are you sure you want to delete the stream "${streamTitle}"? This action cannot be undone.`;
            document.getElementById('deleteConfirmModal').classList.add('active');
        }
        
        // Delete stream function
        async function deleteStream() {
            if (!streamToDelete) return;
            
            const deleteBtn = document.getElementById('confirmDeleteBtn');
            const deleteBtnText = deleteBtn.querySelector('span') || deleteBtn;
            const originalText = deleteBtnText.textContent;
            
            try {
                // Show loading state
                deleteBtn.disabled = true;
                deleteBtnText.textContent = 'Deleting...';
                
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'delete_stream',
                        stream_id: streamToDelete
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Stream deleted successfully!', 'success');
                    
                    // Close modal
                    document.getElementById('deleteConfirmModal').classList.remove('active');
                    
                    // Remove the stream card from the UI
                    const streamCard = document.querySelector(`.stream-card[data-stream-id="${streamToDelete}"]`);
                    if (streamCard) {
                        streamCard.remove();
                    }
                    
                    // Update count
                    const upcomingStreams = document.getElementById('upcomingStreamsList');
                    const streamCards = upcomingStreams.querySelectorAll('.stream-card');
                    const countElement = document.querySelector('.upcoming-streams .stat-trend span');
                    if (countElement) {
                        countElement.textContent = streamCards.length + ' scheduled';
                    }
                    
                    // Reload page after 2 seconds to update configuration
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to delete stream', 'error');
                }
            } catch (error) {
                console.error('Error deleting stream:', error);
                showMessage('Error', 'Error deleting stream', 'error');
            } finally {
                // Reset button state
                deleteBtn.disabled = false;
                deleteBtnText.textContent = originalText;
                streamToDelete = null;
            }
        }
        
        // Save edited stream
        async function saveEditedStream(e) {
            e.preventDefault();
            
            const streamId = document.getElementById('editStreamId').value;
            const title = document.getElementById('editStreamTitle').value;
            const day = document.getElementById('editStreamDay').value;
            const time = document.getElementById('editStreamTime').value;
            
            if (!title || !day || !time) {
                showMessage('Error', 'Please fill in all fields!', 'error');
                return;
            }
            
            // Format time properly (HH:MM:SS)
            const timeParts = time.split(':');
            if (timeParts.length < 2) {
                showMessage('Error', 'Invalid time format!', 'error');
                return;
            }
            
            const formattedTime = timeParts.length === 2 ? time + ':00' : time;
            
            const saveBtn = document.getElementById('saveEditBtn');
            const saveBtnText = saveBtn.querySelector('span') || saveBtn;
            const originalText = saveBtnText.textContent;
            
            try {
                // Show loading state
                saveBtn.disabled = true;
                saveBtnText.textContent = 'Saving...';
                
                const response = await fetch('b_stream.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: JSON.stringify({
                        action: 'update_stream',
                        stream_id: streamId,
                        title: title,
                        day: day,
                        time: formattedTime,
                        status: 'not started'
                    })
                });
                
                const result = await response.json();
                
                if (result.success) {
                    showMessage('Success', 'Stream updated successfully!', 'success');
                    
                    // Close modal
                    document.getElementById('editStreamModal').classList.remove('active');
                    
                    // Reload page after 2 seconds to get updated data
                    setTimeout(() => {
                        window.location.reload();
                    }, 2000);
                } else {
                    showMessage('Error', result.message || 'Failed to update stream', 'error');
                }
            } catch (error) {
                console.error('Error updating stream:', error);
                showMessage('Error', 'Error updating stream', 'error');
            } finally {
                // Reset button state
                saveBtn.disabled = false;
                saveBtnText.textContent = originalText;
            }
        }
        
        // Generate activity log items
        function generateActivityLog() {
            const logContainer = document.getElementById('activityLog');
            if (logContainer) {
                logContainer.innerHTML = `
                    <div class="log-item">
                        <div class="log-icon stream">
                            <i class="fas fa-broadcast-tower"></i>
                        </div>
                        <div class="log-content">
                            <div class="log-message">Stream manager loaded successfully</div>
                            <div class="log-time">Just now</div>
                        </div>
                    </div>
                    <div class="log-item">
                        <div class="log-icon join">
                            <i class="fas fa-user-plus"></i>
                        </div>
                        <div class="log-content">
                            <div class="log-message">System initialized with <?php echo $total_participants; ?> total participants</div>
                            <div class="log-time">Just now</div>
                        </div>
                    </div>
                `;
            }
        }
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon based on type
            icon.className = 'message-icon ' + type;
            if (type === 'success') {
                icon.innerHTML = '<i class="fas fa-check-circle"></i>';
            } else if (type === 'error') {
                icon.innerHTML = '<i class="fas fa-times-circle"></i>';
            } else if (type === 'info') {
                icon.innerHTML = '<i class="fas fa-info-circle"></i>';
            }
            
            modal.classList.add('active');
        }
        
        // Close modal functions
        function closeModal(modalId) {
            document.getElementById(modalId).classList.remove('active');
        }
        
        // Event listeners
        document.getElementById('saveStreamBtn')?.addEventListener('click', saveStreamChanges);
        document.getElementById('startStreamBtn')?.addEventListener('click', startStream);
        document.getElementById('stopStreamBtn')?.addEventListener('click', stopStream);
        
        document.getElementById('refreshBtn')?.addEventListener('click', () => {
            showMessage('Info', 'Refreshing data...', 'info');
            fetchStreamCounts();
            checkActiveStream();
        });
        
        document.getElementById('refreshLogBtn')?.addEventListener('click', () => {
            showMessage('Info', 'Activity log refreshed!', 'info');
            generateActivityLog();
        });
        
        // Edit stream form submission
        document.getElementById('editStreamForm')?.addEventListener('submit', saveEditedStream);
        
        // Delete stream confirmation
        document.getElementById('confirmDeleteBtn')?.addEventListener('click', deleteStream);
        
        // Modal close buttons
        document.getElementById('closeEditModal')?.addEventListener('click', () => closeModal('editStreamModal'));
        document.getElementById('cancelEditBtn')?.addEventListener('click', () => closeModal('editStreamModal'));
        document.getElementById('cancelDeleteBtn')?.addEventListener('click', () => closeModal('deleteConfirmModal'));
        document.getElementById('messageOkBtn')?.addEventListener('click', () => closeModal('messageModal'));
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                }
            });
        });
        
        // Initialize
        document.addEventListener('DOMContentLoaded', () => {
            updateCurrentTime();
            
            // Start countdown if there's a next stream
            if (countdownHours + countdownMinutes + countdownSeconds > 0) {
                setInterval(updateCountdown, 1000);
            }
            
            // Start stream duration timer if stream is live
            if (isStreamActive) {
                setInterval(updateStreamDuration, 1000);
                updateStreamDuration(); // Initial update
                
                // Check for new participants every 3 seconds
                setInterval(fetchNewParticipants, 3000);
            }
            
            generateActivityLog();
            
            // Set interval for updates
            setInterval(updateCurrentTime, 1000);
            setInterval(fetchStreamCounts, 30000); // Update counts every 30 seconds
            setInterval(checkActiveStream, 10000); // Check stream status every 10 seconds
        });
    </script>
</body>
</html>