<?php
// testimonies.php
session_start();

// Check if user is logged in
if (!isset($_SESSION['admin_id'])) {
    header('Location: login.php');
    exit;
}
// Check permission - only moderators and root can access
if (!isset($_SESSION['admin_role']) || ($_SESSION['admin_role'] !== 'moderator' && $_SESSION['admin_role'] !== 'root')) {
    header('Location: overview.php');
    exit;
}

// Include database connection
require_once 'php/db.php';

// Function to get user initials
function getUserInitials($name) {
    $names = explode(' ', $name);
    $initials = '';
    foreach ($names as $name) {
        $initials .= strtoupper(substr($name, 0, 1));
    }
    return substr($initials, 0, 2);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Testimonies Management - BCA Admin Panel</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        :root {
            --primary: #3b82f6;
            --primary-dark: #1d4ed8;
            --secondary: #10b981;
            --secondary-dark: #059669;
            --danger: #ef4444;
            --warning: #f59e0b;
            --dark: #1f2937;
            --light: #f9fafb;
            --gray: #6b7280;
            --gray-light: #e5e7eb;
            --success: #10b981;
        }
        
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%);
            min-height: 100vh;
            color: var(--dark);
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        /* Main Content Styles */
        .main-content {
            flex: 1;
            margin-left: 260px;
            padding: 30px;
            transition: all 0.3s ease;
        }
        
        .sidebar.collapsed ~ .main-content {
            margin-left: 80px;
        }
        
        .header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 30px;
            padding-bottom: 20px;
            border-bottom: 1px solid var(--gray-light);
        }
        
        .header h2 {
            font-size: 28px;
            font-weight: 700;
            color: var(--dark);
        }
        
        .header-actions {
            display: flex;
            gap: 15px;
            align-items: center;
        }
        
        .time-display {
            background: white;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 500;
            color: var(--dark);
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .time-display i {
            color: var(--secondary);
        }
        
        .refresh-btn {
            background: var(--primary);
            color: white;
            border: none;
            padding: 12px 20px;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .refresh-btn:hover {
            background: var(--primary-dark);
            transform: translateY(-2px);
        }
        
        .refresh-btn.loading i {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            from { transform: rotate(0deg); }
            to { transform: rotate(360deg); }
        }
        
        /* Stats Overview */
        .stats-overview {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background: white;
            padding: 20px;
            border-radius: 12px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
            display: flex;
            align-items: center;
            gap: 15px;
            cursor: pointer;
            transition: all 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 8px 20px rgba(0, 0, 0, 0.1);
        }
        
        .stat-card.active {
            border: 2px solid var(--primary);
        }
        
        .stat-icon {
            width: 50px;
            height: 50px;
            border-radius: 12px;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 22px;
            color: white;
        }
        
        .icon-pending { background: linear-gradient(135deg, #f59e0b, #d97706); }
        .icon-approved { background: linear-gradient(135deg, #10b981, #059669); }
        .icon-rejected { background: linear-gradient(135deg, #ef4444, #dc2626); }
        .icon-waiting { background: linear-gradient(135deg, #8b5cf6, #7c3aed); }
        
        .stat-info h3 {
            font-size: 24px;
            font-weight: 700;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .stat-info p {
            font-size: 14px;
            color: var(--gray);
            font-weight: 500;
        }
        
        /* Tabs */
        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 25px;
            border-bottom: 2px solid var(--gray-light);
            padding-bottom: 10px;
        }
        
        .tab {
            padding: 12px 24px;
            background: white;
            border: 2px solid var(--gray-light);
            border-radius: 10px;
            font-size: 14px;
            font-weight: 600;
            color: var(--gray);
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .tab:hover {
            background: var(--light);
            color: var(--primary);
        }
        
        .tab.active {
            background: var(--primary);
            color: white;
            border-color: var(--primary);
        }
        
        /* Testimonies List */
        .testimonies-section {
            background: white;
            border-radius: 15px;
            padding: 30px;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.05);
        }
        
        .section-title {
            font-size: 20px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 25px;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .section-title i {
            color: var(--primary);
        }
        
        .testimonies-list {
            display: flex;
            flex-direction: column;
            gap: 15px;
        }
        
        .testimony-item {
            background: var(--light);
            border-radius: 12px;
            padding: 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            transition: all 0.3s ease;
            border-left: 4px solid var(--warning);
        }
        
        .testimony-item:hover {
            background: #f1f5f9;
            transform: translateX(5px);
        }
        
        .testimony-item.approved {
            border-left-color: var(--success);
        }
        
        .testimony-item.rejected {
            border-left-color: var(--danger);
        }
        
        .testimony-item.waiting {
            border-left-color: var(--primary);
        }
        
        .testimony-user {
            display: flex;
            align-items: center;
            gap: 15px;
            flex: 1;
        }
        
        .user-avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: 600;
            font-size: 18px;
            flex-shrink: 0;
            overflow: hidden;
        }
        
        .user-avatar img {
            width: 100%;
            height: 100%;
            border-radius: 50%;
            object-fit: cover;
        }
        
        .user-info h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 5px;
        }
        
        .user-info p {
            font-size: 14px;
            color: var(--gray);
            margin-bottom: 3px;
        }
        
        .testimony-details {
            flex: 2;
            padding: 0 20px;
        }
        
        .testimony-details h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 8px;
        }
        
        .order-info {
            display: flex;
            gap: 20px;
            flex-wrap: wrap;
        }
        
        .order-info-item {
            display: flex;
            flex-direction: column;
        }
        
        .order-info-item label {
            font-size: 12px;
            color: var(--gray);
            margin-bottom: 4px;
            font-weight: 500;
        }
        
        .order-info-item span {
            font-size: 14px;
            font-weight: 600;
            color: var(--dark);
        }
        
        .testimony-actions {
            display: flex;
            gap: 10px;
            flex-shrink: 0;
        }
        
        .view-btn, .approve-btn, .reject-btn {
            padding: 8px 16px;
            border-radius: 8px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            display: flex;
            align-items: center;
            gap: 6px;
        }
        
        .view-btn {
            background: var(--primary);
            color: white;
        }
        
        .view-btn:hover {
            background: var(--primary-dark);
        }
        
        .approve-btn {
            background: var(--success);
            color: white;
        }
        
        .approve-btn:hover {
            background: var(--secondary-dark);
        }
        
        .reject-btn {
            background: var(--danger);
            color: white;
        }
        
        .reject-btn:hover {
            background: #dc2626;
        }
        
        .status-badge {
            padding: 4px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 600;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }
        
        .status-pending {
            background: #fef3c7;
            color: #92400e;
        }
        
        .status-approved {
            background: #d1fae5;
            color: #065f46;
        }
        
        .status-rejected {
            background: #fee2e2;
            color: #991b1b;
        }
        
        .status-waiting {
            background: #dbeafe;
            color: #1e40af;
        }
        
        /* Tab Content */
        .tab-content {
            display: none;
        }
        
        .tab-content.active {
            display: block;
            animation: fadeIn 0.3s ease;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Image Modal */
        .image-modal {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.9);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 3000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .image-modal.active {
            opacity: 1;
            visibility: visible;
        }
        
        .image-modal-content {
            position: relative;
            max-width: 90%;
            max-height: 90%;
        }
        
        .image-modal-content img {
            max-width: 100%;
            max-height: 80vh;
            border-radius: 8px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.5);
        }
        
        .close-modal {
            position: absolute;
            top: -40px;
            right: 0;
            background: var(--danger);
            color: white;
            border: none;
            width: 36px;
            height: 36px;
            border-radius: 50%;
            font-size: 18px;
            cursor: pointer;
            display: flex;
            align-items: center;
            justify-content: center;
            transition: all 0.3s ease;
        }
        
        .close-modal:hover {
            background: #dc2626;
            transform: rotate(90deg);
        }
        
        .image-info {
            position: absolute;
            bottom: -50px;
            left: 0;
            right: 0;
            color: white;
            text-align: center;
            font-size: 14px;
        }
        
        /* Custom Modal Overlay */
        .modal-overlay {
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0, 0, 0, 0.5);
            display: flex;
            align-items: center;
            justify-content: center;
            z-index: 4000;
            opacity: 0;
            visibility: hidden;
            transition: all 0.3s ease;
        }
        
        .modal-overlay.active {
            opacity: 1;
            visibility: visible;
        }
        
        /* Success/Error Modal */
        .alert-modal {
            background: white;
            border-radius: 15px;
            padding: 40px;
            width: 100%;
            max-width: 400px;
            box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
            transform: translateY(-20px);
            transition: all 0.3s ease;
            text-align: center;
        }
        
        .modal-overlay.active .alert-modal {
            transform: translateY(0);
        }
        
        .alert-icon {
            width: 80px;
            height: 80px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            font-size: 36px;
            margin: 0 auto 25px auto;
        }
        
        .alert-icon.success {
            background: linear-gradient(135deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-icon.error {
            background: linear-gradient(135deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-icon.warning {
            background: linear-gradient(135deg, var(--warning), #d97706);
            color: white;
        }
        
        .alert-icon.info {
            background: linear-gradient(135deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-modal h3 {
            font-size: 24px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 15px;
        }
        
        .alert-modal p {
            font-size: 16px;
            color: var(--gray);
            margin-bottom: 30px;
            line-height: 1.5;
        }
        
        .alert-actions {
            display: flex;
            gap: 15px;
            justify-content: center;
        }
        
        .alert-btn {
            padding: 12px 30px;
            border-radius: 10px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            border: none;
            min-width: 120px;
        }
        
        .alert-btn.primary {
            background: linear-gradient(90deg, var(--primary), var(--primary-dark));
            color: white;
        }
        
        .alert-btn.primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(59, 130, 246, 0.3);
        }
        
        .alert-btn.secondary {
            background: var(--gray-light);
            color: var(--gray);
        }
        
        .alert-btn.secondary:hover {
            background: #d1d5db;
        }
        
        .alert-btn.success {
            background: linear-gradient(90deg, var(--success), var(--secondary-dark));
            color: white;
        }
        
        .alert-btn.success:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(16, 185, 129, 0.3);
        }
        
        .alert-btn.danger {
            background: linear-gradient(90deg, var(--danger), #dc2626);
            color: white;
        }
        
        .alert-btn.danger:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(239, 68, 68, 0.3);
        }
        
        /* Toast Notification */
        .toast {
            position: fixed;
            top: 20px;
            right: 20px;
            background: white;
            padding: 16px 24px;
            border-radius: 10px;
            box-shadow: 0 8px 25px rgba(0, 0, 0, 0.15);
            display: flex;
            align-items: center;
            gap: 12px;
            z-index: 5000;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            max-width: 400px;
        }
        
        .toast.active {
            opacity: 1;
            transform: translateX(0);
        }
        
        .toast.success {
            border-left: 4px solid var(--success);
        }
        
        .toast.error {
            border-left: 4px solid var(--danger);
        }
        
        .toast.warning {
            border-left: 4px solid var(--warning);
        }
        
        .toast.info {
            border-left: 4px solid var(--primary);
        }
        
        .toast-icon {
            font-size: 20px;
        }
        
        .toast.success .toast-icon {
            color: var(--success);
        }
        
        .toast.error .toast-icon {
            color: var(--danger);
        }
        
        .toast.warning .toast-icon {
            color: var(--warning);
        }
        
        .toast.info .toast-icon {
            color: var(--primary);
        }
        
        .toast-content {
            flex: 1;
        }
        
        .toast-content h4 {
            font-size: 16px;
            font-weight: 600;
            color: var(--dark);
            margin-bottom: 4px;
        }
        
        .toast-content p {
            font-size: 14px;
            color: var(--gray);
        }
        
        .close-toast {
            background: none;
            border: none;
            color: var(--gray);
            cursor: pointer;
            font-size: 16px;
            padding: 4px;
            transition: color 0.3s ease;
        }
        
        .close-toast:hover {
            color: var(--danger);
        }
        
        /* Auto-refresh Indicator */
        .auto-refresh-indicator {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: var(--primary);
            color: white;
            padding: 8px 12px;
            border-radius: 20px;
            font-size: 12px;
            font-weight: 500;
            display: flex;
            align-items: center;
            gap: 6px;
            opacity: 0.7;
            z-index: 1000;
        }
        
        .auto-refresh-indicator i {
            animation: pulse 1s infinite;
        }
        
        @keyframes pulse {
            0%, 100% { opacity: 1; }
            50% { opacity: 0.5; }
        }
        
        /* Empty State */
        .empty-state {
            text-align: center;
            padding: 60px 20px;
            color: var(--gray);
        }
        
        .empty-state i {
            font-size: 64px;
            margin-bottom: 20px;
            opacity: 0.5;
        }
        
        .empty-state h3 {
            font-size: 20px;
            font-weight: 600;
            margin-bottom: 10px;
        }
        
        .empty-state p {
            font-size: 16px;
            opacity: 0.7;
        }
        
        /* Responsive */
        @media (max-width: 1200px) {
            .sidebar:not(.collapsed) ~ .main-content {
                margin-left: 80px;
            }
        }
        
        @media (max-width: 992px) {
            .testimony-item {
                flex-direction: column;
                gap: 20px;
                align-items: stretch;
            }
            
            .testimony-user, .testimony-details {
                padding: 0;
            }
            
            .testimony-actions {
                justify-content: flex-end;
                margin-top: 15px;
            }
            
            .tabs {
                flex-wrap: wrap;
            }
        }
        
        @media (max-width: 768px) {
            .main-content {
                padding: 20px;
            }
            
            .header {
                flex-direction: column;
                gap: 15px;
                align-items: flex-start;
            }
            
            .stats-overview {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .testimonies-section {
                padding: 20px;
            }
            
            .order-info {
                flex-direction: column;
                gap: 10px;
            }
            
            .alert-modal {
                padding: 30px 20px;
                margin: 0 20px;
            }
            
            .alert-actions {
                flex-direction: column;
            }
            
            .alert-btn {
                width: 100%;
            }
        }
        
        @media (max-width: 480px) {
            .stats-overview {
                grid-template-columns: 1fr;
            }
            
            .tabs {
                flex-direction: column;
            }
            
            .tab {
                width: 100%;
                justify-content: center;
            }
            
            .testimony-user {
                flex-direction: column;
                text-align: center;
            }
            
            .testimony-actions {
                flex-direction: column;
            }
            
            .view-btn, .approve-btn, .reject-btn {
                width: 100%;
                justify-content: center;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <!-- Include Sidebar Menu -->
        <?php include 'menu.php'; ?>
        
        <!-- Main Content -->
        <div class="main-content">
            <div class="header">
                <h2>Testimonies Management</h2>
                <div class="header-actions">
                    <div class="time-display">
                        <i class="fas fa-clock"></i>
                        <span id="currentTime"><?php echo date('F j, Y H:i:s'); ?></span>
                    </div>
                    <button class="refresh-btn" id="refreshBtn">
                        <i class="fas fa-sync-alt"></i> Refresh
                    </button>
                </div>
            </div>
            
            <!-- Stats Overview -->
            <div class="stats-overview">
                <div class="stat-card active" data-status="pending" id="pendingStat">
                    <div class="stat-icon icon-pending">
                        <i class="fas fa-clock"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="pendingCount">0</h3>
                        <p>Pending Testimonies</p>
                    </div>
                </div>
                
                <div class="stat-card" data-status="approved" id="approvedStat">
                    <div class="stat-icon icon-approved">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="approvedCount">0</h3>
                        <p>Approved</p>
                    </div>
                </div>
                
                <div class="stat-card" data-status="rejected" id="rejectedStat">
                    <div class="stat-icon icon-rejected">
                        <i class="fas fa-times-circle"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="rejectedCount">0</h3>
                        <p>Rejected</p>
                    </div>
                </div>
                
                <div class="stat-card" data-status="waiting" id="waitingStat">
                    <div class="stat-icon icon-waiting">
                        <i class="fas fa-hourglass-half"></i>
                    </div>
                    <div class="stat-info">
                        <h3 id="waitingCount">0</h3>
                        <p>Waiting</p>
                    </div>
                </div>
            </div>
            
            <!-- Tabs -->
            <div class="tabs">
                <div class="tab active" data-tab="pending">
                    <i class="fas fa-clock"></i> Pending Review
                </div>
                <div class="tab" data-tab="approved">
                    <i class="fas fa-check-circle"></i> Approved Testimonies
                </div>
                <div class="tab" data-tab="rejected">
                    <i class="fas fa-times-circle"></i> Rejected Testimonies
                </div>
                <div class="tab" data-tab="waiting">
                    <i class="fas fa-hourglass-half"></i> Waiting for Testimony
                </div>
            </div>
            
            <!-- Testimonies List -->
            <div class="testimonies-section">
                <h3 class="section-title" id="sectionTitle">
                    <i class="fas fa-comment"></i>
                    Pending Testimonies
                </h3>
                
                <!-- Pending Tab Content -->
                <div class="tab-content active" id="pendingTab">
                    <div class="testimonies-list" id="pendingList">
                        <!-- Pending testimonies will be loaded here -->
                        <div class="empty-state">
                            <i class="fas fa-comment-slash"></i>
                            <h3>No Pending Testimonies</h3>
                            <p>All testimonies have been reviewed</p>
                        </div>
                    </div>
                </div>
                
                <!-- Approved Tab Content -->
                <div class="tab-content" id="approvedTab">
                    <div class="testimonies-list" id="approvedList">
                        <!-- Approved testimonies will be loaded here -->
                        <div class="empty-state">
                            <i class="fas fa-comment-check"></i>
                            <h3>No Approved Testimonies</h3>
                            <p>No testimonies have been approved yet</p>
                        </div>
                    </div>
                </div>
                
                <!-- Rejected Tab Content -->
                <div class="tab-content" id="rejectedTab">
                    <div class="testimonies-list" id="rejectedList">
                        <!-- Rejected testimonies will be loaded here -->
                        <div class="empty-state">
                            <i class="fas fa-comment-times"></i>
                            <h3>No Rejected Testimonies</h3>
                            <p>No testimonies have been rejected yet</p>
                        </div>
                    </div>
                </div>
                
                <!-- Waiting Tab Content -->
                <div class="tab-content" id="waitingTab">
                    <div class="testimonies-list" id="waitingList">
                        <!-- Waiting testimonies will be loaded here -->
                        <div class="empty-state">
                            <i class="fas fa-comment-medical"></i>
                            <h3>No Waiting Testimonies</h3>
                            <p>All users have submitted their testimonies</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Image Modal -->
    <div class="image-modal" id="imageModal">
        <div class="image-modal-content">
            <button class="close-modal" id="closeImageModal">
                <i class="fas fa-times"></i>
            </button>
            <img id="modalImage" src="" alt="Testimony Image">
            <div class="image-info">
                <p id="imageInfo">Loading...</p>
            </div>
        </div>
    </div>
    
    <!-- Custom Modals -->
    <div class="modal-overlay" id="confirmModal">
        <div class="alert-modal">
            <div class="alert-icon warning">
                <i class="fas fa-exclamation-triangle"></i>
            </div>
            <h3 id="confirmTitle">Confirm Action</h3>
            <p id="confirmMessage">Are you sure you want to perform this action?</p>
            <div class="alert-actions">
                <button class="alert-btn secondary" id="confirmCancelBtn">Cancel</button>
                <button class="alert-btn danger" id="confirmActionBtn">Confirm</button>
            </div>
        </div>
    </div>
    
    <!-- Success/Error Modal -->
    <div class="modal-overlay" id="messageModal">
        <div class="alert-modal">
            <div class="alert-icon success" id="messageIcon">
                <i class="fas fa-check-circle"></i>
            </div>
            <h3 id="messageTitle">Success</h3>
            <p id="messageText">Operation completed successfully!</p>
            <div class="alert-actions">
                <button class="alert-btn primary" id="messageOkBtn">OK</button>
            </div>
        </div>
    </div>
    
    <!-- Toast Notification -->
    <div class="toast" id="toastNotification">
        <div class="toast-icon">
            <i class="fas fa-info-circle"></i>
        </div>
        <div class="toast-content">
            <h4 id="toastTitle">Notification</h4>
            <p id="toastMessage">This is a notification message</p>
        </div>
        <button class="close-toast" id="closeToast">
            <i class="fas fa-times"></i>
        </button>
    </div>
    
    <!-- Auto-refresh Indicator -->
    <div class="auto-refresh-indicator">
        <i class="fas fa-sync-alt"></i>
        <span>Auto-refresh: 3s</span>
    </div>
    
    <script>
        // Global variables
        let currentTestimonyId = null;
        let confirmCallback = null;
        let autoRefreshInterval;
        let currentTab = 'pending';
        
        // Update current time
        function updateCurrentTime() {
            const now = new Date();
            const options = { 
                weekday: 'long', 
                year: 'numeric', 
                month: 'long', 
                day: 'numeric',
                hour: '2-digit',
                minute: '2-digit',
                second: '2-digit'
            };
            document.getElementById('currentTime').textContent = now.toLocaleDateString('en-US', options);
        }
        
        setInterval(updateCurrentTime, 1000);
        updateCurrentTime();
        
        // Show confirmation modal
        function showConfirm(title, message, callback) {
            document.getElementById('confirmTitle').textContent = title;
            document.getElementById('confirmMessage').textContent = message;
            confirmCallback = callback;
            document.getElementById('confirmModal').classList.add('active');
        }
        
        // Show message modal
        function showMessage(title, message, type = 'success') {
            const modal = document.getElementById('messageModal');
            const icon = document.getElementById('messageIcon');
            const titleEl = document.getElementById('messageTitle');
            const messageEl = document.getElementById('messageText');
            
            titleEl.textContent = title;
            messageEl.textContent = message;
            
            // Set icon and color based on type
            icon.className = 'alert-icon ' + type;
            switch(type) {
                case 'success':
                    icon.innerHTML = '<i class="fas fa-check-circle"></i>';
                    break;
                case 'error':
                    icon.innerHTML = '<i class="fas fa-times-circle"></i>';
                    break;
                case 'warning':
                    icon.innerHTML = '<i class="fas fa-exclamation-triangle"></i>';
                    break;
                case 'info':
                    icon.innerHTML = '<i class="fas fa-info-circle"></i>';
                    break;
            }
            
            modal.classList.add('active');
        }
        
        // Show toast notification
        function showToast(title, message, type = 'info', duration = 3000) {
            const toast = document.getElementById('toastNotification');
            const toastTitle = document.getElementById('toastTitle');
            const toastMessage = document.getElementById('toastMessage');
            const toastIcon = toast.querySelector('.toast-icon i');
            
            toastTitle.textContent = title;
            toastMessage.textContent = message;
            
            // Set type
            toast.className = 'toast ' + type;
            switch(type) {
                case 'success':
                    toastIcon.className = 'fas fa-check-circle';
                    break;
                case 'error':
                    toastIcon.className = 'fas fa-times-circle';
                    break;
                case 'warning':
                    toastIcon.className = 'fas fa-exclamation-triangle';
                    break;
                case 'info':
                    toastIcon.className = 'fas fa-info-circle';
                    break;
            }
            
            toast.classList.add('active');
            
            // Auto hide
            setTimeout(() => {
                toast.classList.remove('active');
            }, duration);
        }
        
        // Load testimonies for specific status
        async function loadTestimonies(status = 'pending') {
            try {
                const formData = new FormData();
                formData.append('action', 'get_testimonies');
                formData.append('status', status);
                
                const response = await fetch('b_testimony.php', {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                
                if (data.success) {
                    updateTestimoniesList(data.testimonies, status);
                    updateStats(data.stats);
                } else {
                    showToast('Error', data.message || 'Failed to load testimonies', 'error');
                }
            } catch (error) {
                console.error('Error loading testimonies:', error);
                showToast('Error', 'Error loading testimonies', 'error');
            }
        }
        
        // Update testimonies list for specific tab
        function updateTestimoniesList(testimonies, status) {
            const listContainer = document.getElementById(status + 'List');
            const sectionTitle = document.getElementById('sectionTitle');
            const icon = sectionTitle.querySelector('i');
            
            // Update section title
            switch(status) {
                case 'pending':
                    sectionTitle.innerHTML = '<i class="fas fa-clock"></i> Pending Testimonies';
                    break;
                case 'approved':
                    sectionTitle.innerHTML = '<i class="fas fa-check-circle"></i> Approved Testimonies';
                    break;
                case 'rejected':
                    sectionTitle.innerHTML = '<i class="fas fa-times-circle"></i> Rejected Testimonies';
                    break;
                case 'waiting':
                    sectionTitle.innerHTML = '<i class="fas fa-hourglass-half"></i> Waiting for Testimony';
                    break;
            }
            
            if (!testimonies || testimonies.length === 0) {
                let emptyStateHtml = '';
                switch(status) {
                    case 'pending':
                        emptyStateHtml = `
                            <div class="empty-state">
                                <i class="fas fa-comment-slash"></i>
                                <h3>No Pending Testimonies</h3>
                                <p>All testimonies have been reviewed</p>
                            </div>
                        `;
                        break;
                    case 'approved':
                        emptyStateHtml = `
                            <div class="empty-state">
                                <i class="fas fa-comment-check"></i>
                                <h3>No Approved Testimonies</h3>
                                <p>No testimonies have been approved yet</p>
                            </div>
                        `;
                        break;
                    case 'rejected':
                        emptyStateHtml = `
                            <div class="empty-state">
                                <i class="fas fa-comment-times"></i>
                                <h3>No Rejected Testimonies</h3>
                                <p>No testimonies have been rejected yet</p>
                            </div>
                        `;
                        break;
                    case 'waiting':
                        emptyStateHtml = `
                            <div class="empty-state">
                                <i class="fas fa-comment-medical"></i>
                                <h3>No Waiting Testimonies</h3>
                                <p>All users have submitted their testimonies</p>
                            </div>
                        `;
                        break;
                }
                listContainer.innerHTML = emptyStateHtml;
                return;
            }
            
            let html = '';
            
            testimonies.forEach(testimony => {
                const userName = testimony.user_name || 'Unknown User';
                const initials = getUserInitials(userName);
                const createdAt = new Date(testimony.created_at).toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric',
                    hour: '2-digit',
                    minute: '2-digit'
                });
                
                // Only show action buttons for pending testimonies
                const actionButtons = status === 'pending' ? `
                    <button class="view-btn" onclick="viewTestimonyImage('${testimony.testimony_url}', '${userName}', '${createdAt}')">
                        <i class="fas fa-eye"></i> View
                    </button>
                    <button class="approve-btn" onclick="approveTestimony(${testimony.id}, '${userName}')">
                        <i class="fas fa-check"></i> Approve
                    </button>
                    <button class="reject-btn" onclick="rejectTestimony(${testimony.id}, '${userName}')">
                        <i class="fas fa-times"></i> Reject
                    </button>
                ` : `
                    <button class="view-btn" onclick="viewTestimonyImage('${testimony.testimony_url}', '${userName}', '${createdAt}')">
                        <i class="fas fa-eye"></i> View Testimony
                    </button>
                `;
                
                // Only show view button if there's a testimony URL
                const viewButton = testimony.testimony_url ? `
                    <button class="view-btn" onclick="viewTestimonyImage('${testimony.testimony_url}', '${userName}', '${createdAt}')">
                        <i class="fas fa-eye"></i> View
                    </button>
                ` : '';
                
                html += `
                    <div class="testimony-item ${testimony.status}" data-id="${testimony.id}">
                        <div class="testimony-user">
                            <div class="user-avatar">
                                ${testimony.profile_url ? 
                                    `<img src="${testimony.profile_url}" alt="${userName}" onerror="this.style.display='none'; this.parentElement.innerHTML='${initials}';">` : 
                                    initials}
                            </div>
                            <div class="user-info">
                                <h4>${userName}</h4>
                                <p>Order ID: #${testimony.order_id}</p>
                                <p>Channel: ${testimony.channel}</p>
                                <p>Submitted: ${createdAt}</p>
                            </div>
                        </div>
                        
                        <div class="testimony-details">
                            <h4>Order Details</h4>
                            <div class="order-info">
                                <div class="order-info-item">
                                    <label>Amount</label>
                                    <span>₦${parseFloat(testimony.amount).toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>User Status</label>
                                    <span>${testimony.user_status}</span>
                                </div>
                                <div class="order-info-item">
                                    <label>Status</label>
                                    <span class="status-badge status-${testimony.status}">${testimony.status}</span>
                                </div>
                            </div>
                        </div>
                        
                        <div class="testimony-actions">
                            ${status === 'pending' ? `
                                ${testimony.testimony_url ? `
                                    <button class="view-btn" onclick="viewTestimonyImage('${testimony.testimony_url}', '${userName}', '${createdAt}')">
                                        <i class="fas fa-eye"></i> View
                                    </button>
                                ` : ''}
                                <button class="approve-btn" onclick="approveTestimony(${testimony.id}, '${userName}')">
                                    <i class="fas fa-check"></i> Approve
                                </button>
                                <button class="reject-btn" onclick="rejectTestimony(${testimony.id}, '${userName}')">
                                    <i class="fas fa-times"></i> Reject
                                </button>
                            ` : `
                                ${testimony.testimony_url ? `
                                    <button class="view-btn" onclick="viewTestimonyImage('${testimony.testimony_url}', '${userName}', '${createdAt}')">
                                        <i class="fas fa-eye"></i> View Testimony
                                    </button>
                                ` : ''}
                            `}
                        </div>
                    </div>
                `;
            });
            
            listContainer.innerHTML = html;
        }
        
        // Update statistics
        function updateStats(stats) {
            document.getElementById('pendingCount').textContent = stats.pending || 0;
            document.getElementById('approvedCount').textContent = stats.approved || 0;
            document.getElementById('rejectedCount').textContent = stats.rejected || 0;
            document.getElementById('waitingCount').textContent = stats.waiting || 0;
        }
        
        // Get user initials
        function getUserInitials(name) {
            const names = name.split(' ');
            let initials = '';
            names.forEach(name => {
                if (name.trim()) {
                    initials += name.trim().charAt(0).toUpperCase();
                }
            });
            return initials.substring(0, 2);
        }
        
        // View testimony image
        function viewTestimonyImage(imageUrl, userName, date) {
            const modal = document.getElementById('imageModal');
            const modalImage = document.getElementById('modalImage');
            const imageInfo = document.getElementById('imageInfo');
            
            modalImage.src = imageUrl;
            imageInfo.textContent = `Testimony from ${userName} - Submitted on ${date}`;
            modal.classList.add('active');
        }
        
        // Approve testimony
        async function approveTestimony(testimonyId, userName) {
            showConfirm(
                'Approve Testimony',
                `Are you sure you want to approve testimony from ${userName}?`,
                async () => {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'approve');
                        formData.append('testimony_id', testimonyId);
                        
                        const response = await fetch('b_testimony.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'Testimony approved successfully!', 'success');
                            // Reload current tab
                            loadTestimonies(currentTab);
                        } else {
                            showMessage('Error', data.message || 'Failed to approve testimony', 'error');
                        }
                    } catch (error) {
                        console.error('Error approving testimony:', error);
                        showMessage('Error', 'Error approving testimony', 'error');
                    }
                }
            );
        }
        
        // Reject testimony
        async function rejectTestimony(testimonyId, userName) {
            showConfirm(
                'Reject Testimony',
                `Are you sure you want to reject testimony from ${userName}?`,
                async () => {
                    try {
                        const formData = new FormData();
                        formData.append('action', 'reject');
                        formData.append('testimony_id', testimonyId);
                        
                        const response = await fetch('b_testimony.php', {
                            method: 'POST',
                            body: formData
                        });
                        
                        const data = await response.json();
                        
                        if (data.success) {
                            showMessage('Success', 'Testimony rejected successfully!', 'success');
                            // Reload current tab
                            loadTestimonies(currentTab);
                        } else {
                            showMessage('Error', data.message || 'Failed to reject testimony', 'error');
                        }
                    } catch (error) {
                        console.error('Error rejecting testimony:', error);
                        showMessage('Error', 'Error rejecting testimony', 'error');
                    }
                }
            );
        }
        
        // Switch tab
        function switchTab(tab) {
            // Update active tab
            document.querySelectorAll('.tab').forEach(t => {
                t.classList.remove('active');
                if (t.dataset.tab === tab) {
                    t.classList.add('active');
                }
            });
            
            // Update active stat card
            document.querySelectorAll('.stat-card').forEach(card => {
                card.classList.remove('active');
                if (card.dataset.status === tab) {
                    card.classList.add('active');
                }
            });
            
            // Update active tab content
            document.querySelectorAll('.tab-content').forEach(content => {
                content.classList.remove('active');
                if (content.id === tab + 'Tab') {
                    content.classList.add('active');
                }
            });
            
            // Update current tab and load data
            currentTab = tab;
            loadTestimonies(tab);
        }
        
        // Modal event listeners
        document.getElementById('confirmCancelBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            confirmCallback = null;
        });
        
        document.getElementById('confirmActionBtn').addEventListener('click', () => {
            document.getElementById('confirmModal').classList.remove('active');
            if (confirmCallback) {
                confirmCallback();
                confirmCallback = null;
            }
        });
        
        document.getElementById('messageOkBtn').addEventListener('click', () => {
            document.getElementById('messageModal').classList.remove('active');
        });
        
        document.getElementById('closeToast').addEventListener('click', () => {
            document.getElementById('toastNotification').classList.remove('active');
        });
        
        // Close modals when clicking outside
        document.querySelectorAll('.modal-overlay').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    modal.classList.remove('active');
                    if (modal.id === 'confirmModal') {
                        confirmCallback = null;
                    }
                }
            });
        });
        
        // Close image modal
        document.getElementById('closeImageModal').addEventListener('click', () => {
            document.getElementById('imageModal').classList.remove('active');
        });
        
        // Close image modal when clicking outside
        document.getElementById('imageModal').addEventListener('click', (e) => {
            if (e.target === document.getElementById('imageModal')) {
                document.getElementById('imageModal').classList.remove('active');
            }
        });
        
        // Refresh button
        document.getElementById('refreshBtn').addEventListener('click', () => {
            const btn = document.getElementById('refreshBtn');
            btn.classList.add('loading');
            
            loadTestimonies(currentTab).finally(() => {
                setTimeout(() => {
                    btn.classList.remove('loading');
                }, 500);
            });
        });
        
        // Tab click event listeners
        document.querySelectorAll('.tab').forEach(tab => {
            tab.addEventListener('click', () => {
                switchTab(tab.dataset.tab);
            });
        });
        
        // Stat card click event listeners
        document.querySelectorAll('.stat-card').forEach(card => {
            card.addEventListener('click', () => {
                switchTab(card.dataset.status);
            });
        });
        
        // Start auto-refresh every 3 seconds
        function startAutoRefresh() {
            autoRefreshInterval = setInterval(() => {
                loadTestimonies(currentTab);
            }, 3000);
        }
        
        // Stop auto-refresh (when modal is open)
        function stopAutoRefresh() {
            clearInterval(autoRefreshInterval);
        }
        
        // Pause auto-refresh when modals are open
        document.querySelectorAll('.modal-overlay, .image-modal').forEach(modal => {
            modal.addEventListener('click', (e) => {
                if (e.target === modal && modal.classList.contains('active')) {
                    stopAutoRefresh();
                }
            });
        });
        
        // Resume auto-refresh when modals are closed
        document.querySelectorAll('.close-modal, .alert-btn, #closeImageModal').forEach(btn => {
            btn.addEventListener('click', () => {
                setTimeout(startAutoRefresh, 1000);
            });
        });
        
        // Start auto-refresh
        startAutoRefresh();
        
        // Initial load
        switchTab('pending');
    </script>
</body>
</html>